/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_ARRAY_
#define  _SO_ARRAY_

#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFShort.h>
#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/nodes/SoGroup.h>
#include <Inventor/nodes/SoTranslation.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoArray
//
//  Array group node: creates an IxJxK array. Each element in the
//  array is drawn by traversing all of the children of this node,
//  saving and restoring state before and after each element. I, J,
//  and K are given by the numElements1, numElements2, and
//  numElements3 fields. The 3D vector separating the centers of
//  adjacent elements in each of the three directions is given by
//  separation1, separation2, and separation3. The origin field
//  indicates whether the array is positioned with the current
//  object-space origin at the center of the first element, the last
//  element, or at the center of all elements.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Group node that creates a regular IxJxK array of copies of its children.
 * 
 * @ingroup GroupNodes
 * 
 * @DESCRIPTION
 *   This group node traverses its children, in order, several times, creating a
 *   regular 3D array of copies of them. The number of copies in each of the three
 *   directions is specified by fields, as are the vectors used to separate the
 *   copies in each of the three dimensions.
 *   
 *   For example, an SoArray node can be used to create a 2x3x4 array of copies of its
 *   children, where the separation vectors between adjacent copies in the three
 *   array dimensions are (1,2,3), (-4,-5,-6), and (7,8,9), respectively. The base
 *   point of the array can be set to one of several values, as described in the
 *   #origin field.
 *   
 *   Copies are traversed so that the first dimension cycles most quickly, followed by
 *   the second, and then the third. This order is important because SoArray sets the
 *   current switch value to N before traversing the children for the Nth time (for
 *   use with inherited switch values - see SoSwitch).
 * 
 * @FILE_FORMAT_DEFAULT
 *    Array {
 *    @TABLE_FILE_FORMAT
 *       @TR numElements1   @TD 1
 *       @TR numElements2   @TD 1
 *       @TR numElements3   @TD 1
 *       @TR separation1    @TD 1 0 0
 *       @TR separation2    @TD 0 1 0
 *       @TR separation3    @TD 0 0 1
 *       @TR origin         @TD FIRST
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction,  SoGetBoundingBoxAction,  SoRayPickAction @BR
 *        Traverses all children for each array element, saving and restoring state
 *        before and after each traversal. Sets: SoSwitchElement
 * 
 *    SoSearchAction @BR
 *        Traverses all children once, setting the inherited switch value to
 *        @B SO_SWITCH_ALL @b first.
 * 
 * 
 * @SEE_ALSO
 *    SoMultipleCopy,
 *    SoSwitch
 * 
 * 
 */
class INVENTOR_API SoArray : public SoGroup {

  SO_NODE_HEADER(SoArray);

 public:

  // Fields
  /**
   * Number of elements in the 1st dimension
   */
  SoSFShort           numElements1;

  /**
   * Number of elements in the 2nd dimension
   */
  SoSFShort           numElements2;

  /**
   * Number of elements in the 3rd dimension
   */
  SoSFShort           numElements3;

  /**
   * Separation vector in the 1st dimension
   */
  SoSFVec3f           separation1;

  /**
   * Separation vector in the 2nd dimension
   */
  SoSFVec3f           separation2;    

  /**
   * Separation vector in the 3rd dimension
   */
  SoSFVec3f           separation3;

  /** Array origin */
  enum Origin {
    /**
     *  First copy is rendered at the current local origin; all other copies are
     * distributed relative to it 
     */
    FIRST,
    /**
     *  Copies are distributed relative to the center of the array 
     */
    CENTER,
    /**
     *  Last copy is rendered at the current local origin; all other copies are
     * distributed relative to it 
     */
    LAST
  };

  /**
   * Defines the base point from which copies are distributed.
   * @useenum{Origin}.  Default is FIRST.
   */
  SoSFEnum            origin;

  /**
   * Creates an array node with default settings.
   */
  SoArray();

  /**
   * Overrides default method on SoNode to return FALSE since arrays
   * are effectively separators
   */
  virtual SbBool      affectsState() const;

 SoEXTENDER public:
  // Implement actions
  virtual void        doAction(SoAction *action);
  virtual void        callback(SoCallbackAction *action);
  virtual void        GLRender(SoGLRenderAction *action);
  virtual void        pick(SoPickAction *action);
  virtual void        getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void        handleEvent(SoHandleEventAction *action);
  virtual void        getMatrix(SoGetMatrixAction *action);
  virtual void        search(SoSearchAction *action);
  virtual void        getPrimitiveCount(SoGetPrimitiveCountAction *action);

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();
  virtual void notify( SoNotList* list );

 protected:
  virtual ~SoArray();

private:
  SoRef<SoTranslation> m_globalTranslation;
  std::vector<SoRef<SoTranslation> > m_translations;
};

#endif /* _SO_ARRAY_ */

