/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/


#ifndef  _SO_BILLBOARD_H_
#define  _SO_BILLBOARD_H_

#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFBitMask.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoMatrixTransform.h>

/*! \cond PRIVATE */
namespace inventor
{
namespace node
{
  class SoBillboardTransform;
}
}
/*! \endcond */

/**
 * @VSGEXT Grouping node that modifies its coordinate system so that its local Z-axis
 * always points at the viewer.
 * 
 * @ingroup GroupNodes
 * 
 * @DESCRIPTION
 *   The SoBillboard node is a grouping node which modifies its coordinate system
 *   so that the billboard node's local Z-axis turns to point at the viewer. The
 *   children of an SoBillboard node may include other grouping nodes as well as
 *   shape nodes. 
 *
 *   @B Performance:@b
 *   - SoBillboard nodes recompute the screen alignment transformation on every
 *     render traversal. A large number of billboard nodes will reduce performance.
 *
 *   - The orientation of an SoBillboard depends on the current
 *     camera settings, so this node should not be render cached.
 *
 *   - SoBillboard nodes prohibit auto-caching, so no SoSeparator (or other render
 *     caching node) above an SoBillboard in the scene graph will automatically create
 *     a render cache.
 *     This can significantly reduce performance and applications should
 *     put nodes that @I can @i be render cached (especially geometry nodes)
 *     under their own SoSeparator below the SoBillboard.
 *
 *   - We recommend leaving the #useSphereBBox field set to true. @BR
 *     This will significantly improve performance when using billboard nodes.
 *
 *   @B Export limitations:@b
 *   - SoBillboard can be exported to PDF3D/U3D (SoToPDFAction) as a group
 *     node, but due to limitations of that file format, it will @I not@i act as a
 *     billboard. In other words, the geometry will not be "screen locked".
 * 
 *   The #axisOfRotation field specifies which axis to use to perform the
 *   rotation. This axis is defined in the local coordinates of the SoBillboard
 *   node. The default (0,1,0) is useful for objects such as images of trees and
 *   lamps positioned on a ground plane. But when an object is oriented at an angle,
 *   for example, on the incline of a mountain, then the #axisOfRotation may
 *   also need to be oriented at a similar angle. 
 *   
 *   A special case of billboarding is @I screen-alignment @i -- the object rotates
 *   to always stay aligned with the viewer even when the viewer elevates, pitches
 *   and rolls. This special case is distinguished by setting the
 *   #axisOfRotation to (0, 0, 0). 
 *   
 *   To rotate the SoBillboard to face the viewer, determine the line between the
 *   SoBillboard's origin and the viewer's position; call this the
 *   billboard-to-viewer line. The #axisOfRotation and the billboard-to-viewer
 *   line define a plane. The local z-axis of the SoBillboard is then rotated
 *   into that plane, pivoting around the #axisOfRotation. 
 *   
 *   If the #axisOfRotation and the billboard-to-viewer line are coincident (the
 *   same line), then the plane cannot be established, and the rotation results of
 *   the SoBillboard are undefined. For example, if the #axisOfRotation is
 *   set to (0,1,0) (Y-axis) and the viewer flies over the SoBillboard and peers
 *   directly down the Y-axis the results are undefined@B . @b
 *   
 *   Multiple instances of SoBillboards (DEF/USE) operate as expected - each
 *   instance rotates in its unique coordinate system to face the viewer.
 *   
 * @FILE_FORMAT_DEFAULT
 *    Billboard {
 *    @TABLE_FILE_FORMAT
 *       @TR axisOfRotation   @TD 0 1 0
 *       @TR useSphereBBox    @TD TRUE
 *    @TABLE_END
 *    }
 * 
 * @NODE_SINCE_OIV 10.0
 *
 */
class INVENTOR_API SoBillboard : public SoSeparator
{
  SO_NODE_HEADER(SoBillboard);

 public:
  /**
   * Specifies which axis to use to perform the rotation.
   */
  SoSFVec3f axisOfRotation;

  /**
   * Specifies if the bounding box returned by SoBillboard::getBoundingBox() method
   * should be extended to completely contain the bounding sphere of the children.
   *
   * This hint avoids bounding box cache breaks due to rotation of the billboard
   * geometry (which changes the default axis aligned bounding box) and provides
   * a significant performance increase when many SoBillboard nodes are used.
   * 
   * Default is TRUE
   */
  SoSFBool useSphereBBox;

  /**
   * Constructor.
   */
  SoBillboard();

  /** Constructor that takes approximate number of children. */
  SoBillboard(int nChildren);

 SoEXTENDER public:
  // Implement actions
  virtual void        doAction(SoAction *action);
  virtual void        callback(SoCallbackAction *action);
  virtual void        GLRender(SoGLRenderAction *action);
  virtual void        getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void        rayPick(SoRayPickAction *action);

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

 protected:

  // Destructor.
  virtual ~SoBillboard();

 private:
   // call by constructor
   void commonConstructor(const int nChildren);

   inventor::node::SoBillboardTransform* m_billboardTransform;

   // OIV_BILLBOARD_EXTRA_BBOX environment variable status
   static bool s_extraBboxActivated;
};

#endif /* _SO_BILLBOARD_H_ */
