/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#if !defined SOBUFFEREDSHAPE_H
#define SOBUFFEREDSHAPE_H

#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoMFEnum.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoMFInt32.h>
#include <Inventor/fields/SoSFShort.h>
#include <Inventor/fields/SoSFBox3f.h>
#include <Inventor/fields/SoSFBufferObject.h>
#include <Inventor/fields/SoMFBufferObject.h>

#include <Inventor/devices/SoCpuBufferObject.h>
#include <Inventor/devices/SoGLBufferObject.h>

#include <Inventor/nodes/SoShape.h>

#include <Inventor/elements/SoDrawStyleElement.h>

#include <Inventor/SbPImpl.h>


class SoGLBufferObject;
class SoAction;

SO_PIMPL_PUBLIC_DECLARATION(SoBufferedShape)

/**
@VSGEXT Node to render geometry stored in SoBufferObject objects.

@ingroup ShapeNodes

@DESCRIPTION
SoBufferedShape is useful to manage the rendering of large geometry, provide 
application control over where the data is stored (CPU or GPU) and to 
integrate rendering with the Open Inventor computing framework (through the 
SoBufferObject classes).

SoBufferedShape provides fields for:

- Vertices
- Indices (optional)
- Colors (optional)
- Normals (optional)
- Texture coordinates (optional)

In this sense it is similar to the SoVertexProperty node, but SoVertexProperty 
is just a property node. SoBufferedShape also does the rendering of the shape. 
Properties that are not specified are taken from the traversal state (e.g. colors) 
or computed (e.g. normals).

SoBufferedShape can render many types of geometric primitives including points, lines,
quads and triangles. (A single type must be specified per instance of SoBufferedShape.)
You specify the type of primitive in the SoSFEnum field #shapeType.

SoBufferedShape can render multiple primitives of the same type.  You can
specify the number of vertices (or indices if provided) to use for each primitive 
in the SoMFInt32 field #numVertices (similar to SoFaceSet).

You can also use the primitive restart feature to define multiple indexed strip shapes,
for example TRIANGLE_STRIP or LINE_STRIP. The end of each primitive is marked by a 
special index value in the index buffer and this value can be specified
in the #primitiveRestartValue field.  The behavior is similar to the "-1" value that can
be used in Open Inventor indexed shape nodes like SoIndexedFaceSet, but is implemented 
on the GPU.  @BR
NOTE:
  - Primitive restart must be explicitly enabled using the #primitiveRestartEnabled field.
  - Check availability of primitive restart using SoShape::isPrimitiveRestartAvailable().

The geometry and its attributes must be stored in buffer objects (see SoBufferObject).
The buffer objects can be SoGpuBufferObjects stored directly on the graphics board or
SoCpuBufferObjects stored in system memory.  This allows the application to control 
what data is stored where.

In the general case, when the underlying buffer object is modified using the SoBufferObject
API (setSize, map, memcpy, memset), the containing SoSFBufferObject field needs to be
notified using touch() in order for Open Inventor to take this modification into account.
If touch() is not called, there is no guarantee that the modifications will be visible
on the next rendering.

If the application needs to frequently modify the values of an SoBufferObject, calling touch()
on the field can result in an undesirable decrease in performance. Instead, it is recommended
to use an instance of SoGpuBufferObject and set the #shapeUsage field to DYNAMIC. These settings
guarantee that this buffer data will be used directly for rendering, and eliminate the need to
call touch() on the field. It also guarantees that modifications to the buffer done by shaders
will be properly taken into account.

If lighting is enabled (there is no SoLightModel node or the model field of the SoLightModel
is set to PHONG) and the #normalBuffer field is not set, then Open Inventor will
automatically compute normal vectors, but only in some cases (see Limitations section).
Normal generation is affected by the creaseAngle field of the SoShapeHints node, but only
if the vertices are NOT indexed (indexBuffer field is not set).  If the vertices are indexed
the creaseAngle is forced to PI, creating a smooth surface rendering. If the application 
needs to render sharp edges on a shape, either compute normal vectors and set the 
normalBuffer field or do not use the indexBuffer field.
It is possible to disable normal generation (if for example the normals are generated by a
geometry shader) by setting the #useNormalsGenerator field to FALSE. If no normal vectors 
are specified or generated, and lighting is enabled, the primitive may not be
rendered correctly.

SoBufferedShape provides fields to describe the content of each buffer, e.g. the data 
type and number of components in each buffer, as well as how to access the buffers, 
e.g. the offset into the buffer and "stride" separating data values in the buffer. 
The default values for offset and stride assume that the vertices, normals, etc are
each in a separate buffer.  However setting appropriate offset and stride allows, for
example, vertices and normals to be interleaved in a single buffer.  In this case the 
same buffer would be set into both the vertexBuffer and normalBuffer fields.

To disable computing the bounding box, which can take a long time with very large geometry,
use the SoBBox node to specify a pre-computed bounding box.

# Limitations

- Transparency:
  If there is no color buffer, making the entire shape transparent using an SoMaterial or an
  SoPhysicalMaterial node works as usual.  However if there is a color buffer with RGBA values, note that
  Open Inventor does not currently check the color buffer for transparency (alpha values
  < 1). So in this case the SoBufferedShape will not be considered transparent geometry
  (even if there are alpha values < 1) and may not be rendered correctly. You can force Open 
  Inventor to handle the shape as transparent geometry by putting a transparent
  @ref SoLightModel-base-color-def "base color" before this shape in the scene graph.

- Normal generation:
  If lighting is enabled and the normalBuffer field is not set, then Open Inventor will
  automatically compute normal vectors, but only in some cases. 
  Automatic generation of normal vectors is ONLY enabled when:
  - The #vertexComponentsCount field is set to 2 or 3,
  - The #primitiveRestartEnabled field is set to FALSE (default), and
  - The primitives are TRIANGLES, TRIANGLE_STRIP or QUADS. @BR
  Note: The crease angle (see SoShapeHints) is not used by the normal
  generator if the vertices are indexed. If the application needs to render sharp edges
  on a shape, either compute normal vectors and set the normalBuffer field or do not use 
  the indexBuffer field.

- SoGetPrimitiveCountAction:
  When using the primitive restart feature, the triangle/line count returned by the
  SoGetPrimitiveCountAction will not be accurate.

- Concave polygons:
  Unlike (for example) SoFaceSet, SoBufferedShape does NOT automatically tesselate
  concave or complex polygons.  Such primitives may not be rendered correctly.

- SoWriteAction:
  SoBufferedShape can be saved to and restored from a .iv file just like any other
  Open Inventor node. However, during the read operation any GPU buffer objects
  (SoGpuBufferObject) in the file will be created as CPU buffers (SoCpuBufferObject).

- Material binding (etc):
  SoBufferedShape effectively only supports per-vertex and per-vertex-indexed
  binding of materials, normals and texture coordinates using the
  values found in its own buffers.

- Double-sided coloring:
  When multiple static buffered shapes (see #shapeUsage) share the same buffers, you cannot use a single color
  buffer for both the front and back faces because SoBufferedShape does not support interlacing colors.
  Therefore, you must use two separate color buffers.

Example using CPU buffer:
\if_cpp
\code
  // Result should be similar to SoLineSet example in PG-GettingStarted.pdf.
  // This example does not show any of the advantages of using SoBufferedShape,
  // just the simplest possible setup and usage.
  // Coordinate data
  const float vertices[][3] = { 1,0.5,0, 0,1,0, -1,0.5,0,
                             -1,-1,0, 1,-1,0, 1,0,0, -1,0,0,
                             -1,-1.5,0, 1,-1.5,0 };
  const int numVerts[] = {3, 4, 2};
  const int NUM_COORDS = sizeof(vertices) / sizeof(SbVec3f);
  const int NUM_COUNTS = sizeof(numVerts) / sizeof(int);

  // Wrap coordinate array in a CPU buffer object
  SoRef<SoCpuBufferObject> vertBuf = 
      new SoCpuBufferObject( (void*)vertices, NUM_COORDS sizeof(SbVec3f) );

  // Create a buffered shape to render the geometry
  SoBufferedShape *pShape = new SoBufferedShape;
  pShape->shapeType = SoBufferedShape::LINE_STRIP;
  pShape->numVertices.setValues( 0, NUM_COUNTS, numVerts );
  pShape->vertexBuffer.setValue( vertBuf.ptr() );
\endcode
\endif
\if_dotnet
\code
  // Result should be similar to SoLineSet example in PG-GettingStarted.pdf.
  // This example does not show any of the advantages of using SoBufferedShape,
  // just the simplest possible setup and usage.
  // Coordinate data
  float[] vertices = new float[9 3] {
          1.0f, 0.5f,0.0f, 0.0f, 1.0f,0.0f, -1.0f,0.5f,0.0f,
         -1.0f,-1.0f,0.0f, 1.0f,-1.0f,0.0f,  1.0f,0.0f,0.0f, -1.0f,0.0f,0.0f,
         -1.0f,-1.5f,0.0f, 1.0f,-1.5f,0.0f
  };
  Int32[] numVerts = new Int32[3] { 3, 4, 2 };

  // Wrap coordinate array in a CPU buffer object
  SbNativeArray<float> vertArray = new SbNativeArray<float>(vertices);
  SoCpuBufferObject   vertBuffer = new SoCpuBufferObject((SbNativeArray<byte>)vertArray);

  // Create a buffered shape to render the geometry
  SoBufferedShape shape = new SoBufferedShape();
  shape.shapeType.SetValue( "LINE_STRIP" );
  shape.numVertices.SetValues( 0, numVerts );
  shape.vertexBuffer.Value = vertBuffer;
\endcode
\endif
\if_java
\code
  // Result should be similar to SoLineSet example in PG-GettingStarted.pdf.
  // This example does not show any of the advantages of using SoBufferedShape,
  // just the simplest possible setup and usage.
  // Coordinate data
 	float[] vertices = {
 	        1.0f, 0.5f,0.0f, 0.0f, 1.0f,0.0f, -1.0f,0.5f,0.0f,
 	       -1.0f,-1.0f,0.0f, 1.0f,-1.0f,0.0f,  1.0f,0.0f,0.0f, -1.0f,0.0f,0.0f,
 	       -1.0f,-1.5f,0.0f, 1.0f,-1.5f,0.0f
 	};
 	int[] numVerts = { 3, 4, 2 };

  // Create a CPU buffer object and set its size (allocate memory)
  SoCpuBufferObject cpuBuffer = new SoCpuBufferObject();
  cpuBuffer.setSize( vertices.length Float.SIZE/8 );

  // Copy vertex data into the buffer object
  FloatBuffer vertData = cpuBuffer.map( SoBufferObject.AccessModes.SET ).asFloatBuffer();
      vertData.put(vertices);
  cpuBuffer.unmap();

  // Create a buffered shape to render the geometry
  SoBufferedShape shape = new SoBufferedShape();
  shape.shapeType.setValue( "LINE_STRIP" );
  shape.numVertices.setValues( 0, numVerts );
  shape.vertexBuffer.setValue( cpuBuffer );
\endcode
\endif

Example using GPU buffer:
\if_cpp
\code
  // Result should be similar to SoLineSet example in PG-GettingStarted.pdf.
  // This example does not show any of the advantages of using SoBufferedShape,
  // just the simplest possible setup and usage.
  // Coordinate data
  const float coords[][3] = { 1,0.5,0, 0,1,0, -1,0.5,0,
                             -1,-1,0, 1,-1,0, 1,0,0, -1,0,0,
                             -1,-1.5,0, 1,-1.5,0 };
  const int numVerts[] = {3, 4, 2};
  const int NUM_COORDS = sizeof(coords) / sizeof(SbVec3f);
  const int NUM_COUNTS = sizeof(numVerts) / sizeof(int);

  // Wrap coordinate array in a CPU buffer object
  SoRef<SoCpuBufferObject> cpuBuffer = 
      new SoCpuBufferObject( (void*)coords, NUM_COORDS sizeof(SbVec3f) );

  // Create a GPU (OpenGL) buffer and load data from CPU buffer
  SoRef<SoGLContext> glContext = new SoGLContext(true);
  glContext->bind();
    SoRef<SoGpuBufferObject> gpuBuffer = new SoGpuBufferObject( SoGpuBufferObject::STATIC, SoGpuBufferObject::SHARED );
    gpuBuffer->setSize  ( cpuBuffer->getSize() ); // Set the buffer size (allocate memory)
    gpuBuffer->memcpy   ( cpuBuffer.ptr() );      // Copy data into the buffer
  glContext->unbind();

  // Create a buffered shape node to render the geometry
  SoBufferedShape *pShape = new SoBufferedShape;
  pShape->shapeType = SoBufferedShape::LINE_STRIP;
  pShape->numVertices.setValues( 0, NUM_COUNTS, numVerts );
  pShape->vertexBuffer.setValue( gpuBuffer.ptr() );
\endcode
\endif
\if_dotnet
\code
  // Result should be similar to SoLineSet example in PG-GettingStarted.pdf.
  // This example does not show any of the advantages of using SoBufferedShape,
  // just the simplest possible setup and usage.
  // Coordinate data
  float[] vertices = new float[9 3] {
      1.0f, 0.5f,0.0f, 0.0f, 1.0f,0.0f, -1.0f,0.5f,0.0f,
      -1.0f,-1.0f,0.0f, 1.0f,-1.0f,0.0f,  1.0f,0.0f,0.0f, -1.0f,0.0f,0.0f,
      -1.0f,-1.5f,0.0f, 1.0f,-1.5f,0.0f
  };
  Int32[] numVerts = new Int32[3] { 3, 4, 2 };
  

  // Wrap coordinate array in a CPU buffer object
  SbNativeArray<float> vertArray = new SbNativeArray<float>(vertices);
  SoCpuBufferObject vertBuf = new SoCpuBufferObject( (SbNativeArray<byte>)vertArray );

  // Create a GPU (OpenGL) buffer and load data from CPU buffer
  SoGLContext glContext = new SoGLContext(true);
  glContext.Bind();
      SoGpuBufferObject gpuBuffer = new SoGpuBufferObject( SoGpuBufferObject.BufferAccessFrequencies.STATIC, SoGpuBufferObject.BufferAccessNatures.SHARED );
      gpuBuffer.SetSize( cpuBuffer.GetSize() );  // Set the buffer size (allocate memory)
      gpuBuffer.Memcpy ( cpuBuffer );            // Copy data into the buffer
  glContext.Unbind();

  // Create a buffered shape to render the geometry
  SoBufferedShape shape = new SoBufferedShape();
  shape.shapeType.SetValue("LINE_STRIP");
  shape.numVertices.SetValues(0, numVerts);
  shape.vertexBuffer.Value = gpuBuffer;
  root.AddChild(shape);
\endcode
\endif
\if_java
\code
  // Result should be similar to SoLineSet example in PG-GettingStarted.pdf.
  // This example does not show any of the advantages of using SoBufferedShape,
  // just the simplest possible setup and usage.
  // Coordinate data
 	float[] vertices = {
 	        1.0f, 0.5f,0.0f, 0.0f, 1.0f,0.0f, -1.0f,0.5f,0.0f,
 	       -1.0f,-1.0f,0.0f, 1.0f,-1.0f,0.0f,  1.0f,0.0f,0.0f, -1.0f,0.0f,0.0f,
 	       -1.0f,-1.5f,0.0f, 1.0f,-1.5f,0.0f
 	};
 	int[] numVerts = { 3, 4, 2 };

  // Create a GPU (OpenGL) buffer and set its size (allocate memory)
  SoGLContext glContext = new SoGLContext( true );
  glContext.bind();
      SoGpuBufferObject gpuBuffer = new SoGpuBufferObject( SoGpuBufferObject.BufferAccessFrequencies.STATIC, SoGpuBufferObject.BufferAccessNatures.SHARED );
      gpuBuffer.setSize( vertices.length Float.SIZE/8 );

  // Copy vertex data into the GPU buffer object
  FloatBuffer vertData = gpuBuffer.map( SoBufferObject.AccessModes.SET ).asFloatBuffer();
      vertData.put(vertices);
  gpuBuffer.unmap();

  glContext.unbind();

  // Create a buffered shape to render the geometry
  SoBufferedShape shape = new SoBufferedShape();
  shape.shapeType.setValue( "LINE_STRIP" );
  shape.numVertices.setValues( 0, numVerts );
  shape.vertexBuffer.setValue( gpuBuffer );
\endcode
\endif

 @B LIMITATIONS: @b 
 SoBufferedShape needs a graphic card supporting vertex buffer objects,
 if not available shape won't be rendered.

 @FILE_FORMAT_DEFAULT
   BufferedShape {
   @TABLE_FILE_FORMAT
      @TR useNormalsGenerator      @TD TRUE
      @TR shapeType                @TD TRIANGLES
      @TR numVertices              @TD 0
      @TR vertexBuffer             @TD NULL
      @TR vertexComponentsCount    @TD 3
      @TR vertexComponentsType     @TD SbDataType::FLOAT
      @TR vertexStride             @TD 0
      @TR vertexOffset             @TD 0
      @TR normalBuffer             @TD NULL
      @TR normalComponentsType     @TD SbDataType::FLOAT
      @TR normalStride             @TD 3 sizeof(float)
      @TR normalOffset             @TD 0
      @TR indexBuffer              @TD NULL
      @TR indexType                @TD SbDataType::UNSIGNED_INT32
      @TR indexOffset              @TD 0
      @TR colorBuffer              @TD NULL
      @TR colorComponentsType      @TD SbDataType::FLOAT
      @TR colorStride              @TD 0
      @TR colorOffset              @TD 0
      @TR colorComponentsCount     @TD 3
      @TR texCoordsBuffer          @TD NULL
      @TR texCoordsComponentsType  @TD SbDataType::FLOAT
      @TR texCoordsStride          @TD 0
      @TR texCoordsOffset          @TD 0
      @TR texCoordsComponentsCount @TD 2
      @TR primitiveRestartEnabled  @TD FALSE
      @TR primitiveRestartValue    @TD -1
   @TABLE_END
   }

@ACTION_BEHAVIOR
   SoGLRenderAction,  SoCallbackAction, SoBoundingBoxAction  
       Do the actual rendering / bounding box computation.

@SEE_ALSO
   SoCpuBufferObject, SoGpuBufferObject, SoBBox


*/
class INVENTOR_API SoBufferedShape : public SoShape 
{
  SO_NODE_HEADER( SoBufferedShape );
  SO_PIMPL_PUBLIC_HEADER(SoBufferedShape)

  // ----------------------- Public usage --------------------------------------

public:

  /** 
  * Type of shape that will be rendered.
  */
  enum Type
  {
    /** Draws each vertex as a single point. */
    POINTS,

    /** Connects all the vertices to form a polyline.
     *  Given vertices A B C D, it draws the line segments AB, BC and CD. */
    LINE_STRIP,

    /** Like LINE_STRIP, but an extra line segment is drawn connecting
     *  the last vertex to the first vertex. */
    LINE_LOOP,

    /**
     * Connects each pair of vertices with a line.
     *  Given vertices A B C D, it draws the line segments AB and CD. */
    LINES,

    /** Draws a strip of connected triangles.
     *  Given vertices A B C D E F, it draws the triangles ABC, CBD, CDE and EDF. */
    TRIANGLE_STRIP,

    /** Draws a fan of triangles.
     *  Given vertices A B C D E F, it draws the triangles ABC, ACD, ADE and AEF. */
    TRIANGLE_FAN,

    /** Draws unconnected triangles.
     *  Given vertices A B C D E F, it draws the triangles ABC and DEF. */
    TRIANGLES,

    /** Draws a strip of connected quadrilaterals.
     *  Given vertices A B C D E F, it draws the quadrilaterals ABDC and DCEF. */
    QUAD_STRIP,

    /** Draws unconnected quadrilaterals.
     *  Given vertices A B C D E F G H, it draws the quadrilaterals ABCD and EFGH. */
    QUADS,

    /** Draws a single polygon using all the vertices (in each primitive). */
    POLYGON
  };

  /** 
   * This enum is used to set the #shapeUsage field.
   */
  enum Usage
  {
    /** 
     * The shape is static, its buffers are never modified or modified infrequently
     */
    STATIC,
    /** 
     * The shape is dynamic, this will prevent some optimizations but it is recommended
     * when the buffers are modified on a regular basis.
     */
    DYNAMIC
  };

  /**
  * Default constructor.
  */
  SoBufferedShape();

  /** 
   * Defines the usage of the shape.
   * Some optimizations can be performed when we know the usage of the shape.
   * Most of the time if the buffer objects attached to the shape are updated on a regular basis
   * prefer using DYNAMIC, otherwise prefer STATIC.
   *
   * STATIC provides the best performance when the content of the buffer is never modified or
   * modified infrequently.
   *
   * @useenum{Usage}. The default value is STATIC.
   *
   * @FIELD_SINCE_OIV 9.2
   */
  SoSFEnum shapeUsage;

  /**
   * Enable/disable the primitive restart feature.
   * Default is FALSE.
   * Primitive restart allows you to define multiple indexed strip shapes using
   * only one index buffer. Each time the primitive restart index is reached
   * a new strip or loop of primitives is emitted.
   * This feature is similar to the "-1" that can be used in the OIV indexed shapes
   * This also means that the availability must be checked before being used
   *
   * @B Limitations: @b 
   *   Enabling primitive restart disables the normal generator.
   *
   * @FIELD_SINCE_OIV 8.5
   */
  SoSFBool primitiveRestartEnabled;

  /**
   * Index value for the primitive restart feature.
   * Default is -1.  
   *
   * @FIELD_SINCE_OIV 8.5
   */
  SoSFInt32 primitiveRestartValue;

  /**
   * Indicates if the node should use the internal normal vector generator
   * if no normals are defined. Default is TRUE.
   *
   * This mode is only supported for shapes with float coordinates 
   * and 3 components per vertex. It is not supported for the points and the lines.
   *
   * Disabling the normal generator can be useful if the normals are computed in 
   * a shader or if the shaders don't need any normal at all.
   *
   * Normal generation is affected by the creaseAngle field of SoShapeHints.
   */
  SoSFBool useNormalsGenerator;

  /**
  * Shape type to render.
  * @useenum{Type}. Default is TRIANGLES.
  */
  SoSFEnum shapeType;

  /** 
   * Total number of vertices/indices or number of vertices/indices to be used per primitive.
   *
   * Specifically:
   * - For the shape types POINTS, LINES, TRIANGLES and QUADS \n
   *   Only the first value is meaningful and it specifies the number of
   *   vertices to be used for rendering. Specifically:
   *   - For a list of points: numVertices should be the number of points to be drawn.
   *   - For a list of lines: numVertices should be num_lines * 2 where num_lines is the number of lines to be drawn.
   *   - For a list of quadrangles: numVertices should be num_quads * 4 where num_quads is the number of quadrangles to be drawn.
   *   - For a list of triangles: numVertices should be num_tri * 3 where num_tri is the number of triangles to be drawn.
   *
   * - For all other types: \n
   *   The number of values in this field specifies the number of primitives that will be drawn. \n
   *   Each value in the field specifies the number of vertices (or indices if given)
   *   to be used for each primitive.
   */
  SoMFInt32 numVertices;

  /**
   * Buffer object that contains the vertex data. @BR
   * Default is no buffer.
   * Note: This buffer must be an SoCpuBufferObject or SoGpuBufferObject.
   *       (SoGLBufferObject with target = ARRAY_BUFFER is also allowed but not recommended.)
   */
  SoSFBufferObject vertexBuffer;

  /** 
   * Number of components in each vertex. 
   * Default is 3 (i.e. X, Y and Z).
   */
  SoSFShort vertexComponentsCount;

  /** 
   * SbDataType::DataType for vertices. 
   * @useenum{SbDataType::DataType}. Default is SbDataType::FLOAT.
   */ 
  SoSFEnum vertexComponentsType;

  /**
   * Stride in bytes between the first component of two consecutive vertices. @BR
   * Default is 0.
   * e.g: If the vertices are composed of 3 float components the stride should be
   * 3 * sizeof(float). If RGB colors are packed in the same buffer the stride should be
   * 3 * sizeof(float) + 3 * sizeof(float), the second part stands for the extra data padding.
   *
   * Note: When the values are packed (only vertices in the buffer) the value
   * 0 can be used and OpenGL will compute the stride value.
   */
  SoSFShort vertexStride;

  /** 
   * Offset in bytes to the first vertex within the buffer. 
   * Default is 0.
   */
  SoSFInt32 vertexOffset;

  /** 
   * Buffer object that contains the (optional) normal vectors. @BR
   * Default is no buffer.
   * Note: This buffer must be an SoCpuBufferObject or SoGpuBufferObject.
   *       (SoGLBufferObject with target = ARRAY_BUFFER is also allowed but not recommended.)
   */
  SoSFBufferObject normalBuffer;

  /** 
   * SbDataType::DataType type for the normal vectors. 
   * @useenum{SbDataType::DataType}. Default is SbDataType::FLOAT.
   */
  SoSFEnum normalComponentsType;

  /** 
   * Stride in bytes between the first component of two consecutive normals.
   * Default is 0.
   * e.g: If the normals are composed of 3 float components the stride should be
   * 3 * sizeof(float). If RGB colors are packed in the same buffer the stride should be
   * 3 * sizeof(float) + 3 * sizeof(float), the second part stands for the extra data padding.
   *
   * Note: When the values are packed (only normals in the buffer) the value
   * 0 can be used and OpenGL will compute the stride value.
   */
  SoSFShort normalStride;

  /** 
   * Offset in bytes to the first normal vector in the buffer.
   * Default is 0.
   */
  SoSFInt32 normalOffset;

  /** 
   * Buffer object that contains the (optional) indices.
   * Default is no buffer.
   * Note: This buffer must be an SoCpuBufferObject or SoGpuBufferObject.
   *       (SoGLBufferObject with target = ARRAY_BUFFER is also allowed but not recommended.)
   */
  SoSFBufferObject indexBuffer;

  /** 
   * SbDataType::DataType type for the indices. 
   * @useenum{SbDataType::DataType}. Default is SbDataType::UNSIGNED_INT32.
   * Note: Due to OpenGL restrictions, only 3 values are accepted here:
   * SbDataType::UNSIGNED_INT32, SbDataType::UNSIGNED_BYTE or SbDataType::UNSIGNED_SHORT.
   */
  SoSFEnum indexType;

  /** 
   * Offset in bytes to the first index in the buffer. 
   * Default is 0.
   */
  SoSFInt32 indexOffset;

  /** 
   * Buffer object that contains the (optional) color values.
   * Default is no buffer. Colors are always per-vertex or per-vertex-indexed.
   * Note: This buffer must be an SoCpuBufferObject or SoGpuBufferObject.
   *       (SoGLBufferObject with target = ARRAY_BUFFER is also allowed but not recommended.)
   */
  SoSFBufferObject colorBuffer;

  /** 
   * SbDataType::DataType type for the color values. 
   * @useenum{SbDataType::DataType}. Default is SbDataType::FLOAT.
   */
  SoSFEnum colorComponentsType;

  /** 
   * Stride in bytes between the first component of two consecutive colors. @BR
   * Default is 0.
   * e.g: If the colors are composed of 3 float components the stride should be
   * 3 * sizeof(float). If vertices are packed in the same buffer the stride should be
   * 3 * sizeof(float) + 3 * sizeof(float), the second part stands for the extra data padding.
   *
   * Note: When the values are packed (only color values in the buffer) the value
   * 0 can be used and OpenGL will compute the stride value.
   */
  SoSFInt32 colorStride;

  /** 
   * Offset in bytes to the first color value in the buffer. 
   * Default is 0.
   */
  SoSFInt32 colorOffset;

  /** 
   * Number of components in each color value. 
   * Default is 3 (i.e. red, green and blue)
   */
  SoSFInt32 colorComponentsCount;

  /** 
   * Buffer objects that contains the (optional) texture coordinates. @BR
   * Default is no buffer.
   * Note: This buffer must be an SoCpuBufferObject or SoGpuBufferObject.
   *       (SoGLBufferObject with target = ARRAY_BUFFER is also allowed but not recommended.)
   */
  SoMFBufferObject texCoordsBuffer;

  /** 
   * SbDataType::DataType type for the texture coordinates. 
   * @useenum{SbDataType::DataType}. Default is SbDataType::FLOAT.
   */
  SoMFEnum texCoordsComponentsType;

  /** 
   * Stride in bytes between the first component of two consecutive texture coordinates. @BR
   * Default is 0.
   * e.g: If each element is composed of 2 float components the stride should be
   * 2 * sizeof(float). If vertices are packed in the same buffer the stride should be
   * 2 * sizeof(float) + 3 * sizeof(float), the second part stands for the extra data padding.
   *
   * Note: When the values are packed (only texture coordinates in the buffer) the value
   * 0 can be used and OpenGL will compute the stride value.
   */
  SoMFInt32 texCoordsStride;

  /** 
   * Offset in bytes to the first texture coordinate in the buffer. 
   * Default is 0.
   */
  SoMFInt32 texCoordsOffset;

  /** 
   * Number of components in each texture coordinate. 
   * Up to 3 components are allowed (i.e. S, T and R).
   * Default is 2 (i.e. S and T)
   */
  SoMFInt32 texCoordsComponentsCount;
 
SoEXTENDER public:
  /** 
  * Manages GLRender action. 
  *
  * @param action stores the information about the action to apply.
  */
  virtual void GLRender(SoGLRenderAction *action );

  /** 
  * Manages getBoundingBoxAction action. 
  *
  * @param action stores the information about the action to apply.
  */  
  virtual void getBoundingBox( SoGetBoundingBoxAction *action );

  /**
   * Computes the AABB bounding box if it must be updated.
   */
  virtual void computeBBox( SoAction *action, SbBox3f &box, SbVec3f &center );

  /**
   * Computes the OOB bounding box if it must be updated.
   */
  virtual void computeBBox(SoAction *action, SbXfBox3d &box, SbVec3d &center);

  /**
   * Manages ray pick action
   *
   * @param action The ray pick action.
   */
  virtual void rayPick(SoRayPickAction *action );

  /**
   * Manages getPrimitiveCount action
   *
   * @param action The SoGetPrimitiveCount action.
   */
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction *action );

// ----------------------- Private usage -------------------------------------
// INTERNAL 
SoINTERNAL public:

  /** Static init function called during the OpenInventor init. */
  static void initClass();

  /** Static init function called by OpenInventor during the finish function. */
  static void exitClass();

  void notify( SoNotList* list );

  int getNumPrim( SoState* state = NULL );

  // ----------------------- Protected usage -------------------------------------

protected:

  /** Protected desctructor */
  virtual ~SoBufferedShape();

  virtual SbBool shouldCheckShapeHints();

  virtual SbBool shouldGLRender( SoGLRenderAction *action, SbBool isPointsOrLines = FALSE );

  /**
  * Parameters:
  * - normalBufferPtr: Pointer on the normal buffer in order to know if normals 
  * are defined or not.
  *
  * Returns:
  * - TRUE if the lighting is usable.
  * - FALSE otherwise
  */
  virtual SbBool isLightingUsable( const void* normalBufferPtr );

  /**
   * Returns TRUE if the shapes colorBuffer has transparent values.
   */
  virtual SbBool isTransparent(SoState* state) const;

  virtual void generatePrimitives( SoAction *action );


  /** 
   * The GLRenderBoundingBox function. 
   *
   * @param action stores the information about the action to apply.
   */  
  virtual void GLRenderBoundingBox( SoGLRenderAction *action );

  // Overrides standard method to create an SoFaceDetail instance
  virtual SoDetail* createTriangleDetail( SoRayPickAction *action,
                                            const SoPrimitiveVertex *v1,
                                            const SoPrimitiveVertex *v2,
                                            const SoPrimitiveVertex *v3,
                                            SoPickedPoint *pp );

  virtual SoDetail* createLineSegmentDetail( SoRayPickAction *action,
                                             const SoPrimitiveVertex *v1,
                                             const SoPrimitiveVertex *v2,
                                             SoPickedPoint *pp );

  virtual SoDetail* createPointDetail( SoRayPickAction *action,
                                       const SoPrimitiveVertex *v,
                                       SoPickedPoint *pp );

  //Transforms the polygons into lines before rendering them.
  //It can be used if :
  //- shapeStyle = TRIANGLE, TRIANGLE_STRIP, TRIANGLE_FAN, TRIANGLES, QUAD_STRIP, QUADS or POLYGON
  //- SoDrawStyle::style = LINES
  // ----------------------- Private usage -------------------------------------
private:
  /** gather constructor initialization */
  void commonConstructor();

  /** 
   * Compute the smallest batch size according to the vertex attrib list and its divisors
   */
  unsigned int computeSmallestBatchSize(const SoNodeList* vertexAttribsList, unsigned int numInstances);

  //In some cases primitives need to be transformed to POINTS to have a correct rendering
  //with POINTS as drawStyle.
  bool arePointsNeededAsPrimitives( SoDrawStyleElement::Style drawStyle ) const;

  template< class T > void computeBBoxTemplate_POINTS( SoAction *action, SbBox3f &box, SbVec3f &center );

  template< class T > void computeBBoxTemplate( SoAction *action, SbBox3f &box, SbVec3f &center, const SbVec2d &psize );

  template< class T > void computeBBoxTemplateSP( SoAction *action, SbBox3f &box, SbVec3f &center, const SbVec2d &psize );

  template< class T, int MODE > void computeBBoxTemplate_( SoAction *, SbBox3f &box, SbVec3f &center, const SbVec2d &pSize );
  
  //A GL context must exist to use this method.
  bool isRenderExtraDataNeededInCurrentContext() const;

  //A GL context must exist to use this method.
  bool isRenderExtraDataNeededByGraphicCardInCurrentContext() const;

  //For some shapeType values, the render extra data process must be launched ( without considering
  //OIV_SHAPES_PRIMITIVES_OPTIM ) in order to have a correct rendering with LINES as drawStyle.
  bool isRenderExtraDataNeededByShapeType() const;

  //Some graphic cards has bad performance when glPolygonMode is used to render the
  //shape with LINES as draw style.
  //
  //Returns true if the polygons must be transformed in lines before rendering them.
  static bool isRenderExtraDataNeededByGraphicCard( const SbString& graphicCardName );
};

//------------------------------------------------------------------------------

#endif // SOBUFFEREDSHAPE_H

