/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Jan 2011)
**=======================================================================*/

#if !defined _SO_CSG_SHAPE_H_
#define _SO_CSG_SHAPE_H_

#include <Inventor/nodes/SoShape.h>
#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/SoPrimitiveVertex.h>
#include <Inventor/threads/SbThreadMutex.h>

class SoFaceSet;
class SoCallbackAction;
class SoTransformation;
class SoCache;
class CachedElement;

/*! \cond PRIVATE */
namespace GetCacheDeps {
class CachedElementsVector;
}
/*! \endcond */

//@TOBEWRAPPED
/**
 * @VSGEXT Shape defined by a binary boolean operation on two sub scene graphs.
 *
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This shape represents the result of a Constructive Solid Geometry (CSG)
 *   operation applied on the two scene graphs #leftOperand and #rightOperand.
 *   Constructive solid geometry allows the application to create a (potentially)
 *   complex object by using Boolean operators to combine (relatively) simple objects.
 *
 *   The left and right operands can be any scene graph, including other SoCSGShape
 *   nodes. You can define a complex CSG geometry by creating a tree using multiple
 *   SoCSGShape nodes. The specified scene graph's shapes must be closed (no holes)
 *   and CCW-orientated (all triangles composing shapes must be counter-clockwise).
 *
 *   Each time an operand is modified, a CSG Operation is performed.
 *   Keep this in mind if you create a CSGTree (SoCSGShape which contains other
 *   SoCSGShapes), especially if you use associative and commutative operations 
 *   (only additions or only intersections) because the result of such operations 
 *   does not depend on the tree ( (A + B) + C = A + (B + C) ).
 *   In this case, try to build your tree by putting the most often modified 
 *   nodes at the top of the tree. 
 *
 *   For example, a tree optimized for modification on first node (@B N1 @b) will 
 *   look like this:
 *
 * \verbatim
       S
     __|__S
    |   __|__S
   N1  |   __|__S
      N2  |   __|__
         N3  |     |
            N4    N5
   \endverbatim
 *
 *   Where S are SoCSGShape and N1, N2, ... are ordinary nodes. In this case, 
 *   a modification on N1 will need only 1 csgOperation, whereas a modificaion 
 *   on N5 will need 4 csgOperations to regenerate the resulting shape.
 *
 *   A tree optimized for random modification will try to minimize its depth:
 *
 * \verbatim
                     S
             ________|_________
            |                  |
        ____S____          ____S____
       |         |        |         |
     __S__       N3       N4        N5
    |     |
    N1    N2
   \endverbatim
 *
 * In this case, any modification will require at most 3 CSG Operations.
 *
 * @NODE_SINCE_OIV 8.6
 * 
 * @FILE_FORMAT_DEFAULT
 *    CSGShape {
 *    @TABLE_FILE_FORMAT
 *       @TR leftOperand  @TD NULL
 *       @TR rightOperand @TD NULL
 *       @TR csgOperation @TD ADD
 *    @TABLE_END
 *    }
 *
 * @NOTES
 *   - CSG means Constructive Solid Geometry.
 *   - Some operations are not commutative (A - B) != (B - A).
 *
 * @B LIMITATIONS @b
 *   Only vertices are handled in the generated shape. None of the normal, color,
 *   or texCoord properties of the left or right operands are taken into account.
 *   Shaders are also not taken in account, even vertex shaders. But note that
 *   you can use a shader above SoCSGShape. The shader will be applied to the
 *   result of the CSG operation.
 *
 * @SEE_ALSO
 *   SoShape
 *   
 *
 */
class INVENTOR_API SoCSGShape : public SoShape
{
  SO_NODE_HEADER( SoCSGShape );

public:

  /** Constructor */
  SoCSGShape();

  /** CSG operation to be used. */
  enum CsgOperationType
  {
    /** leftOperand union rightOperand. Remove useless faces and create edges if necessary. */
    ADD,
    /** LeftOperand - rightOperand. Remove useless faces and create edges if necessary. */
    SUB,  
    /** LeftOperand intersect rightOperand. Remove useless faces and create edges if necessary. */
    INTERSECTION,
    /** Keep only leftOperand. */
    LEFT_ONLY, 
    /** Keep only rightOperand. */
    RIGHT_ONLY,
    /** Keep nothing (result will be empty). May be useful for debugging. */
    NOTHING,
    /** Keep left and right primitives (concatenate all resulting faces). May be useful for debugging. */
    ALL
  };

  /** Geometry scene graph to be used as left operand for the Boolean operation. */ 
  SoSFNode leftOperand;
  
  /** Geometry scene graph to be used as right operand for the Boolean operation. */ 
  SoSFNode rightOperand;

  /** CSG boolean operation to apply.
   *  @useenum{CsgOperationType}. Default is ADD. 
   */
  SoSFEnum csgOperation;

  /** 
   * Status of CSG operation.
   * Status can be queried using the #getStatus() method.
   */
  enum CsgStatus
  {
    /** No error */
    CSG_NO_ERROR,
    /** CSGLib is not available. */
    CSG_LIB_NOT_AVAILABLE,
    /** an error occurred */
    CSG_ERROR,
  };

  /** 
   * If the CSG Operation failed for any reason, it will be retried by applying a small 
   * random transformation to the operand vertices. This can solve the problem if 
   * the geometries are well formed, but for example 2 faces are coplanar.
   * This method sets the max number of times SoCSGShape will retry before giving up 
   * and using the rescueOperation (see #setRescueOperation).
   * A max attempt of 5 is enough. If CSG Operation fails after
   * 5 attempts, it will probably fail all the time.
   *
   * Default is 5.
   *
   * @warning A value > 0 may change the coordinates of geometries as they
   * will be transformed a little bit to apply the CSG operation. This transformation
   * is normally not visible to the naked eye (it's about 0.01%), but may cause issues 
   * if you need precise values.
   *
   */
  void setMaxAttempt( int maxAttempt );

  /** 
   * Set operation to use if the CSG operation fails or CSGLib is not available. Default is NOTHING.
   * Must be one of LEFT_ONLY, RIGHT_ONLY, NOTHING or ALL. @useenum{CsgOperationType}.
   */
  static void setRescueOperation( CsgOperationType op );

  /** 
   * Returns status of csg operation.
   * @note Status is updated only when the node is traversed. Initially returns CSG_NO_ERROR.
   * @useenum{CsgStatus}. */
  CsgStatus getStatus();

SoEXTENDER public:
  // Implement actions

  /** @copydoc SoShape::doAction */
  virtual void doAction( SoAction *action );

  /** @copydoc SoShape::callback */
  virtual void callback( SoCallbackAction *action );

  /** @copydoc SoShape::GLRender */
  virtual void GLRender( SoGLRenderAction *action );

  /** @copydoc SoShape::rayPick */
  virtual void rayPick( SoRayPickAction *action );

  /** @copydoc SoShape::computeBBox */
  virtual void computeBBox( SoAction *action, SbBox3f &box, SbVec3f &center );

  /** @copydoc SoShape::computeBBox */
  virtual void computeBBox(SoAction *action, SbXfBox3d &box, SbVec3d &center);

  /** @copydoc SoShape::getPrimitiveCount */
  virtual void getPrimitiveCount( SoGetPrimitiveCountAction *action );

SoINTERNAL public:

  /** Register in database */
  static void initClass();

  /** Unregister from database */
  static void exitClass();

  /** catch some fields changes */
  virtual void notify( SoNotList *list );

protected:
  /** Destructor */
  virtual ~SoCSGShape();

  /** sentinel check */
  virtual void generatePrimitives ( SoAction *action );

private:

  /** Collapse points that are nearly equals (using eps as a threshold).
   * The result is returned as an indexed list of point.
   * indexedPoints are pointers to remainingPoints elements.
   * Note: It is always true that:
   * remainingPoints.size() <= points.size().
   * remainingPoints.size() == points.size() only when no points had been collapsed.
   * indexedPoints.size() == points.size() and are in the same order:
   * points[i] ~= *(indexedPoints[i])
   */
  void collapse(const std::vector<SoPrimitiveVertex>& points, float eps, std::vector<SbVec3f*>& indexedPoints, std::vector<SbVec3f>& remainingPoints) const;

  /** Apply a small random shift on each point. The random offset is base on the current
   * seed of SbMathHelper::srand(); */
  void shake(std::vector<SbVec3f>& points, float shift) const;
  
  /** Update m_shape if needed and return it.
   * The returned shape is expressed in world space depending on current state. Don't
   * forget to reset modelMatrix before using it. */
  SoFaceSet* getShape(SoState* state);

  /** Return list of elements that are used by specified node cache.
   * for example, if node contains a node that depend on SoComplexity (like SoSphere),
   * SoComplexityElement will be added to elements. 
   * return false if the results is incomplete because of abortion.
   */
  bool getCacheDependencies(SoAction* action, SoNode* node, GetCacheDeps::CachedElementsVector& elements);

  /** update (destroy and recreate if needed) specified cache with specified dependencies. */
  void updateCache( SoState *state, const GetCacheDeps::CachedElementsVector& dependencies, SoCache*& cache );

  /** Return true if cache is valid with specified state. */
  bool isCacheValid( SoState *state, SoCache* cache ) const;

  /** Generate triangles of specified node (using callback action). 
   * Transform is apply before calculating triangles. 
   * return false if the action has been aborted, and resulting triangles is not complete.
   */
  bool generateTriangles(SoAction* action, SoNode* node, std::vector<SoPrimitiveVertex>& triangles, SoTransformation* transform = NULL );

  /** 
   * Update m_triangles and m_shape using csgOperation between  m_trianglesLeft 
   * and m_trianglesRight. Of course, m_trianglesLeft and m_trianglesRight must
   * be up to date.
   * this method update m_csgStatus.
   */
  void updateResult();

  /** Callback used by triangleCallbackAction. */
  static void pushTriangleCallback(void* vec, 
                            SoCallbackAction* cb, 
                            const SoPrimitiveVertex* v1, 
                            const SoPrimitiveVertex* v2, 
                            const SoPrimitiveVertex* v3);

  /** 
   * Compute operation between trianglesLeft and triangleRight. Result is in trianglesResult. 
   * return success.
   */
  static CsgStatus computeCSG(const std::vector<SoPrimitiveVertex>& trianglesLeft, 
                        const std::vector<SoPrimitiveVertex>& trianglesRight, 
                        const CsgOperationType& operation, 
                        std::vector<SoPrimitiveVertex>& trianglesResult);


  /** Triangles of left operand. Updated when leftOperand has changed. 
   * It can be expressed in local or world space depending on m_expressedInWolrdSpace.*/
  std::vector<SoPrimitiveVertex> m_trianglesLeft;

  /** Triangles of right operand. Updated when rightOperand has changed. 
   * It can be expressed in local or world space depending on m_expressedInWolrdSpace.*/
  std::vector<SoPrimitiveVertex> m_trianglesRight;

  /** Cache of left triangle. Check if cache is valid with #isCacheValid().
   * If return false, m_trianglesLeft should be updated. */
  SoCache* m_leftCache;

  /** Cache of right triangle. Check if cache is valid with #isCacheValid().
   * If return false, m_trianglesRight should be updated. */
  SoCache* m_rightCache;

  /** Cache of result triangle. Check if cache is valid with #isCacheValid().
   * If return false, m_triangles should be updated. */
  SoCache* m_resultCache;

  /** True if triangles (ie m_trianglesLeft, m_trianglesRight, m_triangles and m_shape) are expressed 
   * in world space. This can be useful if one of operand depend on current model matrix
   * (like SoCircularExtrusion for example).
   * In this case, current model matrix should be set to identity before applying 
   * actions on m_shape. 
   * This value is updated in the getShape() method and correspond to the last computed shape. */ 
  bool m_expressedInWolrdSpace;

  /** 
   * Triangles of result. This correspond to m_shape triangles.
   * m_shape must be updated every time m_triangles is updated.
   * It can be expressed in local or world space depending on m_expressedInWolrdSpace.
   */
  std::vector<SoPrimitiveVertex> m_triangles;

  /**
   * Shape representing result of CSG operation. It will be updated when needed
   * And all actions will be redirected on it.
   * Don't call directly m_shape. Use the getShape() method instead.
   * Important note: 
   * It can be expressed in local or world space depending on m_expressedInWolrdSpace.
   * Don't forget to reset modelMatrix to identity before traversing m_shape if expressed
   * in world space.
   */
  SoFaceSet* m_shape;

  /** status of csg operation. Updated only once csgOperation had been apply, 
   * ie. when getShape had been called. */
  CsgStatus m_csgStatus;

  /** Mutex to protect access to m_shape */
  mutable SbThreadMutex m_mutex;

  /** nb of attempt before using rescue operation. See #setMaxAttempt for details. */
  int m_maxAttempt;

  /** Operation used in case of CSG operation failed. See #setRescueOperation. */
  static CsgOperationType s_rescueOp;

};

#endif // _SO_CSG_SHAPE_H_


