#if !defined _SO_CIRCEXTRSHAPE_H_
#define _SO_CIRCEXTRSHAPE_H_

/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      :  Roberto Calabrese (Feb 2011)
**=======================================================================*/

#include <Inventor/SbBasic.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFVec2f.h>
#include <Inventor/nodes/SoBaseExtrusion.h>
#include <Inventor/SbPImpl.h>

class SoState;
class SoAction;

SO_PIMPL_PUBLIC_DECLARATION(SoCircularExtrusion)

//@TOBEWRAPPED
/**
 * @VSGEXT Geometric shape formed by extruding a circle along a 3D spine.
 * 
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   The SoCircularExtrusion node is a geometric shape based on a constant #radius circle 
 *   cross section extruded along a three-dimensional polyline (spine).
 *   The circle center lies on the spine and the circle lies on the plane
 *   normal to the spine at each point.  This is similar to specifying an SoExtrusion
 *   with a circle cross-section, which is convenient for many applications, but this
 *   node has additional advantages because it responds to the SoComplexity value.
 *
 *   The application can specify which caps are created using the fields #beginCap and #endCap.
 *   By default both caps are visible.
 *
 *   The application can specify the visible portion of the shape along the spine 
 *   using normalized curvilinear coordinates (activeSection). By default the entire spine
 *   is visible.
 *
 *   The SoCircularExtrusion representation is dependent on SoComplexity::value.
 *   The shape is always simplified when the SoComplexity value is less than 1.
 *   Both the cross section (number of segments used for the circle) and the spine
 *   (number of points used for the spine) are affected by the simplification.
 *   The simplification algorithm approximates the spine by taking into account its curvature.
 *   The cross section circle is approximated by at least 10 points.
 *   With a SoComplexity value of 0 the extruded shape has the roughest representation.
 *
 *   Approximately the same geometry could be defined using an SoExtrusion node, but that
 *   node does not currently respond the SoComplexity value.  Likewise the clipping
 *   behavior could be obtained using two SoClipPlane nodes, but SoCircularExtrusion uses
 *   less memory when the activeSection is a subset of the full spine.  These two features
 *   make SoCircularExtrusion valuable when a large number of cylindrical objects must be
 *   displayed.
 *
 *   The #scaleMode field (inherited from SoBaseExtrusion) is used to select the points that
 *   will be scaled by the current
 *   transformation (for example SoTransform), if any.  Translation and rotation
 *   are applied in all cases.  The options are:
 *   - DEFAULT    : Every point in the generated geometry is scaled.
 *   - SPINE_ONLY : Only the spine points are scaled.
 *                  The cross section points are not scaled, so the cross 
 *                  section shape is undistorted even is scaling is non-uniform.
 *   - SECTION_ONLY : Only the cross section points are scaled. Not currently implemented.
 *   - NONE         : SoExtrusion ignores the scaling state. Not currently implemented.
 *
 *   The #extrusionMode field (inherited from SoBaseExtrusion) is used to select the extrusion algorithm. 
 *
 * @B Limitations:@b @BR
 *   The spine polyline segments must be C0 continous, that is, the spine must be connected at joints.
 *   The following segments may be C1 discontinous but be aware that very narrow angles between
 *   consecutive polyline semgments can produce odd results with SMOOTH #extrusionMode modes.
 *   This is especially noticeable when the angles between segments are very small with respect to the  #radius.
 *   In the following images the consecutive points A, B, C in the spine show one of these cases.
 *   The problem is at point B whose cross sections are abnormally large. 
 *   @TABLE_1B
 *         @TR Spine
 *         @TD Wireframe 
 *         @TD Shaded
 *         @TR @IMAGE OddSpine1.png
 *         @TD @IMAGE OddCircExtr1W.png
 *         @TD @IMAGE OddCircExtr1S.png
 *      @TABLE_END
 *
 *
 *   @NODE_SINCE_OIV 8.6
 *   
 * @FILE_FORMAT_DEFAULT
 *    CircularExtrusion {
 *    @TABLE_FILE_FORMAT
 *      @TR beginCap        @TD TRUE
 *      @TR endCap          @TD TRUE
 *      @TR spine           @TD [ 0 0 0, 0 1 0 ]
 *      @TR scaleMode       @TD DEFAULT
 *      @TR extrusionMode   @TD SMOOTH
 *      @TR radius          @TD [ 1 ]
 *      @TR activeSection   @TD [ 0, 1 ]
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *   SoExtrusion, SoComplexity, SoBaseExtrusion
 * 
 */
class INVENTOR_API SoCircularExtrusion : public SoBaseExtrusion
{
  SO_NODE_HEADER( SoCircularExtrusion );
  SO_PIMPL_PUBLIC_HEADER(SoCircularExtrusion)

public:

  enum SpineSimplificationMode
  {
    /** The spine simplification is done based only on geometric data.
     *  This is sufficient if there is a single material applied to the entire extrusion,
     *  but may produce incorrect results if multiple materials are applied. */
    DEFAULT,
    /** The spine simplification is done preserving the materials applied to the spine.  */
    RESPECT_MATERIALS,
  };

  // Fields
  /** Radius of the circular cross-section.
    * The default value is 1.
    */
  SoSFFloat radius;

  /** Defines the visible section of the spine in normalized coordinates.
   *  The default is (0,1) meaning that the entire spine is rendered.
   */
  SoSFVec2f activeSection;

  /**
  *  Defines the spine simplification behavior of the extrusion.
  *  @useenum{SpineSimplificationMode}. Default is DEFAULT.
  *  DEFAULT means spine simplification depends only on geometry.  This is appropriate
  *  in the typical case where a single material is applied to the entire extrusion.
  *  RESPECT_MATERIALS means spine simplification preserves multiple materials (if any)
  *  applied to the extrusion. This could reduce rendering performance.
  */
  SoSFEnum  spineSimplificationMode;

  /** Constructor */
  SoCircularExtrusion();

SoEXTENDER public:

  // Implement actions

  /** @copydoc SoShape::doAction */
  virtual void doAction( SoAction *action );

  /** @copydoc SoShape::callback */
  virtual void callback( SoCallbackAction *action );

  /** @copydoc SoShape::rayPick */
  virtual void rayPick( SoRayPickAction *action );

  /** @copydoc SoShape::getPrimitiveCount */
  virtual void getPrimitiveCount( SoGetPrimitiveCountAction *action );

  /** @copydoc SoShape::computeBBox */
  virtual void computeBBox( SoAction *action, SbBox3f &box, SbVec3f &center );

  /** @copydoc SoShape::computeBBox */
  virtual void computeBBox( SoAction *action, SbXfBox3d &box, SbVec3d &center );


 SoINTERNAL public:
  /** register in database */
  static void initClass();

  /** unregister from database */
  static void exitClass();

  /** catch some fields changes */
  virtual void notify( SoNotList *list );

protected:
  /** Destructor */
  virtual ~SoCircularExtrusion();

  /** Generates quads representing shape */
  virtual void generatePrimitives( SoAction *action );

};

#endif /* _SO_CIRCEXTRSHAPE_H_ */


