/*==============================================================================
**    THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),     **
**              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 **
**                                                                            **
**  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   **
**  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   **
**  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       **
**                                                                            **
**                        RESTRICTED RIGHTS LEGEND                            **
**  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  **
**  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  **
**  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  **
**  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  **
**  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         **
**                                                                            **
**                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                    **
**                              BORDEAUX, FRANCE                              **
**                            ALL RIGHTS RESERVED                             **
==============================================================================*/

#pragma once

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFVec3i32.h>
#include <Inventor/fields/SoSFTrigger.h>
#include <Inventor/renderer/RendererResourceMacro.h>

class SoGLRenderAction;
class SoNotList;

/**
 * @VSGEXT Compute shader scheduler node
 *
 * @ingroup ShaderNodes
 *
 * @DESCRIPTION
 *   This node allows the application to setup the invocation of a compute shader.
 *
 *   When traversed by a rendering action, this node launches one or more
 *   compute work groups. Each work group is processed by the current
 *   SoShaderProgram for the SoComputeShader stage.
 *
 *   See <a href="https://www.khronos.org/opengl/wiki/Compute_Shader">Compute Shader</a>
 *   section of the OpenGL wiki for more information about the execution model
 *   of compute shaders.
 *
 * @FILE_FORMAT_DEFAULT
 *   ComputeShaderScheduler {
 *     @TABLE_FILE_FORMAT
 *       @TR numWorkGroups  @TD 1 1 1
 *       @TR policy         @TD EACH_FRAME
 *     @TABLE_END
 *   }
 *
 * @SEE_ALSO
 *    SoComputeShader,
 *    SoShaderParameterBufferObject,
 *    SoShaderParameterImage,
 *    SoShaderProgram
 *
 * @NODE_SINCE_OIV 10.12
 */
class INVENTOR_API SoComputeShaderScheduler : public SoNode
{
  SO_NODE_HEADER(SoComputeShaderScheduler);
  RENDERER_RESOURCE(SoComputeShaderScheduler);

public:
  /**
   * Scheduling Policies
   */
  enum Policy
  {
    /**
     * The computation is executed each time this node is traversed by a rendering action.
     */
    EACH_FRAME,

    /**
     * The computation is executed only if the #trigger field is touched.
     *
     * See #trigger.
     */
    ON_TRIGGER,
  };

  /**
   * Specifies the number of local work groups dispatched in the X, Y and Z dimensions
   * for the computation.

   * Default value is (1, 1, 1)
   */
  SoSFVec3i32 numWorkGroups;

  /**
   * Specifies the policy used for scheduling the computation.
   *
   * @useenum{Policy}. Default value is #EACH_FRAME
   */
  SoSFEnum policy;

  /**
   * Trigger field to execute the shader computation.
   *
   * Effective only if #policy is #ON_TRIGGER.
   */
  SoSFTrigger trigger;

  SoComputeShaderScheduler();

  /**
   * Returns false. See SoNode::affectsState() for detail.
   */
  virtual SbBool affectsState() const
  {
    return FALSE;
  }

SoEXTENDER public:
  virtual void GLRender(SoGLRenderAction* action);

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  /**
   * Specifies the size of the local work groups in the X, Y and Z dimensions
   * for the computation.
   *
   * In order for this field to have any effect, the compute shader source must
   * set the local_size layout parameters with the following names:
   * \code
   * layout(local_size_x = OIV_COMPUTE_LOCAL_SIZE_X, local_size_y = OIV_COMPUTE_LOCAL_SIZE_Y, local_size_z = OIV_COMPUTE_LOCAL_SIZE_Z) in;
   * \endcode
   *
   * Default value is (1, 1, 1)
   */
  SoSFVec3i32 workGroupSize;

  virtual void fieldHasChanged(SoField* field);
  virtual void notify(SoNotList* list);

  virtual bool shouldCompute(SoGLRenderAction* action);

  /**
   * Calculates the optimal work group size and number of work groups for a
   * compute shader, taking hardware limitations into account.
   *
   * @param state the current state
   * @param totalComputeSize the total size of the computation in the X, Y and Z dimensions
   * @param numWorkGroups the calculated number of work groups to dispatch in the X, Y and Z dimensions
   * @param workGroupSize the calculated size of the local work groups in the X, Y and Z dimensions
   */
  static void computeWorkGroups(SoState* state, const SbVec3i32& totalComputeSize, SbVec3i32& numWorkGroups, SbVec3i32& workGroupSize);

protected:
  virtual ~SoComputeShaderScheduler();

private:
  bool m_triggerCalled;
};
