/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_DRAW_STYLE_
#define  _SO_DRAW_STYLE_

#include <Inventor/SbString.h>
#include <Inventor/elements/SoDrawStyleElement.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFUShort.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/nodes/SoNode.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoDrawStyle
//
//  Draw style node.
//
//////////////////////////////////////////////////////////////////////////////

class SoGLRenderAction;
class SoCallbackAction;
class SoRayPickAction;
class SoGetBoundingBoxAction;

/**
 * Node that defines the style to use when rendering.
 * 
 * @ingroup PropertyNodes
 * 
 * @DESCRIPTION
 *   This node defines the current drawing style for all subsequent shape nodes in a
 *   scene graph. SoDrawStyle specifies how primitives should be rendered. 
 *
 *   Draw style
 *
 *   The drawing style has no effect on picking or callback primitive generation.
 *   In particular, geometry with style INVISIBLE is not rendered but is still pickable.  
 *   It can be useful to put, for example, an invisible sphere around an assembly to
 *   ensure that it is always picked as a unit.
 *
 *   Style INVISIBLE should generally not be used to "turn off" geometry.  It is more
 *   efficient to put an SoSwitch node above the geometry and all its associated nodes.
 *   Using the Switch's whichChild field to "turn off" geometry avoids traversing any
 *   of the associated nodes.
 *
 *   Point size and line width notes
 *
 *   - The default point size and line width are both 0, which means to use the OpenGL
 *     default value (1 in both cases).  Point size and line width valuess greater than
 *     zero are considered to be specified in printer's points, where 1 inch = 72 printer's
 *     points.  However the default pixels-per-inch (managed by SbViewportRegion) is also 72.
 *     So effectively point size and line width are specified in pixels unless the
 *     application specifically sets a different pixels-per-inch value.
 *
 *   - The maximum line width is hardware dependent. Only width 1 is guaranteed.
 *     An application could simulate wider lines using polygons if necessary.
 *
 *   - If line antialiasing is disabled (which is the default), then OpenGL
 *     rounds the specified line width to the nearest integer. If rounding results in the
 *     value 0, it is as if the line width was set to 1.
 *
 *   - When the specified point size and/or line width is greater than zero, the SoDrawStyle
 *     node will have a dependency on the SoViewportRegionElement (for the pixels-per-inch
 *     value). As a result render caches may be invalidated when the viewport size changes,
 *     e.g. if the drawing window is resized.
 * 
 * @FILE_FORMAT_DEFAULT
 *    DrawStyle {
 *    @TABLE_FILE_FORMAT
 *       @TR style                    @TD FILLED
 *       @TR pointSize                @TD 0
 *       @TR lineWidth                @TD 0
 *       @TR linePattern              @TD 0xffff
 *       @TR linePatternScaleFactor   @TD 1
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction @BR
 *        Sets the current drawing style. 
 *        Sets: SoDrawStyleElement, SoPointSizeElement,
 *        SoLineWidthElement, SoLinePatternElement.
 * 
 * @SEE_ALSO
 *    SoLightModel,
 *    SoPickStyle,
 *    SoShapeHints,
 *    SoColorMask
 * 
 * 
 */
class INVENTOR_API SoDrawStyle : public SoNode {

  SO_NODE_HEADER(SoDrawStyle);
  
 public:
   /** Draw styles */
  enum Style {
    /**
     *  Draw filled regions 
     */
    FILLED = SoDrawStyleElement::FILLED,
    /**
     *  Draw only outlines (wireframe) 
     */
    LINES = SoDrawStyleElement::LINES,
    /**
     *  Draw points at vertices 
     */
    POINTS = SoDrawStyleElement::POINTS,
    /**
     *  Do not draw anything at all 
     */
    INVISIBLE = SoDrawStyleElement::INVISIBLE,
    /**
     *  Draw markers
     */
    MARKERS = SoDrawStyleElement::MARKERS
  };

  // Fields
  /**
   * Drawing style.
   * @useenum{Style}. Default is FILLED.
   */
  SoSFEnum style;

  /**
   * Radius of points (for POINTS style).
   * Default is 0, meaning to use the default OpenGL value (1).
   * Point sizes greater than zero are considered to be specified in printer's
   * point, where 1 inch = 72 printer's points. However the default pixels-per-inch
   * value (managed by SbViewportRegion) is also 72.  So effectively point size is
   * specified in pixels unless the application sets a different pixels-per-inch value.
   *
   * Please see the notes in the class description for more information.
   */
  SoSFFloat pointSize;

  /**
   * Width of lines (for LINES style).
   * The default line width is 0, meaning to use the default OpenGL value (1).
   * Line widths greater than zero are considered to be specified in printer's points,
   * where 1 inch = 72 printer's points.  However the default pixels-per-inch value (managed
   * by SbViewportRegion) is also 72.  So effectively line width is specified in pixels
   * unless the application sets a different pixels-per-inch value.
   *
   * Please see the notes in the class description for more information.
   */
  SoSFFloat lineWidth;

  /**
   * Stipple pattern for lines (for LINES style). 
   * This specifies how dashed or dotted lines will be drawn.
   * The pattern is a 16-bit series of 0s and 1s and is repeated
   * as necessary to stipple a given line. A 1 indicates that 
   * drawing occurs, and a 0 that it does not, on a 
   * pixel-by-pixel basis, starting with the low-order bits of the pattern.
   * Values can range from 0 (invisible) to 0xffff (solid). Default is 0xffff.
   *
   * The line pattern can be stretched using the #linePatternScaleFactor field.
   */
  SoSFUShort linePattern;

  /** 
   * Stipple pattern scale factor (for LINES style). It stretches the line pattern
   * (see #linePattern) by multiplying each subseries of consecutive 1s and 0s. 
   * Scale factors are clamped to lie between 1 and 255. Default is 1.
   *
   * @FIELD_SINCE_OIV 4.0
   */
  SoSFInt32 linePatternScaleFactor;

  /**
   * Creates a drawing style node with default settings.
   */
  SoDrawStyle();

  /**
   * Set the state of the override field.
   * see SoNode::setOverride doc.
   */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * Returns the state of the override field.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

 SoEXTENDER public:
  virtual void doAction(SoAction *action);
  virtual void GLRender(SoGLRenderAction *action);
  virtual void callback(SoCallbackAction *action);
  virtual void rayPick(SoRayPickAction *action);

  virtual void getBoundingBox(SoGetBoundingBoxAction *action);

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  SoSFBool override;

 protected:
  virtual ~SoDrawStyle();
};

#endif /* _SO_DRAW_STYLE_ */

