/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : David Mott (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_EVENT_CALLBACK_
#define  _SO_EVENT_CALLBACK_

#include <Inventor/misc/SoCallbackList.h>
#include <Inventor/SoPath.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/actions/SoHandleEventAction.h>

class SoEvent;
class SoEventCallback;

// callback function prototypes
/** 
 * @memberof SoEventCallback
 *
 * [OIV-WRAPPER NAME{EventCB}] 
 */
typedef void SoEventCallbackCB(void *userData, SoEventCallback *node);

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoEventCallback
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Node which invokes callbacks for events.
 * 
 * @ingroup nodes
 * 
 * @DESCRIPTION
 *   SoEventCallback will invoke application supplied \if_dotnet delegates \else callbacks \endif during
 *   SoHandleEventAction traversal. Methods allow the application to specify which
 *   Open Inventor events should trigger callbacks, and which path must be picked, if
 *   any, for the \if_dotnet delegate \else callback \endif invocation to occur. The application \if_dotnet delegate \else callback \endif is able to
 *   get information about the event and the pick detail, and may grab events,
 *   release events, and set whether the event was handled.
 *   
 *   If you register more than one \if_dotnet delegate \else callback \endif in an SoEventCallback node, all
 *   the \if_dotnet delegates \else callbacks \endif will be invoked when an event occurs, even if one of the
 *   callbacks handles the event. However, if the event is handled 
 *   (i.e. \if_dotnet SetHandled \else setHandled \endif is called) by any of the
 *   \if_dotnet delegates \else callbacks \endif, no subsequent node in the scene graph will see the event.
 *   Generally SoEventCallback nodes should be placed near the beginning of the scene graph, so the
 *   SoHandleEventAction does not need to traverse the rest of the scene graph if the event is handled.
 *   Note that events may be handled by other nodes in the scene graph, for example draggers and
 *   manipulators.  If the event is handled before the SoEventCallback node is traversed, then none of
 *   the \if_dotnet delegates \else callbacks \endif will be invoked.
 *
 *   Remember that when using the Open Inventor viewer classes, SoHandleEventAction traversal is
 *   only done when the viewer is in "selection" mode (the arrow cursor is displayed).  Also note
 *   that some methods may only be called from the \if_dotnet delegate \else callback \endif\.
 *
 *   When invoked, this SoEventCallback node will be passed to the \if_dotnet delegate \else callback \endif\.
 *   Using this object, you can get the event being handled (getEvent())
 *   and query what geometry is under the cursor (getPickedPoint()).  These are convenient wrappers
 *   around the corresponding methods in the SoHandleEventAction (which you can get using getAction()).  
 *
 *   Picking: @BR
 *   When getPickedPoint() is called, the handle event action automatically applies an SoRayPickAction 
 *   to the scene graph using the cursor position in the event. The handle event action 
 *   remembers the result in case another node needs pick information during the traversal.  Often this 
 *   is more convenient than creating and applying a pick action explicitly.
 *
 *   But note: The handle event action does @I not@i enable computation of normal vectors and texture
 *   coordinates in the pick action that it creates internally.  If you need the normal vector at the
 *   point of intersection of the pick ray, then you must create your own pick action and apply it.
 *   In this case you can get the node to apply the pick action to by calling the handle event action's
 *   getPickRoot() method.

 *   The application method can conveniently find out what, if any, geometry is
 *   under the cursor by querying the SoHandleEventAction (getAction()), then
 *   calling the getPickedPoint() method. The first time this method is called
 *   during a handle event traversal, the handle event action will automatically
 *   apply its internal SoRayPickAction to the scene graph returned by getPickRoot().
 *   The result is stored in case other nodes make the same query during the same
 *   traversal. The stored result can be cleared by calling clearApplyResult().
 *
 *   Some, but not all, options can be modified on the internal pick action (see for
 *   example setPickRadius()). Note that the internal pick action does not compute
 *   texture coordinates or normal vector for the picked point. Thus,
 *   getPickedPoint().getNormal() returns (0,0,0) and
 *   getPickedPoint().getTextureCoords() returns (0,0,0,0).
 *
 *   If your application needs to apply the pick action itself, for example to set different
 *   options, get the appropriate root node by calling SoHandleEventAction::getPickRoot().
 *
 * @EXAMPLE
 * \if_cpp
 *   An event callback node that handles key and button events
 *   \code
 *   SoEventCallback* eventCB = new SoEventCallback;
 *   eventCB->addEventCallback<SoKeyboardEvent>( keyEventCB );
 *   eventCB->addEventCallback<SoMouseButtonEvent>( btnEventCB );
 *   \endcode
 *   Handler for key press events
 *   \code
 *   void keyEventCB( void* userData, SoEventCallback* node ) {
 *     const SoEvent* evt = node->getEvent();
 *     if (SO_KEY_PRESS_EVENT(evt, UP_ARROW)) {
 *       // Do something, then tell action to stop traversal
 *       node->setHandled();
 *     } else if (SO_KEY_PRESS_EVENT(evt, DOWN_ARROW)) {
 *       // Do something, then tell action to stop traversal
 *       node->setHandled();
 *     }
 *   }
 *   \endcode
 *   Handler for mouse button events
 *   \code
 *   void btnEventCB( void* userData, SoEventCallback* node ) 
 *   {
 *     const SoEvent* evt = node->getEvent();
 *   
 *     if (SO_MOUSE_PRESS_EVENT( evt, BUTTON1 )) {
 *       // Check if any geometry is under the cursor
 *       SoHandleEventAction* action   = node->getAction();
 *       const SoPickedPoint* pickedPt = action->getPickedPoint();
 *       if (pickedPt != NULL) {
 *         SoFullPath* pickedPath = (SoFullPath*)pickedPt->getPath();
 *         SoNode*     pickedNode = pickedPath->getTail();
 *       }
 *     }
 *   }
 *   \endcode
 * \endif
 * \if_dotnet
 *   An event callback node that handles key and button events
 *   \code
 *   SoEventCallback eventNode = new SoEventCallback();
 *   eventNode.AddEventCallback( typeof(SoKeyboardEvent)   , new SoEventCallback.EventCB(keyEventCB) );
 *   eventNode.AddEventCallback( typeof(SoMouseButtonEvent), new SoEventCallback.EventCB(btnEventCB) );
 *   . . .
 *   \endcode
 *   Handler for key press events
 *   \code
 *   private void keyEventCB( SoEventCallback node )
 *   {
 *     SoKeyboardEvent evt = (SoKeyboardEvent)node.GetEvent();
 *     if (evt.GetKey() == SoKeyboardEvent.Keys.UP_ARROW)
 *     {
 *       // Do something, then tell action to stop traversal
 *       node.SetHandled();
 *     }
 *     else if (evt.GetKey() == SoKeyboardEvent.Keys.DOWN_ARROW)
 *     {
 *       // Do something, then tell action to stop traversal
 *       node.SetHandled();
 *     }
 *   }
 *   \endcode
 *   Handler for mouse button events
 *   \code
 *   private void btnEventCB( SoEventCallback node )
 *   {
 *       SoEvent evt = node.GetEvent();
 *       // If button 1 was pressed
 *       if (SoMouseButtonEvent.IsButtonPressEvent( evt, SoMouseButtonEvent.Buttons.BUTTON1 ))
 *       {
 *           // Check if any geometry is under the cursor
 *           SoHandleEventAction action = node.GetAction();
 *           SoPickedPoint     pickedPt = action.GetPickedPoint();
 *           if (pickedPt != null)
 *           {
 *               SoPath pickedPath = pickedPt.GetPath();
 *               SoNode pickedNode = pickedPath.FullPath.GetTail();
 *           }
 *       }
 *   }
 *   \endcode
 * \endif
 * \if_java
 *   An event callback node that handles key and button events
 *   \code
 *   SoEventCallback  eventNode = new SoEventCallback();
 *   eventNode.addEventCallback( SoKeyboardEvent.class   , new KeyEventHandler() );
 *   eventNode.addEventCallback( SoMouseButtonEvent.class, new BtnEventHandler() );
 *   \endcode
 *   Handler for key press events
 *   \code
 *   class KeyEventHandler extends SoEventCallbackCB {
 *        @Override
 *      public void invoke( SoEventCallback node ) {
 *          // Get the event that triggered the callback
 *          SoEvent evt = node.getEvent();
 *          if (SoKeyboardEvent.isKeyPressEvent(evt, SoKeyboardEvent.Keys.UP_ARROW)) {
 *              // Do something, then tell action to stop traversal
 *              node.setHandled();
 *          }
 *          else if (SoKeyboardEvent.isKeyPressEvent(evt, SoKeyboardEvent.Keys.DOWN_ARROW)) {
 *              // Do something, then tell action to stop traversal
 *              node.setHandled();
 *          }
 *      }
 *   }
 *   \endcode
 *   Handler for mouse button events
 *   \code
 *   class BtnEventHandler extends SoEventCallbackCB {
 *      @Override
 *      public void invoke( SoEventCallback node ) {
 *          SoEvent evt = node.getEvent();
 *          // If button 1 was pressed
 *          if (SoMouseButtonEvent.isButtonPressEvent( evt, SoMouseButtonEvent.Buttons.BUTTON1 ))
 *          {
 *              // Check if any geometry is under the cursor
 *              SoHandleEventAction action = node.getAction();
 *              SoPickedPoint     pickedPt = action.getPickedPoint();
 *              if (pickedPt != null)
 *              {
 *                  SoPath pickedPath = pickedPt.getPath();
 *                  SoNode pickedNode = pickedPath.full.getTail();
 *              }
 *          }
 *      }
 *   }
 *   \endcode
 * \endif
 * 
 * @FILE_FORMAT_DEFAULT
 *    EventCallback {
 *    @TABLE_FILE_FORMAT
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *    SoInteraction,
 *    SoSelection,
 *    SoHandleEventAction,
 *    SoRayPickAction,
 *    SoDragger
 * 
 * 
 */
class INVENTOR_API SoEventCallback : public SoNode {

  SO_NODE_HEADER(SoEventCallback);

 public:
  /**
   * Constructor creates an event callback node with no event interest and a NULL
   * path.
   */
  SoEventCallback();
    
  /**
   * Sets the path which must be picked in order for the callbacks to be
   * invoked. If the path is NULL, the callbacks will be invoked for every
   * interesting event, as specified by addEventCallback(), regardless of what
   * is picked. The setPath() method makes its own copy of the passed path.
   */
  void                setPath(SoPath *path);
  /**
   * Gets the path which must be picked in order for the callbacks to be
   * invoked.
   */
  const SoPath *      getPath()               { return pathOfInterest; }

  /**
   * Add an event \if_dotnet delegate \else callback \endif\.
   * Specifies the \if_dotnet delegates \else callbacks \endif to be invoked for different event types. When
   * invoked, the \if_dotnet delegate \else callback \endif will be passed the userData, along with a
   * \if_dotnet reference \else pointer \endif to this SoEventCallback node.
   * For example, passing the type of
   * SoMouseButtonEvent means callbacks will be invoked only
   * when a mouse button is pressed or released. Passing the type of
   * SoEvent for the eventType will cause the callback to be
   * invoked for every event which passes through this event callback node.
   */
  void                addEventCallback(SoType eventType,
                                       SoEventCallbackCB *f,
                                       void *userData = NULL);

  /**
   * @copydoc SoEventCallback::addEventCallback(SoType,SoEventCallbackCB*,void*)
   *
   * @M_SINCE 10.8
   * [OIV-WRAPPER-NOT-WRAP]
   */
  template<typename EventType>
  void addEventCallback( SoEventCallbackCB* f, void* userData = nullptr )
  {
    addEventCallback( EventType::getClassTypeId(), f, userData );
  }

  /**
   * Removes a previously specified event \if_dotnet delegate \else callback \endif\.
   */
  void                removeEventCallback(SoType eventType,
                                          SoEventCallbackCB *f,
                                          void *userData = NULL);

  /**
   * @copydoc SoEventCallback::removeEventCallback(SoType,SoEventCallbackCB*,void*)
   *
   * @M_SINCE 10.8
   * [OIV-WRAPPER-NOT-WRAP]
   */
  template<typename EventType>
  void removeEventCallback( SoEventCallbackCB* f, void* userData = nullptr )
  {
    removeEventCallback( EventType::getClassTypeId(), f, userData );
  }

  //////////////////////
  //
  // These all provide information to callback functions. They
  // return NULL when called from anywhere but a callback function.
  //
    
  /**
   * Returns the SoHandleEventAction currently traversing this node, or NULL if
   * traversal is not taking place. This should be called only from \if_dotnet delegate \else callback \endif\.
   */
  SoHandleEventAction *       getAction() const { return eventAction; }

  /**
   * Returns the event currently being handled, or NULL if traversal is not taking
   * place. This should be called only from \if_dotnet delegate \else callback \endif\.
   */
  const SoEvent *             getEvent() const
    { return (eventAction != NULL ? eventAction->getEvent() : NULL); }

  /**
   * Returns pick information during SoHandleEventAction traversal, or NULL if
   * traversal is not taking place. This should be called only from \if_dotnet delegate \else callback \endif\.
   *
   * When this method is called, an SoRayPickAction is automatically applied to the
   * scene graph that the SoHandleEventAction is traversing, using the current event.
   * However this is only done once for each SoHandleEventAction traversal and the
   * result is cached for subsequent queries during the current traversal.
   *
   * Note: The handle event action does @I not@i enable computation of normal vectors and texture
   *   coordinates in the pick action that it creates internally.  If you need the normal vector at the
   *   point of intersection of the pick ray, then you must create your own pick action and apply it.
   *   In this case you can get the node to apply the pick action to by calling the handle event action's
   *   getPickRoot() method.
   */
  const SoPickedPoint *       getPickedPoint() const
    { return (eventAction != NULL ? eventAction->getPickedPoint() : NULL);}
    
  //
  //////////////////////

  /**
   * Tells the node the event was handled. The \if_dotnet delegate \else callback \endif is responsible for
   * setting whether the event was handled or not. If there is more than one \if_dotnet delegate \else callback \endif
   * registered with an SoEventCallback node, all of them will be invoked,
   * regardless of whether one has handled the event or not. This should be called
   * only from \if_dotnet delegate \else callback \endif\.
   */
  void                setHandled()
    { if (eventAction != NULL) eventAction->setHandled(); }

  /**
   * Returns whether the event has been handled. This should be called only from
   * \if_dotnet delegate \else callback \endif\.
   */
  SbBool              isHandled() const
    { return (eventAction != NULL) ? eventAction->isHandled() : FALSE; }
    
  /**
   * Tells the event callback node to grab events. While
   * grabbing, the node will consume all events; however, each \if_dotnet delegate \else callback \endif will
   * only be invoked for events of interest.
   */
  void                grabEvents()
    { if (eventAction != NULL) eventAction->setGrabber(this); }

  /**
   * Tells the event callback node to release the grab of events. Event grabbing
   * is requested using grabEvents().
   */
  void                releaseEvents()
    { if (eventAction != NULL) eventAction->releaseGrabber(); }

 SoEXTENDER public:
  // This will be called during handleEventAction traversal.
  virtual void        handleEvent(SoHandleEventAction *ha);

 SoINTERNAL public:
  static void         initClass();    // initialize the class
  static void         exitClass();

 protected:
  // Destructor - protected since ref/unref is what should destroy this
  virtual ~SoEventCallback();
    
 private:
  // Only invoke callbacks if this path was picked.
  // If path is NULL, always invoke callbacks.
  SoPath              *pathOfInterest;
    
  // List of callback functions, event types, and user data.
  SbPList *           cblist;
    
  // This is set for each SoHandleEventAction traversal of this node
  // so that the apps callback routine can invoke methods on the action.
  SoHandleEventAction *eventAction;
};

#endif /* _SO_EVENT_CALLBACK_ */

