/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#ifndef _SO_IMAGE_BACKGROUND_H_
#define _SO_IMAGE_BACKGROUND_H_

#include <Inventor/fields/SoSFFilePathString.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFImage.h>
#include <Inventor/SbString.h>

#include <Inventor/nodes/SoBackground.h>

class SoFieldSensor;
class SbRasterImage;

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

SO_PIMPL_PUBLIC_DECLARATION(SoImageBackground)

/**
 * Image background node
 *
 * @ingroup BackgroundNodes
 *
 * @DESCRIPTION
 *  Draws a background image.
 *
 *  This node provides a convenient way of rendering an image
 *  in the background of the scene. The position options like
 *  LOWER_LEFT can be used, for example, to place a logo in
 *  the corner of the window. The STRETCH and TILE options
 *  cause the image to fill the window and automatically adjust
 *  if the window size changes.
 *
 *  Note that the SoImage node can also be used to place an
 *  image in the scene, but the position of the image is
 *  specified in 3D coordinates. This node positions images
 *  relative to the physical drawing window.
 *
 *  Generally background nodes should be placed at the
 *  beginning of the scene graph so that geometry is drawn on
 *  top of the background. Background nodes have no size, are
 *  not pickable, and do not modify the OpenGL depth buffer.
 *
 *  Another convenient feature is that this node renders
 *  correctly on a tiled display that represents a single large
 *  virtual window, for example using the MultiPipe extension
 *  (see SoXtMPExaminerViewer). The image will be stretched or
 *  tiled across the entire virtual window.
 *
 * @FILE_FORMAT_DEFAULT
 *    ImageBackground {
 *    @TABLE_FILE_FORMAT
 *       @TR filename                 @TD ""
 *       @TR image                    @TD 0 0 0 0
 *       @TR style                    @TD TILE
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Renders background, does not affect state.
 *
 * @SEE_ALSO
 *    SoBackground,
 *    SoGradientBackground
 *
 *
 */
class INVENTOR_API SoImageBackground : public SoBackground {

  SO_PIMPL_PUBLIC_HEADER(SoImageBackground)
  SO_NODE_HEADER(SoImageBackground);

public:

  /** Image background style */
  enum Style {
    /**
     *  No image is rendered.
     */
    NONE       ,
    /**
     *  Image is rendered actual size, centered in the window @BR
     *  (image aspect ratio is preserved but image may be cropped and background may be visible).
     */
    CENTER     ,
    /**
     *  Image is rendered actual size in lower left corner @BR
     *  (image aspect ratio is preserved but image may be cropped and background may be visible).
     */
    LOWER_LEFT ,
    /**
     *  Image is rendered actual size in upper left corner @BR
     *  (image aspect ratio is preserved but image may be cropped and background may be visible).
     */
    UPPER_LEFT ,
    /**
     *  Image is rendered actual size in upper right corner @BR
     *  (image aspect ratio is preserved but image may be cropped and background may be visible).
     */
    UPPER_RIGHT,
    /**
     *  Image is rendered actual size in lower right corner @BR
     *  (image aspect ratio is preserved but image may be cropped and background may be visible).
     */
    LOWER_RIGHT,
    /**
     *  Image is stretched to exactly fill the viewport in both directions @BR
     *  (image aspect ratio is @I not@i preserved).
     */
    STRETCH    ,
    /**
     *  Image is tiled to fill window starting from lower left corner @BR
     *  (image aspect ratio is preserved).
     */
    TILE,
    /**
     *  Image is centered and stretched to exactly fill the longer dimension of the viewport (width or height) @BR
     *  (image aspect ratio is preserved but portions of image will be cropped if the aspect
     *  ratio of the image and viewport are different).
     */
    CROP,
    /**
     *  Image is centered and stretched to exactly fill the shorter dimension of the viewport (width or height) @BR
     *  (image aspect ratio is preserved but background color will be visible on sides or top
     *  and bottom of the image if the aspect ratio of the image and viewport are different).
     */
    FIT
  };

   // Fields:
   //
  /**
   * Names file from which to read texture image. The standard image file formats are
   * supported. See SoRasterImageRW for the list. If the
   * filename is not an absolute path name, the list of directories maintained by
   * SoInput is searched. If the texture is not found in any of those directories,
   * then the file is searched for relative to the directory from which the
   * node was read. For example, if a node with a filename of
   * "../tofu.rgb" is read from /usr/people/bob/models/food.iv, then
   * /usr/people/bob/tofu.rgb will be read (assuming tofu.rgb isn't found in the
   * directories maintained by SoInput).
   */
  SoSFFilePathString       filename;

  /**
   * Contains an in-memory representation of the texture map. It is either the
   * contents of the file read from #filename, an image read directly from an
   * Open Inventor file, or an image set programmatically using the methods provided
   * by SoSFImage.
   */
  SoSFImage        image;

  /**
   * Specifies how the image will be mapped onto the window.
   * @useenum{Style}. Default is TILE.
   */
  SoSFEnum         style;

  /**
   * Creates a background image node with default settings.
   */
  SoImageBackground();

SoINTERNAL public:
  // Initializes this class for use in scene graphs. This
  // should be called after database initialization and before
  // any instance of this node is constructed.
  static void    initClass();
  static void    exitClass();

  /** Notify */
  virtual void notify( SoNotList *list );

SoEXTENDER public:
  // Implement render action
  // We will inherit all other action methods from SoNode.
  // We don't need a doAction because this node has no effect
  // on traversal state (or at least it shouldn't :-).
  virtual void   GLRender(SoGLRenderAction *action);

protected:
  // Destructor. Protected to keep people from trying to delete
  // nodes, rather than using the reference count mechanism.
  virtual ~SoImageBackground();

  // Reads stuff into instance. Returns FALSE on error.
  virtual SbBool readInstance(SoInput *in, unsigned short flags);

private:
  void commonConstructor();
};

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif //SoImageBackground

