/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : AJO (May 2011)
**=======================================================================*/


#ifndef  _SO_INDEXED_FACE_SET_
#define  _SO_INDEXED_FACE_SET_

#include <Inventor/nodes/SoIndexedShape.h>
#include <Inventor/sys/SoGLType.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoIndexedFaceSet
//
//  Indexed set of faces. Each face consists of 3 or more vertices,
//  each of which is denoted by an integer index (from the coordIndex
//  field) into the current coordinates. Depending on the current
//  material and normal binding values, the materials and normals for
//  the faces or vertices may be accessed in order or indexed. If they
//  are indexed, the materialIndex and normalIndex fields are used.
//
//////////////////////////////////////////////////////////////////////////////


// This coordinate index indicates that the current face ends and the
// next face begins.  Use this for simple (1 contour) polygons.
#define SO_END_FACE_INDEX       (-1)

// If SoShapeHints is used with a windingType other than NO_WINDING_TYPE,
// this coordinate index indicates that the current contour ends
// and the next contour begins.
#define SO_END_CONTOUR_INDEX    (-1)

// This coordinate index indicates that the current complex face (block
// of contours) ends.
#define SO_END_POLYGON_INDEX    (-2)

class SoNormalBundle;
class SoShapeStyleElement;
class SoVolumeIndexedFaceSet;

SO_PIMPL_PUBLIC_DECLARATION(SoIndexedFaceSet)

/**
* Indexed polygonal face shape node.
* 
* @ingroup ShapeNodes
* 
* @DESCRIPTION
*   This node represents a 3D shape formed by constructing faces (polygons) from
*   vertices located at the coordinates specified in the #vertexProperty field
*   (from SoVertexShape), or the current inherited state. For optimal performance,
*   the #vertexProperty field is recommended.
*
*   SoIndexedFaceSet is a general purpose polygon rendering node.  It can render 
*   triangles, quadrilaterals, and other types of polygons or any combination thereof.
*
*   Tips:
*   - For best performance, arrange the faces with all triangles first, then all quadrilaterals, 
*     then general polygons.
*   - If the shape has only triangles, use SoIndexedTriangleSet instead (simpler and possibly faster).
*
*   SoIndexedFaceSet supports both convex and non-convex polygons (unlike OpenGL).
*   However polygons are assumed to be convex by default. Rendering non-convex
*   polygons in this mode may produce incorrect results.  Enable non-convex
*   polygons using the faceType field of an SoShapeHints node.  In this mode
*   SoIndexedFaceSet will automatically convert non-convex polygons, if necessary,
*   to faces that the hardware can handle.  
*
*   SoIndexedFaceSet can also directly
*   handle complex polygons with "holes".  See the following paragraphs.
*
*   If SoShapeHints is not used, or if windingType field from SoShapeHints is equal to 
*   NO_WINDING_TYPE, SoIndexedFaceSet uses the indices in the #coordIndex field (from
*   SoIndexedShape) to specify simple polygonal faces. An index of SO_END_FACE_INDEX
*   (-1) indicates that the current face has ended and the next one begins.
*
*   If SoShapeHints is used with a windingType field other than NO_WINDING_TYPE, 
*   SoIndexedFaceSet uses the indices in the coordIndex field (from SoIndexedShape)
*   to specify complex polygonal faces (possibly) made up of multiple contours.
*   An index of SO_END_CONTOUR_INDEX (-1) indicates that the current
*   contour has ended and that the next one begins. An index of SO_END_POLYGON_INDEX (-2)
*   indicates that the current complex face (group of contours) has ended.
*   After the end of a complex face, you can create another complex face, and so on.
*
* \if_cpp
*   @B Skeleton to create a polygon with holes@b
*  \code
*   // 1) Choose a winding rule with windingType field from SoShapeHints.
*
*            SoShapeHints *myShapeHints = new SoShapeHints;
*            myShapeHints->windingType = SoShapeHints::ODD_TYPE;
*
*   // Create list of contours.
*
*            static int32_t indices[21] =  {
*                        0,  3,  1,      SO_END_CONTOUR_INDEX,  
*                        5,  6,  4,      SO_END_POLYGON_INDEX, // To end the first polygon.
*                        0,  7,  3,      SO_END_CONTOUR_INDEX,
*                        10, 9,  8,      SO_END_CONTOUR_INDEX,
*                        9,  7,  0,  8,  SO_END_POLYGON_INDEX  // To end the second polygon.
*                        };
*
*   // Note: The last polygon must end with either SO_END_POLYGON_INDEX or SO_END_CONTOUR_INDEX or nothing
*
*            static int32_t indices[21] =  {
*                        0,  3,  1,      SO_END_CONTOUR_INDEX,  
*                        5,  6,  4,      SO_END_POLYGON_INDEX, 
*                        0,  7,  3,      SO_END_CONTOUR_INDEX,
*                        10, 9,  8,      SO_END_CONTOUR_INDEX,
*                        9,  7,  0,  8
*                        };
*  \endcode
* \endif
*
*   The vertices of the faces are transformed by the current transformation matrix.
*   The faces are drawn with the current light model and drawing style.
*   Diffuse color and opacity may be specified using the
*   @I orderedRGBA@i field of SoVertexProperty. Diffuse color, transparency and
*   other color parameters used in the lighting equation may be specified using
*   SoMaterial.
*   
*   Treatment of the current material and normal binding is as follows: The PER_PART
*   and PER_FACE bindings specify a material or normal for each face. PER_VERTEX
*   specifies a material or normal for each vertex. The corresponding _INDEXED
*   bindings are the same, but use the #materialIndex or #normalIndex
*   indices (see SoIndexedShape). The default material binding is OVERALL. The
*   default normal binding is PER_VERTEX_INDEXED.
*   
*   If any normals (or materials) are specified, Open Inventor assumes you provide
*   the correct number of them, as indicated by the binding. You will see unexpected
*   results if you specify fewer normals (or materials) than the shape requires. If
*   no normals are specified, they will be generated automatically.
*
*   Lighting is enabled by default (see SoLightModel).  If lighting is enabled, by
*   default only the "front face" of polygons are lit (the back face will be black).
*   Which side is the front face is determined by the normal vector, if specified,
*   else by the ordering of the vertices and the @I vertexOrdering@i field of
*   SoShapeHints. You can also enable "two sided lighting" using SoShapeHints.
*
*   If lighting is enabled and no normals are specified, they will be generated automatically.
*   The normal generator computes "facet" or "smooth" (or a combination) normal vectors
*   depending on the @I creaseAngle@i field of SoShapeHints.
*
*   Textures may be applied using (for example) SoTexture2. Texture coordinates may be
*   supplied using SoVertexProperty or SoTextureCoordinate2.  If a texture is applied
*   and no texture coordinates were specified, they will be computed automatically.
*   By default the computed coordinates map the texture to the axis-aligned bounding box
*   of the shape, with the texture S axis corresponding to the longest axis of the box.
*
*   @B Warning@b: The shape may not render or pick correctly if any of its polygons are
*   self-intersecting or non-planar.  
*
*   Limitations:
*   - Currently does not use OpenGL VBOs (vertex buffer objects) for rendering (and
*     performance may be lower) when both texture coordinates and texture coordinate
*     indices are specified.  Just setting texture coordinates (indexed by the coordIndex
*     values) is OK.
*
* @FILE_FORMAT_DEFAULT
*    IndexedFaceSet {
*    @TABLE_FILE_FORMAT
*       @TR vertexProperty      @TD NULL
*       @TR coordIndex          @TD -1
*       @TR materialIndex       @TD -1
*       @TR normalIndex         @TD -1
*       @TR textureCoordIndex   @TD -1
*    @TABLE_END
*    }
* 
* @ACTION_BEHAVIOR
*    SoGLRenderAction @BR
*        Draws faces based on the current coordinates, normals, materials, drawing
*        style, and so on.
* 
*    SoRayPickAction @BR
*        Picks faces based on the current coordinates and transformation. Details about
*        the intersection are returned in an SoFaceDetail.
* 
*    SoGetBoundingBoxAction @BR
*        Computes the bounding box that encloses all vertices of the face set with the
*        current transformation applied to them. Sets the center to the average of the
*        coordinates of all vertices.
* 
*    SoCallbackAction @BR
*        If any triangle callbacks are registered with the action, they will be invoked
*        for each successive triangle generated from each face in the set.
* 
* 
* @SEE_ALSO
*    SoCoordinate3,
*    SoDrawStyle,
*    SoFaceDetail,
*    SoFaceSet,
*    SoFullSceneAntialiasing,
*    SoIndexedTriangleSet,
*    SoShapeHints,
*    SoVertexProperty
* 
*/
class INVENTOR_API SoIndexedFaceSet : public SoIndexedShape
{
  SO_NODE_HEADER(SoIndexedFaceSet);
  SO_PIMPL_PUBLIC_HEADER(SoIndexedFaceSet)

public:

  /**
   * Creates an indexed face set node with default settings.
   */
  SoIndexedFaceSet();

SoEXTENDER public:
  // Implements actions
  virtual void GLRender( SoGLRenderAction* action );

 // Generates default normals using the given normal bundle. 
  // Returns TRUE.
  virtual void generateDefaultNormals( SoState* state,
                                       const SbVec3f* coords,
                                       int numCoords,
                                       SoNormalBundle* nb,
                                       SbBool storeRef = FALSE);

  /** Disable tangents generation for triangle shapes */
  virtual void shouldGenerateTangents( SoGLRenderAction* /*action*/, const SoShapeStyleElement* /*shapeStyle*/ ) {}

  virtual void getPrimitiveCount( SoGetPrimitiveCountAction* action );
  virtual void rayPick(SoRayPickAction *action);
                                                  
SoINTERNAL public:

  static void initClass();
  static void exitClass();
  
  // set numTris/Quads/Faces to -1 when notified
  virtual void notify( SoNotList* list );

  // Compute the number of vertices. This is just an approximation,
  //so using 5 verts per unknown polygon is close enough. 
  virtual int getNumPrim( SoState* state = NULL );

 protected:
  // This enum is used to indicate the current material or normal binding
  enum Binding {
    OVERALL,
    PER_FACE,
    PER_FACE_INDEXED,
    PER_VERTEX,
    PER_VERTEX_INDEXED
  };

  //Whether to check or not shapeHints
  virtual SbBool shouldCheckShapeHints(){ return TRUE;}

  virtual SbBool shouldGenerateTexture3DCoordinates(){return TRUE;}

  virtual void callSetupIndices( SoState* state, const SoShapeStyleElement* shapeStyle, uint32_t useTexCoordsAnyway );

  //Whether to check or not 3d tex coords
  virtual SbBool checkTex3Coords(){return TRUE;}

  virtual SbBool shapeHintsTest( SoState* state = NULL );

  int getNumPrim( int& _numTris, int& _numQuads, int& _numFaces );

  virtual void countPrim( SoState* state = NULL );

  // Generates triangles representing faces
  virtual void generatePrimitives( SoAction* action );

  // render function
  virtual void doRendering( SoGLRenderAction* action, const SoShapeStyleElement* shapeStyle );

  // Overrides standard method to create an SoFaceDetail instance
  virtual SoDetail* createTriangleDetail( SoRayPickAction* action,
                                          const SoPrimitiveVertex* v1,
                                          const SoPrimitiveVertex* v2,
                                          const SoPrimitiveVertex* v3,
                                          SoPickedPoint* pp );

  //When tesselating complex polygons, keep the result cached
  virtual bool hasGeneratePrimitiveCache()
  {
    return true;
  }

  // destructor
  virtual ~SoIndexedFaceSet();
};

#endif /* _SO_INDEXED_FACE_SET_ */


