/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : F. DELLA MAESTRA (Sep 2005)
**=======================================================================*/

#ifndef  _SO_INDEXED_POINT_SET_
#define  _SO_INDEXED_POINT_SET_

#include <Inventor/nodes/SoIndexedShape.h>
#include <Inventor/SbPImpl.h>

SO_PIMPL_PUBLIC_DECLARATION(SoIndexedPointSet)

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoIndexedPointSet
//
//  Indexed set of points. Each point consists of one
//  vertex, which is denoted by an integer index (from the
//  coordIndex field) into the current coordinate. Depending on the
//  current material and normal binding values, the materials and
//  normals for the vertices may be accessed in order or
//  indexed. If they are indexed, the materialIndex and normalIndex
//  fields are used. /*Texture coordinates may also be indexed, using
//  the textureCoordIndex field.*/
//
//////////////////////////////////////////////////////////////////////////////
//#define SO_END_POINT_INDEX (-1)

/**
* 
* Indexed point set shape node.
* 
* @ingroup ShapeNodes
* 
* @DESCRIPTION
*
*   This node represents a set of points located at the coordinates specified by the
*   #vertexProperty field (from SoVertexShape) or the current inherited
*   coordinates. For optimal performance, the #vertexProperty field is
*   recommended.
*
*   Each point consists of one vertex, which is denoted by an integer index (from the
*   #coordIndex field) into the current coordinates.  The number of values in the
*   coordIndex field defines the number of points. Depending on the
*   current material and normal binding values, the materials and
*   normals for the vertices may be accessed in order or
*   indexed. If they are indexed, the materialIndex and normalIndex
*   fields are used.
*   
*   The coordinates of the point set are transformed by the current cumulative
*   transformation. 
*
*   Lighting is not applied to points (i.e., they are rendered with light model BASE_COLOR)
*   unless the application explicitly sets normal vectors using SoVertexProperty or SoNormal.
*   Note that this is specific to direct rendering of point sets.  If you render polygonal
*   geometry, e.g. SoFaceSet, in POINTS mode (SoDrawStyle), normal vectors are computed by
*   Open Inventor and lighting is enabled.
*
*   NOTE: If you want to apply lighting in a shader, you still need to explicitly set normal
*   vectors, otherwise lighting will be disabled for the point set.
*
*   Point size is controlled by the SoDrawStyle field @I pointSize@i.
*
*   Automatic simplification (by skipping points) is controlled by the SoComplexity
*   field @I value@i.  Only values < 0.5 have any effect.
*   
*   Treatment of the current material and normal binding is as follows: PER_PART,
*   PER_FACE, and PER_VERTEX bindings bind one material or normal to each point. The
*   default material binding is OVERALL. The default normal binding is PER_VERTEX.
*
*   Shape Antialiasing type is SoShape::POINTS.
*
*   Limitations:
*   - The OpenGL VBO (vertex buffer object) rendering model only allows one set of indices for a primitive.
*     Therefore it is not possible to use VBO rendering (and performance may be lower) if indices
*     are set in the @I materialIndex@i, @I normalIndex@i or @I textureCoordIndex@i fields.
*
* @FILE_FORMAT_DEFAULT
*    IndexedPointSet {
*    @TABLE_FILE_FORMAT
*       @TR vertexProperty      @TD NULL
*       @TR coordIndex          @TD -1
*       @TR materialIndex       @TD -1
*       @TR normalIndex         @TD -1
*       @TR textureCoordIndex   @TD -1
*    @TABLE_END
*    }
* 
* @ACTION_BEHAVIOR
*    SoGLRenderAction @BR
*        Draws points based on the current coordinates, normals, materials, drawing
*        style, and so on.
* 
*    SoRayPickAction @BR
*        Picks points based on the current coordinates and transformation. Details about
*        the intersection are returned in an SoPointDetail.
* 
*    SoGetBoundingBoxAction @BR
*        Computes the bounding box that encloses all vertices of the point set with the
*        current transformation applied to them. Sets the center to the average of the
*        coordinates of all points.
* 
*    SoCallbackAction @BR
*        If any point callbacks are registered with the action, they will be
*        invoked for each successive point in the set.
* 
* 
* @SEE_ALSO
*    SoCoordinate3,
*    SoDrawStyle,
*    SoFullSceneAntialiasing,
*    SoPointDetail,
*    SoPointSet,
*    SoVertexProperty
* 
* 
*/
class INVENTOR_API SoIndexedPointSet : public SoIndexedShape
{
  SO_NODE_HEADER(SoIndexedPointSet);
  SO_PIMPL_PUBLIC_HEADER(SoIndexedPointSet)

public:
  // Inherits all fields

  /**
  * Creates an indexed point set node with default settings.
  */
  SoIndexedPointSet();

SoEXTENDER public:

  /** Implements SoGLRenderAction actions */
  virtual void GLRender( SoGLRenderAction* action );

  /** 
  * Implements getBoundingBox actions
  * to tell open caches that they contain lines
  */
  virtual void getBoundingBox( SoGetBoundingBoxAction* action );

  /** Implements SoGetPrimitiveCountAction action */
  virtual void getPrimitiveCount( SoGetPrimitiveCountAction* action );

  // Computes bounding box
  virtual void computeBBox( SoAction* action, SbBox3f& box, SbVec3f& center );

SoINTERNAL public:

  /** register in database */
  static void initClass();

  /** unregister from database */
  static void exitClass();

  // return num points for test  
  virtual int getNumPrim( SoState* state = NULL );

protected:

  // Generates points representing point set
  virtual void generatePrimitives( SoAction* action );

  // really do the rendering part
  virtual void doRendering( SoGLRenderAction* action, const SoShapeStyleElement* shapeStyle );

  virtual void callSetupIndices( SoState* state, const SoShapeStyleElement* shapeStyle, uint32_t useTexCoordsAnyway );

  // No normal generation
  virtual void shouldGenerateNormals( SoGLRenderAction* action, const SoShapeStyleElement* shapeStyle );

  // No tangent generation
  virtual void shouldGenerateTangents( SoGLRenderAction* /*action*/, const SoShapeStyleElement* /*shapeStyle*/ ) {}

  // check if glrender should be called
  virtual SbBool shouldGLRender( SoGLRenderAction* action, SbBool isPointsOrLines = FALSE );

  // says that rendering is PointOrLine
  virtual SbBool isRenderingPointsOrLines( SoGLRenderAction* action );

  // Overrides standard method to create an SoPointDetail instance
  virtual SoDetail* createPointDetail( SoRayPickAction* action,
                                       const SoPrimitiveVertex* v,
                                       SoPickedPoint* pp);
  
  /** Destructor */
  virtual ~SoIndexedPointSet();
};

inline SbBool 
SoIndexedPointSet::isRenderingPointsOrLines( SoGLRenderAction* )
{
  return TRUE;
}

#endif /* _SO_INDEXED_POINT_SET_ */


