/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_INDEXED_SHAPE_
#define  _SO_INDEXED_SHAPE_

#include <Inventor/fields/SoMFInt32.h>
#include <Inventor/nodes/SoVertexShape.h>
#include <Inventor/SbPImpl.h>

class SbVec2d;

SO_PIMPL_PUBLIC_DECLARATION(SoIndexedShape)

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoIndexedShape
//
//  Abstract indexed shape node class. All nodes derived from this
//  (such as SoIndexedFaceSet and SoIndexedLineSet) are shapes
//  that are constructed from vertices defined by indexing into the
//  current coordinates. The coordinate indices are stored in the
//  coordIndex field, which is used by all subclasses.
//
//  One rule is used: all coordinate indices less than 0 may be used
//  for special purposes, such as to denote the end of a face or
//  polyline. This means that all indices < 0 can be ignored safely
//  when looking at vertices.
//
//  Depending on the current material, normal, tangent and texture coordinate
//  binding values, materials, normals, tangents and texture coordinates may be
//  accessed in order or may be indexed using the materialIndex,
//  normalIndex, tangentIndex and textureCoordIndex fields.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Abstract base class for all indexed vertex-based shapes.
 * 
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This node is the abstract base class for all vertex-based shapes that are
 *   constructed from indices, including SoIndexedFaceSet, SoIndexedTriangleStripSet,
 *   and SoIndexedLineSet. SoIndexedShape defines fields that are used in all of its
 *   subclasses.
 *   
 *   All subclasses of SoNonIndexedShape construct objects by using the coordinates
 *   specified by the #vertexProperty field (from SoVertexShape), or the
 *   current inherited coordinates. The #coordIndex field defined by this class
 *   contains the indices into the current coordinates of the vertices of the shape.
 *   These indices are also used for materials, normals, tangents or texture coordinates when
 *   the appropriate binding is PER_VERTEX_INDEXED.
 *   
 *   Material, normal and tangent bindings are interpreted as follows for each subclass:
 *   
 *   
 *      @TABLE_0B
 *         @TR OVERALL             @TD One material for the entire shape.
 *         @TR PER_PART            @TD Specific to the subclass.
 *         @TR PER_PART_INDEXED    @TD Same as PER_PART, using indices from the #materialIndex,
 *   #normalIndex or #tangentIndex field.
 *         @TR PER_FACE            @TD Specific to the subclass.
 *         @TR PER_FACE_INDEXED    @TD Same as PER_FACE, using indices from the #materialIndex,
 *   #normalIndex or #tangentIndex field.
 *         @TR PER_VERTEX          @TD One material per vertex.
 *         @TR PER_VERTEX_INDEXED  @TD One material per vertex, using indices from the #materialIndex,
 *   #normalIndex or #tangentIndex field.
 *      @TABLE_END 
 *   
 *   When any _INDEXED binding is used for materials, normals or tangents, the
 *   #materialIndex, #normalIndex or #tangentIndex field is used to determine the
 *   indices for the materials, normals or tangents. If this field contains a single value of
 *   -1 (the default), the coordinate indices from the #coordIndex field are
 *   used as well for materials, normals or tangents. When the binding is PER_VERTEX_INDEXED,
 *   indices in these fields that correspond to negative indices in #coordIndex
 *   are skipped; for other index bindings all the values in the fields are used, in
 *   order.
 *   
 *   Explicit texture coordinates (as defined by SoTextureCoordinate2) may be bound to
 *   vertices of an indexed shape consecutively (if the texture coordinate binding is
 *   PER_VERTEX) or by using the indices in the #textureCoordIndex field (if
 *   the binding is PER_VERTEX_INDEXED). As with all vertex-based shapes, if there is
 *   a current texture but no texture coordinates are specified, a default texture
 *   coordinate mapping is calculated using the bounding box of the shape.
 *   
 *   Be sure that the indices contained in the #coordIndex,
 *   #materialIndex, #normalIndex, #tangentIndex, and #textureCoordIndex fields
 *   are valid with respect to the current state, or errors will occur.
 *
 *   Limitations:
 *   - Due to limitations of the OpenGL VBO (vertex buffer object) rendering model, in general it is not possible
 *     to use VBO rendering (and performance may be lower) if either the normal binding, tangent binding or the material
 *     binding is set to either PER_PART(_INDEXED) or PER_FACE(_INDEXED).
 *     See individual derived classes for exceptions.
 * 
 * @FILE_FORMAT_DEFAULT
 *    This is an abstract class. See the reference page of a derived class for the
 *   format and default values.
 * 
 * @SEE_ALSO
 *    SoIndexedFaceSet,
 *    SoIndexedLineSet,
 *    SoIndexedTriangleStripSet,
 *    SoMaterialBinding,
 *    SoNonIndexedShape,
 *    SoNormalBinding,
 *    SoTangentBinding,
 *    SoShapeHints,
 *    SoTextureCoordinateBinding
 * 
 * 
 */
class INVENTOR_API SoIndexedShape : public SoVertexShape
{
  SO_NODE_ABSTRACT_HEADER(SoIndexedShape);
  SO_PIMPL_PUBLIC_HEADER(SoIndexedShape)

public:
  /**
   * The indices of the coordinates that the shape uses as its vertices. The
   * coordinates connect to form faces, lines, or other shapes. Each subclass defines
   * special negative indices to use to indicate separation between faces, lines, and
   * so on.
   */
  SoMFInt32           coordIndex;
  /**
   * The indices of the materials that are used for
   * the shape. This field is used only when the material binding is one of the
   * _INDEXED bindings. By default, the value of this field indicates that the
   * coordinate indices should be used for materials as well.
   */
  SoMFInt32           materialIndex;
  /**
   * The indices of the normals that are used for
   * the shape. This field is used only when the normal binding is one of the
   * _INDEXED bindings. By default, the value of this field indicates that the
   * coordinate indices should be used for normals as well.
   */
  SoMFInt32           normalIndex;
  /**
   * The indices of the tangents that are used for
   * the shape. This field is used only when the tangent binding is one of the
   * _INDEXED bindings. By default, the value of this field indicates that the
   * coordinate indices should be used for tangents as well.
   */
  SoMFInt32           tangentIndex;
  /**
   * The indices of the texture coordinates that are used for
   * the shape. This field is used only when the texture coordinate binding is one of the
   * _INDEXED bindings. By default, the value of this field indicates that the
   * coordinate indices should be used for texture coordinates as well.
   */
  SoMFInt32           textureCoordIndex;

SoEXTENDER public:
  /**
   * This handles AABB bounding box computation for all shapes derived
   * from this class. It sets the bounding box to contain all
   * vertices of the shape, assuming that the shape uses the
   * coordinates indexed by all non-negative values in the
   * coordIndex field. It also sets the center to the average of the
   * vertices' coordinates.
   */
  virtual void computeBBox(SoAction *action, SbBox3f &box, SbVec3f &center);

  /**
   * This handles OOBB bounding box computation for all shapes derived
   * from this class. It sets the bounding box to contain all
   * vertices of the shape, assuming that the shape uses the
   * coordinates indexed by all non-negative values in the
   * coordIndex field. It also sets the center to the average of the
   * vertices' coordinates.
   */
  virtual void computeBBox(SoAction *action, SbXfBox3d &box, SbVec3d &center);
  
  /**
   * This handles pointPicking mode.
   */
  virtual void rayPick(SoRayPickAction *action);

SoINTERNAL public:

  static void         initClass();
  static void         exitClass();

  // Keep indices up to date if things change
  virtual void notify(SoNotList *list);

protected:
  /** Constructor */
  SoIndexedShape();

  /** Destructor */
  virtual ~SoIndexedShape();

private:

  static bool s_evaluateIndexingGainFactor;

  void commonConstructor();

};

#endif /* _SO_INDEXED_SHAPE_ */
