/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : R. ALBOU (Jan 2005)
**=======================================================================*/

#ifndef  _SO_INDEXED_TEXTURE2_
#define  _SO_INDEXED_TEXTURE2_

#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFArray2D.h>

#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/threads/SbThreadStorage.h>
#include <Inventor/caches/SoCache.h>

#include <Inventor/sensors/SoFieldSensor.h>

class SoGLRenderAction ;
class SoCallbackAction ;
class SoColorMap;

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

SO_PIMPL_PUBLIC_DECLARATION(SoIndexedTexture2);

/*----------------------------------------------------------------------------*/

/**
 * @VSGEXT Class for 2D indexed texture.
 * 
 * @ingroup TextureNodes
 * 
 * @DESCRIPTION
 *   This property node defines an "indexed" texture image. In a standard texture image,
 *   e.g. SoTexture2, each texel contains a color value that is directly applied to
 *   the geometry. In this node each texel contains a @I data value@i that is mapped
 *   to a color value using the color map defined by the current SoColorMap node.
 *
 *   This image is stored in the current texture unit (see SoTextureUnit) and
 *   the texture is applied to subsequent shapes as they are rendered.
 *   
 *  The image data is stored in an SoSFArray2D.
 *  This array can contain different types of data (UNSIGNED_BYTE, UNSIGNED_SHORT,
 *  UNSIGNED_INT32, SIGNED_BYTE, SIGNED_SHORT, SIGNED_INT32, FLOAT). 
 * 
 *  When Open Inventor creates the OpenGL texture image from the data
 *  array, the data values must be scaled into the range of values of the
 *  texture (0-255 for an 8-bit indexed texture). The #minValue and #maxValue fields
 *  specify the range of data values that will be scaled into the range
 *  of texture values. 
 *  For example, if 
 *  #minValue is set to 10000 and #maxValue to 38000, all values 
 *  less than or equal to 10000 will be mapped to 
 *  the entry 0 of the color map and all values greater than or equal 
 *  to 38000 to the last entry.
 *  The image below illustrates the process of mapping and shows 
 *  how it can be used to map only the used data range to the color map:
 *  @IMAGE dataremap.jpg     
 *
 *  If #minValue or #maxValue change, then the OpenGL texture
 *  must be recreated. If the texture is large, this may be time consuming.
 *  Generally it's more efficient to modify the SoColorMap node because
 *  the OpenGL texture corresponding to the data does not need to be recreated --
 *  only the texture containing the colormap, which is normally quite small.
 *
 *  @B NOTE: @b Texture data values are stored with 12 bits of precision instead of the usual 8 bits, when:
 *  @UL
 *  @LI @I ARB_fragment_program @i is supported by the graphics card.
 *  @LI The number of color map entries is greater than 256.
 *  @LI The texture data type has more than 8 bits of precision (i.e., not UNSIGNED_BYTE or SIGNED_BYTE).
 *  @ul
 *
 * @EXAMPLE
 *  The following example shows how to apply an indexed texture on a geometry node:
 *  \if_cpp
 *    \code
 *      SoColorMap *colorMap = new SoColorMap();
 *        colorMap->predefinedColorMap = SoColorMap::TEMPERATURE;
 *        colorMap->min.setValue( -1000 );
 *        colorMap->max.setValue( 20000 );
 *
 *      SoIndexedTexture2 *indexedTexture = new SoIndexedTexture2();
 *        indexedTexture->imageIndex.setValue(size, SoSFArray2D::SIGNED_SHORT, data);
 *
 *      root->addChild( colorMap );
 *      root->addChild( indexedTexture );
 *      root->addChild( geometry );
 *    \endcode
 *  \endif
 *  \if_dotnet
 *    \code
 *      SoColorMap colorMap = new SoColorMap();
 *        colorMap.predefinedColorMap.SetValue( (int)SoColorMap.PredefinedColorMaps.TEMPERATURE );
 *        colorMap.min.Value = -1000;
 *        colorMap.max.Value = 20000;
 *
 *      SoIndexedTexture2 indexedTexture = new SoIndexedTexture2();
 *        indexedTexture.imageIndex.SetValue(size, (int)SoSFArray.DataTypes.SIGNED_SHORT, data);
 *
 *      root.AddChild( colorMap );
 *      root.AddChild( indexedTexture );
 *      root.AddChild( geometry ); 
 *    \endcode
 *  \endif
 *  \if_java
 *    \code
 *      SoColorMap colorMap = new SoColorMap();
 *        colorMap.predefinedColorMap.setValue( SoColorMap.PredefinedColorMaps.TEMPERATURE );
 *        colorMap.min.setValue( -1000 );
 *        colorMap.max.setValue( 20000 );
 *
 *      SoIndexedTexture2 indexedTexture = new SoIndexedTexture2();
 *        indexedTexture.imageIndex.setValue(size, SoSFArray.DataTypes.SIGNED_SHORT, data);
 *
 *      root.addChild( colorMap );
 *      root.addChild( indexedTexture );
 *      root.addChild( geometry ); 
 *    \endcode
 *  \endif
 *
 * @EXAMPLE
 *  On graphics cards supporting @I ARB_fragment_program @i, the #model field will 
 *  be ignored and multitexturing 
 *  will not work. It is possible to circumvent this limitation by using a custom shader. The following
 *  GLSL code shows how to do multitexturing between two indexed textures sharing the same color map:
 *
 *  \par
 *  \code
 *  // Fragment shader
 *  uniform sampler2D colormap;
 *  uniform sampler2D indexedTex1; 
 *  uniform sampler2D indexedTex2;  
 *
 *  void main()
 *  {
 *    // Fetch first value and its assigned color
 *    float value1 = texture(indexedTex1, gl_TexCoord[0].xy).x;
 *    vec4  color1 = texture(colormap, vec2(value1, 0));
 *
 *    // Fetch second and its assigned color
 *    float value2 = texture(indexedTex2, gl_TexCoord[0].xy).x;
 *    vec4  color2 = texture(colormap, vec2(value2, 0));
 *
 *    // Combine the colors
 *    gl_FragColor = color1 * color2 * gl_Color;
 *  }
 *  \endcode
 *
 * The scene graph would be initialized like this:
 * \par
 * \if_cpp
 * \code
 *   SoColorMap* colorMap = new SoColorMap();
 *     . . .
 *   SoIndexedTexture2* indexedTex1 = new SoIndexedTexture2();
 *     . . .
 *   SoIndexedTexture2* indexedTex2 = new SoIndexedTexture2();
 *     . . .
 *
 *   SoFragmentShader* fragShader = new SoFragmentShader();
 *     fragShader->sourceProgram = "filename.glsl";
 *     fragShader->addShaderParameter1i( "colorMap"   , 0 );
 *     fragShader->addShaderParameter1i( "indexedTex1", 1 );
 *     fragShader->addShaderParameter1i( "indexedTex2", 2 );
 *
 *   SoShaderProgram*  shaderProgram = new SoShaderProgram();
 *     shaderProgram->shaderObject = fragShader;
 *
 *   SoTextureUnit* textureUnit = new SoTextureUnit();
 *   textureUnit->unit = 2;  //Texture unit for the second indexed texture
 *
 *   root->addChild(colorMap);      //Color map must come first (texture unit 1)
 *   root->addChild(indexedTex1);   //First indexed texture (texture unit 0)
 *   root->addChild(textureUnit); 
 *   root->addChild(indexedTex2);   //Second indexed texture (texture unit 2)
 *   root->addChild(shaderProgram);
 *   root->addChild(geometry);
 * \endcode
 * \endif
 * \if_dotnet
 * \code
 *   SoColorMap colorMap = new SoColorMap();
 *   . . .
 *   SoIndexedTexture2 indexedTex1 = new SoIndexedTexture2();
 *   . . .
 *   SoIndexedTexture2 indexedTex2 = new SoIndexedTexture2();
 *   . . .
 *
 *   SoFragmentShader fragShader = new SoFragmentShader();
 *     fragShader.sourceProgram.Value = "filename.glsl";
 *
 *   SoShaderParameter1i parameter = new SoShaderParameter1i();
 *     parameter.name.Value = "colorMap";
 *     parameter.value.Value = 0;
 *     fragShader.parameter[0] = parameter;
 *
 *   parameter = new SoShaderParameter1i();
 *     parameter.name.Value = "indexedTex1";
 *     parameter.value.Value = 1;
 *     fragShader.parameter[1] = parameter;
 *
 *   parameter = new SoShaderParameter1i();
 *     parameter.name.Value = "indexedTex2";
 *     parameter.value.Value = 2;
 *     fragShader.parameter[2] = parameter;
 *
 *   SoShaderProgram shaderProgram = new SoShaderProgram();
 *     shaderProgram.shaderObject.SetValue(fragShader);
 *
 *   SoTextureUnit textureUnit = new SoTextureUnit();
 *     textureUnit.unit.Value = 2;  //Texture unit for the second indexed texture
 *
 *   root.AddChild(colorMap);      //Color map must come first (texture unit 1)
 *   root.AddChild(indexedTex1);   //First indexed texture (texture unit 0)
 *   root.AddChild(textureUnit);
 *   root.AddChild(indexedTex2);   //Second indexed texture (texture unit 2)
 *   root.AddChild(shaderProgram);
 *   root.AddChild(geometry);
 * \endcode
 * \endif
 * \if_java
 * \code
 *   SoColorMap colorMap = new SoColorMap();
 *   . . .
 *   SoIndexedTexture2 indexedTex1 = new SoIndexedTexture2();
 *   . . .
 *   SoIndexedTexture2 indexedTex2 = new SoIndexedTexture2();
 *   . . .
 *
 *   SoFragmentShader fragShader = new SoFragmentShader();
 *     fragShader.sourceProgram.setValue( "filename.glsl" );
 *
 *   SoShaderParameter1i parameter = new SoShaderParameter1i();
 *     parameter.name.setValue( "colorMap" );
 *     parameter.value.setValue( 0 );
 *     fragShader.parameter.set1Value( 0, parameter );
 *
 *   parameter = new SoShaderParameter1i();
 *     parameter.name.setValue( "indexedTex1" );
 *     parameter.value.setValue( 1 );
 *     fragShader.parameter.set1Value( 1, parameter );
 *
 *   parameter = new SoShaderParameter1i();
 *     parameter.name.setValue( "indexedTex2" );
 *     parameter.value.setValue( 2 );
 *     fragShader.parameter.set1Value( 2, parameter );
 *
 *   SoShaderProgram shaderProgram = new SoShaderProgram();
 *     shaderProgram.shaderObject.setValue(fragShader);
 *
 *   SoTextureUnit textureUnit = new SoTextureUnit();
 *     textureUnit.unit.setValue( 2 );  //Texture unit for the second indexed texture
 *
 *   root.addChild(colorMap);      //Color map must come first (texture unit 1)
 *   root.addChild(indexedTex1);   //First indexed texture (texture unit 0)
 *   root.addChild(textureUnit);
 *   root.addChild(indexedTex2);   //Second indexed texture (texture unit 2)
 *   root.addChild(shaderProgram);
 *   root.addChild(geometry);
 * \endcode
 * \endif
 *  
 * @FILE_FORMAT_DEFAULT
 *    IndexedTexture2 {
 *    @TABLE_FILE_FORMAT
 *       @TR minValue                 @TD 0
 *       @TR maxValue                 @TD 0
 *       @TR imageIndex               @TD 0 0
 *       @TR rescaleTexCoord          @TD FALSE
 *       @TR wrapS                    @TD REPEAT
 *       @TR wrapT                    @TD REPEAT
 *       @TR model                    @TD MODULATE
 *       @TR blendColor               @TD 0 0 0
 *       @TR enableBorder             @TD FALSE
 *       @TR borderColor              @TD 0 0 0 0
 *       @TR maxAnisotropy            @TD 1.0
 *       @TR minFilter                @TD AUTO
 *       @TR magFilter                @TD AUTO 
 *       @TR useAutoMipmap            @TD FALSE
 *       @TR internalFormat           @TD AUTO_INTERNAL_FORMAT
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction, SoCallbackAction @BR
 *    Sets: SoTextureImageElement
 * 
 * @SEE_ALSO
 *    SoTexture2Transform, SoTextureCoordinate2,  
 *    SoTextureCoordinateFunction, SoColorMap, SoSFArray2D.
 *
 * 
 */
class INVENTOR_API SoIndexedTexture2 : public SoTexture {  
  SO_NODE_HEADER(SoIndexedTexture2) ;

  SO_PIMPL_PUBLIC_HEADER(SoIndexedTexture2);

 public:
  /**
  * Specifies the range of values which is mapped onto the color map (see SoColorMap). 
  * When #minValue and #maxValue are equal to 0 (the default), the entire range of the data type is 
  * mapped onto the color map, except in the case of float data. For example, for a color map
  * of size N:
  * @UL
  * @LI With unsigned byte values, [0-255] is mapped onto the color map [0 - N-1]
  * @LI With unsigned short values, [0-65535] is mapped onto the color map [0 - N-1]
  * @LI With signed short values, [-32768 - 32767] is mapped onto the color map [0 - N-1].
  * @LI With float data type, [0-1] is mapped onto the color map [0 - N-1]
  * @ul
  * All values less than or equal to #minValue will be mapped to the first entry of the color map.
  * Likewise, all values greater than or equal to #maxValue will be mapped to the last entry of 
  * the color map.
  */
  SoSFFloat   minValue ;

  /**
   * See #minValue
   */
  SoSFFloat   maxValue ;

  /**
   * This field contains the in-memory representation of the indexed texture image.
   * Each value of this array is a color index. In the case of float values, data must 
   * be between 0 and 1.
   */
  SoSFArray2D imageIndex ;

  /**
  * This field controls the way an image with non-power-of-two dimension is handled:
  * - If the graphics card supports @I GL_ARB_texture_non_power_of_two @i, 
  *   this field is ignored and the image is sent directly to OpenGL.
  * - If #rescaleTexCoord is FALSE (the default), the image is rescaled to the next lower
  *   power-of-two dimensions. The scaling is done with a box filter.
  * - If #rescaleTexCoord is TRUE, the image is not rescaled but is only copied into a
  *   texture with the next higher power-of-two dimension, and its texture coordinates 
  *   are scaled by applying a texture transform in order to only display the actual image.
  *
  * Default is FALSE.
  * 
  * When this field is enabled, if other texture transforms are applied to the texture coordinates 
  * (especially if the texture is repeated), the unused part of the larger texture 
  * could be displayed. 
  * Furthermore, in this mode, if non-power-of-two textures occur one after another 
  * in the scene graph, an incorrect image may result because texture transforms are
  * cumulative. For example, the following code would give incorrect results: 
  *
  * \if_cpp
  *   \par
  *   \code
  *   root->addChild( colorMap );
  *   root->addChild( nonPowerOfTwoIndexedTexture1 );
  *   root->addChild( nonPowerOfTwoIndexedTexture2 );
  *   root->addChild( nonPowerOfTwoIndexedTexture3 );
  *   \endcode
  * \endif
  * \if_dotnet
  *   \par
  *   \code
  *   root.AddChild( colorMap );
  *   root.AddChild( nonPowerOfTwoIndexedTexture1 );
  *   root.AddChild( nonPowerOfTwoIndexedTexture2 );
  *   root.AddChild( nonPowerOfTwoIndexedTexture3 );
  *   \endcode
  * \endif
  * \if_java
  *   \par
  *   \code
  *   root.addChild( colorMap );
  *   root.addChild( nonPowerOfTwoIndexedTexture1 );
  *   root.addChild( nonPowerOfTwoIndexedTexture2 );
  *   root.addChild( nonPowerOfTwoIndexedTexture3 );
  *   \endcode
  * \endif
  *
  * The correct way is to use a separator above each texture, as shown:
  *
  * \if_cpp
  *   \par
  *   \code
  *   root->addChild( colorMap );
  *   root->addChild( separator1 );
  *     separator1->addChild( nonPowerOfTwoIndexedTexture1 );
  *   root->addChild( separator2);
  *     separator2->addChild( nonPowerOfTwoIndexedTexture2 );
  *   root->addChild( separator3);
  *     separator3->addChild( nonPowerOfTwoIndexedTexture3 );
  *   \endcode
  * \endif
  * \if_dotnet
  *   \par
  *   \code
  *   root.AddChild( colorMap );
  *   root.AddChild( separator1 );
  *     separator1.AddChild( nonPowerOfTwoIndexedTexture1 );
  *   root.AddChild( separator2 );
  *     separator2.AddChild( nonPowerOfTwoIndexedTexture2 );
  *   root.AddChild( separator3 );
  *     separator3.AddChild( nonPowerOfTwoIndexedTexture3 );
  *   \endcode
  * \endif
  * \if_java
  *   \par
  *   \code
  *   root.addChild( colorMap );
  *   root.addChild( separator1 );
  *     separator1.addChild( nonPowerOfTwoIndexedTexture1 );
  *   root.addChild( separator2 );
  *     separator2.addChild( nonPowerOfTwoIndexedTexture2 );
  *   root.addChild( separator3 );
  *     separator3.addChild( nonPowerOfTwoIndexedTexture3 );
  *   \endcode
  * \endif
  * 
  */
  SoSFBool  rescaleTexCoord;

  /**
   * Indicates what to do when texture coordinates in the T
   * (vertical) direction lie outside the range 0-1.
   * @useenum{SoTexture::Wrap}. Default is REPEAT.
   */
  SoSFEnum   wrapT ;

  /**
   * Compute the min and max value of the indexed image and put the result in #minValue and #maxValue.
   *
   */
  void computeDataRange();

  /**
   * Constructor.
   */
  SoIndexedTexture2() ;

 SoEXTENDER public:
  virtual void doAction( SoAction* action );
  virtual void GLRender( SoGLRenderAction* action );

  /** 
   * This method returns the alternate represention (currently only for PDF and U3D exports).
   * The node returned is an SoTexture2.
   */
  virtual SoNode* getAlternateRep( SoAction* action );

  /*----------------------------------------------------------------------------*/
  
 SoINTERNAL public:
  static void initClass() ;
  static void exitClass() ;

  /**
   * Create a subgraph for the alternateRep field that gives approximately
   * the same appearance, but using only core Open Inventor nodes.
   * Might be useful in other cases.
   */
  virtual SoNode* createAlternateRep( SoAction *action );

  // Manage field dependencies  when one of them change   AJO eBug #1972
  virtual void fieldHasChanged( SoField *field ); // override of SoFieldContainer

 protected:
  // Destructor
  virtual ~SoIndexedTexture2() ;

} ;
/*----------------------------------------------------------------------------*/

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif /* _SO_INDEXED_TEXTURE2_ */


