#pragma once

#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoIndexedShape.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoIndexedTriangleFanSet
//
//  Indexed set of triangles fans. Fans are separated by the
//  special index SO_END_FAN_INDEX (-1). The N indices in the fan
//  define N-2 triangles, which are formed by indexing into the
//  current coordinates. Depending on the current material and normal
//  binding values, the materials and normals for the triangles or
//  vertices may be accessed in order or indexed. If they are indexed,
//  the materialIndex and normalIndex fields are used.
//
//////////////////////////////////////////////////////////////////////////////

#define SO_END_FAN_INDEX (-1)

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoIndexedTriangleFanSet);

/**
 * Indexed triangle fan set shape node.
 *
 * @ingroup ShapeNodes
 *
 * @DESCRIPTION
 *   This shape node constructs triangle fans out of vertices located at the
 *   coordinates specified in the #vertexProperty field (from SoVertexShape),
 *   or the current inherited coordinates. For optimal performance, the
 *   #vertexProperty field is recommended.
 *
 *   SoIndexedTriangleFanSet uses the indices in the #coordIndex field (from
 *   SoIndexedShape) to specify the vertices of the triangle fans. An index of
 *   SO_END_FAN_INDEX (-1) indicates that the current fan has ended and the next
 *   one begins.
 *
 *   The vertices of the faces are transformed by the current transformation matrix.
 *   The faces are drawn with the current light model and drawing style.
 *
 *   Treatment of the current material and normal binding is as follows: PER_PART
 *   specifies a material or normal per fan. PER_FACE binding specifies a material
 *   or normal for each triangle. PER_VERTEX specifies a material or normal for each
 *   vertex. The corresponding _INDEXED bindings are the same, but use the
 *   #materialIndex or #normalIndex indices (see SoIndexedShape). The
 *   default material binding is OVERALL. The default normal binding is
 *   PER_VERTEX_INDEXED
 *
 *   If any normals (or materials) are specified, Open Inventor assumes you provide
 *   the correct number of them, as indicated by the binding. You will see unexpected
 *   results if you specify fewer normals (or materials) than the shape requires. If
 *   no normals are specified, they will be generated automatically.
 *
 *   To render triangles that are not in fans, see the SoIndexedTriangleSet and
 *   SoIndexedTriangleStripSet nodes.
 *
 *   Limitations:
 *   - Due to limitations of the OpenGL VBO (vertex buffer object) rendering model, it is not possible
 *     to use VBO rendering (and performance may be lower) if either the normal binding or the material
 *     binding is set to either PER_PART(_INDEXED) or PER_FACE(_INDEXED).
 *   - For texture coordinates and normal binding, only OVERALL and PER_VERTEX_INDEXED
 *     mode are supported yet.
 *
 * @FILE_FORMAT_DEFAULT
 *    IndexedTriangleFanSet {
 *    @TABLE_FILE_FORMAT
 *       @TR vertexProperty      @TD NULL
 *       @TR coordIndex          @TD -1
 *       @TR materialIndex       @TD -1
 *       @TR normalIndex         @TD -1
 *       @TR textureCoordIndex   @TD -1
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Draws a fan set based on the current coordinates, normals, materials, drawing
 *        style, and so on.
 *
 *    SoRayPickAction @BR
 *        Picks on the fan set based on the current coordinates and transformation.
 *        Details about the intersection are returned in an SoFaceDetail.
 *
 *    SoGetBoundingBoxAction @BR
 *        Computes the bounding box that encloses all vertices of the fan set with the
 *        current transformation applied to them. Sets the center to the average of the
 *        coordinates of all vertices.
 *
 *    SoCallbackAction @BR
 *        If any triangle callbacks are registered with the action, they will be invoked
 *        for each successive triangle forming the fan of the set.
 *
 *
 * @SEE_ALSO
 *    SoCoordinate3,
 *    SoDrawStyle,
 *    SoFaceDetail,
 *    SoFullSceneAntialiasing,
 *    SoIndexedFaceSet,
 *    SoIndexedTriangleSet,
 *    SoTriangleStripSet,
 *    SoVertexProperty
 *
 *
 */

class INVENTOR_API SoIndexedTriangleFanSet : public SoIndexedShape
{
  SO_NODE_HEADER(SoIndexedTriangleFanSet);
  SO_PIMPL_BASE_PUBLIC_HEADER(SoIndexedTriangleFanSet);

public:
  /**
   * Constructs a default IndexedTriangleFanSet 
   */
  SoIndexedTriangleFanSet();
  
  /**
   * Returns the number of triangles
   */
  int getNumTriangles();

  /**
   * Returns the number of fans (parts)
   */
  int getNumFans();

SoEXTENDER public:
  virtual void GLRender(SoGLRenderAction* action);
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction* action);
  virtual int getNumNeededNormals(SoState* state);
  virtual void generateDefaultNormals(SoState* state,
    const SbVec3f* coords,
    int numCoords,
    SoNormalBundle* nb,
    SbBool storeRef = FALSE);

  /** Disable tangents generation for triangle shapes */
  virtual void shouldGenerateTangents( SoGLRenderAction* /*action*/, const SoShapeStyleElement* /*shapeStyle*/ ) {}

SoINTERNAL public:
  static void initClass();
  static void exitClass();
  virtual void notify(SoNotList* list);

protected:
  virtual ~SoIndexedTriangleFanSet();
  virtual void generatePrimitives(SoAction *action);  
};