/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef SO_INTERACTIVE_COMPLEXITY_H
#define SO_INTERACTIVE_COMPLEXITY_H

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoMFString.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>

#include <Inventor/STL/map>
#include <Inventor/sensors/SoAlarmSensor.h>

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251) // disable warning 4251 "needs to have dll-interface to be used by clients of class..." caused by ScalarBoundsMap
#endif

/**
 * Field interactive complexity node.
 *
 * @ingroup PropertyNodes
 *
 * @DESCRIPTION
 * This node allows the application to define different parameter values
 * for certain fields of specific node classes, depending on whether a user interaction,
 * for example moving the camera, is occurring.  This means that while the
 * camera is moving these fields will use a specified "interaction" parameter
 * value, but when interactive manipulation is stopped these fields will
 * automatically change to a specified "still" parameter value.  Optionally,
 * for scalar fields, the transition from interaction value to still value
 * can be automatically animated using a specified increment.
 *
 * Note: The values specified in SoInteractiveComplexity override the values in
 * the fields of the specified nodes during rendering. But calling getValue() on 
 * the fields still returns the value set directly into the field (or the default
 * value if none was set).
 *
 * This is a powerful technique for maintaining an interactive framerate
 * when interacting with GPU intensive datasets or rendering effects, while
 * still getting a final image with very high quality and giving the user
 * a "progressive refinement" effect while transitioning from interaction
 * back to "still".
 *
 * These settings are applied to all instances of the node class containing the
 * specified field and are declared with a specially formatted string set in the
 * #fieldSettings field.  For scalar fields like SoSFInt, the string looks like this: @BR
 * "ClassName FieldName InteractionValue StillValue [IncrementPerSecond]"
 *
 *   If @I IncrementPerSecond @i is omitted, @I StillValue @i is applied as soon as
 *   interaction stops.  Else the transition from
 *   @I InteractionValue @i to @I StillValue @i is automatically animated.
 *   Because incrementing is actually done at each redraw, and redraw happens many times
 *   per second, IncrementPerSecond is allowed to be greater than StillValue.
 *
 *   In the following code, the field named @I value @i belonging to the class
 *   SoComplexity will be set to 0.1 during an interaction. When the interaction
 *   stops, @I value @i will be increased by 2 every second until it reaches 0.5.
 *   Effectively this means that the StillValue (0.5) will be reached 
 *   in (0.5 - 0.1) / 2 = 0.2 seconds.
 *   \if_cpp
 *     \code
 *     SoInteractiveComplexity* icplx = new SoInteractiveComplexity;
 *     icplx->fieldSettings.set1Value( 0, "SoComplexity value 0.1 0.5 2" );
 *     root->addChild(icplx);
 *     \endcode
 *   \endif
 *   \if_dotnet
 *     \code
 *     SoInteractiveComplexity icplx = new SoInteractiveComplexity();
 *     icplx.fieldSettings[0] = "SoComplexity value 0.1 0.5 2";
 *     root.AddChild(icplx);
 *     \endcode
 *   \endif
 *   \if_java
 *     \code
 *     SoInteractiveComplexity icplx = new SoInteractiveComplexity();
 *     icplx.fieldSettings.set1Value( 0, "SoComplexity value 0.1 0.5 2" );
 *     root.addChild(icplx);
 *     \endcode
 *   \endif
 *
 *  A time delay before changing the value, or starting the animation, can be
 *  set using the #refinementDelay field.
 *
 *  The field #interactiveMode allows to force the use of the InteractionValue or the StillValue.
 *
 *  Only a limited number of fields are supported by this node:
 *  - SoComplexity
 *    - textureQuality
 *    - value
 *  - SoOrthoSlice
 *    - largeSliceSupport
 *  - SoShadowGroup
 *    - isActive
 *    - quality
 *  - SoVolumeGroup
 *    - multiVolumes
 *  - SoVolumeRender
 *    - interpolation
 *    - lowScreenResolutionScale
 *    - numSlices
 *    - projectedTileSubdivision
 *    - samplingAlignment
 *  - SoVolumeRenderingQuality
 *    - boundaryOpacity
 *    - colorInterpolation
 *    - edgeColoring
 *    - edgeDetect2D
 *    - gradientQuality
 *    - lighting
 *    - segmentedInterpolation
 *    - deferredLighting
 *    - ambientOcclusion
 *    - cubicInterpolation
 *  - SoVolumeSkin
 *    - largeSliceSupport
 *
 * @FILE_FORMAT_DEFAULT
 *    InteractiveComplexity {
 *    @TABLE_FILE_FORMAT
 *       @TR fieldSettings             @TD []
 *       @TR refinementDelay           @TD 0.5
 *       @TR interactiveMode           @TD AUTO
 *    @TABLE_END
 *    }
 *
 * @SEE_ALSO
 *    SoComplexity
 */
class INVENTOR_API SoInteractiveComplexity : public SoNode
{
  SO_NODE_HEADER(SoInteractiveComplexity);

public:
  SoInteractiveComplexity();

  /** Interactive mode */
  enum InteractiveMode {
    /** Automatically detect when a user interaction is occurring (default) */
    AUTO,
    /** Behave like a user interaction is occurring (always use InteractionValue) */
    FORCE_INTERACTION,
    /** Behave like there isn't any interaction (always use StillValue) */
    FORCE_STILL,
    /** Behave like there isn't any SoInteractiveComplexity node */
    DISABLED
  };

  /**
   * Each string in this field specifies the interactionValue and stillValue
   * for a specific field of a specific node class.  The format is:
   * <table>
   *  <tr>
   *   <td>For scalar fields 
   *       (SoSFInt32, SoSFFloat...)</td>
   *   <td>ClassName FieldName InteractionValue StillValue [IncrementPerSecond]</td>
   *  </tr>
   *  <tr>
   *   <td>For SoSFBool fields 
   *       (NumberOfSeconds = delay before switching state)</td>
   *   <td>ClassName FieldName InteractionValue [StillValue] [NumberOfSeconds]</td>
   *  </tr>
   *  <tr>
   *   <td>For SoSFEnum fields</td>
   *   <td>ClassName FieldName InteractionValue StillValue</td>
   *  </tr>
   * </table>
   *
   * The following example defines values for a SoSFFloat, a SoSFBool and a SoSFEnum:
   * \if_cpp
   *   \code
   *     SoInteractiveComplexity* icplx = new SoInteractiveComplexity();
   *     icplx->fieldSettings.set1Value(0, "SoComplexity value 0.1 0.5 2" );
   *     // isActive will switch to TRUE 2 seconds after stopping interaction
   *     icplx->fieldSettings.set1Value(1, "SoShadowGroup isActive FALSE 2" );
   *     icplx->fieldSettings.set1Value(2, "SoVolumeRenderingQuality gradientQuality LOW HIGH");
   *   \endcode
   * \endif
   * \if_dotnet
   *   \code
   *     SoInteractiveComplexity icplx = new SoInteractiveComplexity();
   *     icplx.fieldSettings[0] = "SoComplexity value 0.1 0.5 2";
   *     // isActive will switch to TRUE 2 seconds after stopping interaction
   *     icplx.fieldSettings[1] = "SoShadowGroup isActive FALSE 2";
   *     icplx.fieldSettings[2] = "SoVolumeRenderingQuality gradientQuality LOW HIGH";
   *   \endcode
   * \endif
   * \if_java
   *   \code
   *     SoInteractiveComplexity icplx = new SoInteractiveComplexity();
   *     icplx.fieldSettings.set1Value( 0, "SoComplexity value 0.1 0.5 2" );
   *     // isActive will switch to TRUE 2 seconds after stopping interaction
   *     icplx.fieldSettings.set1Value( 1, "SoShadowGroup isActive FALSE 2" );
   *     icplx.fieldSettings.set1Value( 2, "SoVolumeRenderingQuality gradientQuality LOW HIGH" );
   *   \endcode
   * \endif
   */
  SoMFString fieldSettings;

  /**
   * Time in seconds after interaction stops before field values start to be increased. Default is 0.0.
   */
  SoSFFloat refinementDelay;

  /**
  * Interactive mode.
  * @useenum{InteractiveMode}. Default is AUTO.
  *
  * @FIELD_SINCE_OIV 8.6
  */
  SoSFEnum interactiveMode;

SoEXTENDER public:
  virtual void doAction(SoAction *action);
  virtual void GLRender(SoGLRenderAction *action);

SoINTERNAL public:
  /**
   * Handle field change
   */
  virtual void notify(SoNotList *list);

  static void initClass();
  static void exitClass();

  /** Return the current value of the field*/
  double getScalarFieldValue(SoState* state, const SoSField* field);
  SbBool getBooleanFieldValue(SoState* state, const SoSFBool* field);
  int getEnumFieldValue(SoState* state, const SoSFEnum* field);

  /** True if the field has values for interactive mode */
  bool isBounded(const SoField* field) const;

protected:
  struct ScalarBounds
  {
    double startBound;
    double endBound;
    double increment;
    // used to return the previous val during delayed pass
    double currentVal;
    bool immediate;
  };
  typedef std::map<SbString, ScalarBounds> ScalarBoundsMap;

  virtual ~SoInteractiveComplexity();

  /** True if field is a scalar field */
  static SbBool isScalarField(const SoField* field);

  /** recreate the parameter map */
  void updateFieldList();
  void addParameters(const SbString& paramString);

  /** Handle paramString for scalar field and add parameters to the parameter map */
  void handleScalarParam(const SbString& paramString, const std::vector<SbString>& token);
  void handleBoolParam(const SbString& paramString, const std::vector<SbString>& token);
  void handleEnumParam(const SbString& paramString, const std::vector<SbString>& token,
                       const SoField* field);

  /** Display en error if numParam != expected */
  bool checkParamNumber(size_t numParam, size_t expected, const SbString& fieldName) const;

  /** Get the name of the class containing field and the name of the field */
  void getClassAndFieldName(const SoField* field, SbString& className, SbString& fieldName) const;

  /** Get the name of the class containing field and the name of the field */
  double getScalarFieldValue(SoState* state, const SoSField* field,
                             const SbString& classAndFieldName, bool updateValue = false);

  /** Compute current value of all field types */
  void updateAllFieldsValue(SoState* state);

  /** Called by the redrawSensor. Just shedule a redraw for this node. 
   * data must be this node. */
  static void sensorCB(void *data, SoSensor *sensor);

  /** All bounds are converted to a scalar bound */
  ScalarBoundsMap m_scalarBounds;

  /** Sensor use to schedule a display at the end of refinementDelay */
  SoAlarmSensor m_redrawSensor;
};

#if defined(_WIN32)
#pragma warning( pop )
#endif

#endif //SO_INTERACTIVE_COMPLEXITY_H
