/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_MATERIAL_BINDING_
#define  _SO_MATERIAL_BINDING_

#include <Inventor/SbVertexAttributeBinding.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/elements/SoMaterialBindingElement.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoMaterialBinding
//
//  Material binding node. This indicates how multiple materials are
//  to be distributed over an object.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Node that specifies how multiple materials are bound to shapes.
 * 
 * @ingroup PropertyNodes
 * 
 * @DESCRIPTION
 *   This node specifies how the current materials are bound to shapes that follow in
 *   the scene graph. Each shape node may interpret bindings differently. The current
 *   material always has a base value, which is defined by the first value of all
 *   material fields. Since material fields may have multiple values, the binding
 *   determines how these values are distributed over a shape.
 *   
 *   The bindings for faces and vertices are meaningful only for shapes that are made
 *   from faces and vertices. Similarly, the indexed bindings are only used by the
 *   shapes that allow indexing.
 *   
 *   The bindings apply only to diffuse colors and transparency. Other materials
 *   (emissive, specular, ambient, shininess) will have the first value applied to
 *   the entire shape, regardless of the material binding or the number provided.
 *   
 *   If the number of transparencies is less than the number of diffuse colors, only
 *   the first transparency value will be used, regardless of the material binding.
 *   If the number of diffuse colors in the state is less than the number required
 *   for the given binding, a debug warning will be printed and unpredictable colors
 *   will result.
 * 
 * @FILE_FORMAT_DEFAULT
 *    MaterialBinding {
 *    @TABLE_FILE_FORMAT
 *       @TR value   @TD OVERALL
 *       @TR face    @TD FRONT_AND_BACK
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction @BR
 *        Sets the current material binding type.
 *        Sets: SoMaterialBindingElement
 * 
 * @SEE_ALSO
 *    SoMaterial,
 *    SoNormalBinding,
 *    SoShape,
 *    SoShapeHints,
 *    SoVertexProperty,
 *    SoTextureCoordinateBinding
 * 
 * 
 */
class INVENTOR_API SoMaterialBinding : public SoNode {

  SO_NODE_HEADER(SoMaterialBinding);

 public:
  enum Binding {
    /**
     *  Whole object has same material 
     */
    OVERALL           = SbVertexAttributeBinding::OVERALL,
    /**
     *  One material for each part of object 
     */
    PER_PART          = SbVertexAttributeBinding::PER_PART,
    /**
     *  One material for each part, indexed 
     */
    PER_PART_INDEXED  = SbVertexAttributeBinding::PER_PART_INDEXED,
    /**
     *  One material for each face of object 
     */
    PER_FACE          = SbVertexAttributeBinding::PER_FACE,
    /**
     *  One material for each face, indexed 
     */
    PER_FACE_INDEXED  = SbVertexAttributeBinding::PER_FACE_INDEXED,
    /**
     *  One material for each vertex of object 
     */
    PER_VERTEX        = SbVertexAttributeBinding::PER_VERTEX,
    /**
     *  One material for each vertex, indexed 
     */
    PER_VERTEX_INDEXED= SbVertexAttributeBinding::PER_VERTEX_INDEXED

    // Obsolete bindings:
    ,  // Comma to continue list
    DEFAULT = OVERALL,
    NONE = OVERALL
  };

  // Fields
  /**
   * Specifies how to bind materials to shapes.
   * @useenum{Binding}. Default is OVERALL.
   */
  SoSFEnum value;

  /**
   * Specifies values to apply material on the front face, back face or both front and back face of an object.
   */
  enum Face
  {
    /**
      * Applies the material to the front faces of the object.
      * The front face is determined by the SoShapeHints::vertexOrdering field.
     */
    FRONT = SoMaterialBindingElement::FRONT,

    /**
     * Applies the material to the back faces of the object.
     * The back face is determined by the SoShapeHints::vertexOrdering field.
     */
    BACK = SoMaterialBindingElement::BACK,

    /**
     * Applies the material to both the front and back faces of the object.
     */
    FRONT_AND_BACK = SoMaterialBindingElement::FRONT_AND_BACK,
  };

  /**
   * Applies materials to different faces of an object.
   * This enum specifies whether to apply the material to the front faces, the back faces, or both.
   * It is particularly useful for applying different materials to each side of a non-closed surface,
   * allowing for distinct appearances on either side.
   *
   * For closed surfaces, the use of FRONT_AND_BACK is generally sufficient, except where a SoClipPlane is used.
   * This ensures that the material is applied to both the front and back faces, providing a consistent appearance.
   *
   * Notes:
   * - If you specify a material for one side only, the other side uses the default material values (see SoMaterial).
   *
   * - If you enable face culling, only the front faces are rendered (see SoShapeHints), even if a material
   *   is specified for the back faces. The same applies to front face culling.
   *
   * - If you disable two-sided lighting (see SoShapeHints), the front material is applied to both front and back faces.
   *
   * - If you specify colors using SoVertexProperty::orderedRGBA or SoBufferedShape::colorBuffer,
   *   you can only specify one set of colors. The current value of #face on state determines which
   *   faces the color buffer impacts.
   *
   * - To visualize only the front or back faces, use the SoShapeHints::shapeType field.
   *   The enum SOLID activates back face culling and INNER_SOLID activates front face culling.
   *
   * @useenum{Face}. Default is FRONT_AND_BACK.
   *
   * @FIELD_SINCE_OIV 2025.1
   */
  SoSFEnum face;

  /**
   * Creates a material binding node with default settings.
   */
  SoMaterialBinding();

  /**
   * Set the state of the override field.
   * see SoNode::setOverride doc.
   */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * Returns the state of the override field.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

 SoEXTENDER public:
  virtual void        doAction(SoAction *action);
  virtual void        GLRender(SoGLRenderAction *action);
  virtual void        callback(SoCallbackAction *action);
  virtual void        pick(SoPickAction *action);
  virtual void        getPrimitiveCount(SoGetPrimitiveCountAction *action);

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  SoSFBool override;

 protected:
  virtual ~SoMaterialBinding();

  // Reads stuff into instance. Upgrades old bindings.
  virtual SbBool      readInstance(SoInput *in, unsigned short flags);
};

#endif /* _SO_MATERIAL_BINDING_ */

