/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author(s) : VSG (oct 2011)
**=======================================================================*/

#ifndef  _SO_MULTI_SWITCH_
#define  _SO_MULTI_SWITCH_

#include <Inventor/fields/SoMFInt32.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/nodes/SoGroup.h>

SO_PIMPL_PUBLIC_DECLARATION( SoMultiSwitch )

/** 
 * Group node that traverses a set of chosen children.
 * 
 * @ingroup GroupNodes
 * 
 * @DESCRIPTION
 *   This group node traverses none, all or a specified list of its children.
 *   Effectively it is an extended version of the SoSwitch node.
 *   
 *   The #whichChildren field specifies a list of children (by index) to traverse, 
 *   where the first child has index 0.  The #traversalMode field can specify to
 *   traverse none of the children, all of the children, the children in the list
 *   or all the children not in the list.
 *
 *   @B NOTES: @b
 *   - Remember that the default value for #traversalMode is NONE, meaning
 *     that nothing is displayed by default.
 *
 *   - Remember that this node is an SoGroup, NOT an SoSeparator, and therefore does not
 *     provide render caching.  Generally if the children contain geometry, it is good 
 *     practice to make each child an SoSeparator to make render caching more effective.
 * 
 * @FILE_FORMAT_DEFAULT
 *    MultiSwitch {
 *    @TABLE_FILE_FORMAT
 *       @TR whichChildren   @TD [ ]
 *       @TR traversalMode   @TD NONE
 *       @TR inherit         @TD NO
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction,  SoGetBoundingBoxAction,  SoGetMatrixAction,  SoHandleEventAction,  SoRayPickAction @BR
 *        Traverses the chosen child or children. Sets: SoSwitchElement
 * 
 *    SoSearchAction @BR
 *        If the action's Searching-All flag is set, always traverses all children.
 *        Otherwise, traverses just the chosen child or children.
 * 
 * 
 * @SEE_ALSO
 *    SoArray,
 *    SoLevelOfDetail,
 *    SoMultipleCopy,
 *    SoPathSwitch,
 *    SoMultiPathSwitch,
 *    SoSwitch,
 *    SoInteractiveSwitch
 * 
 */
class INVENTOR_API SoMultiSwitch : public SoGroup
{

  SO_NODE_HEADER(SoMultiSwitch);
  SO_PIMPL_PUBLIC_HEADER(SoMultiSwitch)

 public:
  /** Indices of the children to traverse. Default is empty. */
  SoMFInt32 whichChildren;

  /** Traversal Mode enumerated values. */
  enum TraversalMode
  {
    /** None of the children are traversed. */
    NONE     = 0,
    /** All the children are traversed. */
    ALL      = 1,
    /** Traverse children defined by the #whichChildren field. */
    INCLUDE  = 2,
    /** Traverse all children except those defined by the #whichChildren field. */
    EXCLUDE  = 3
  };
  
  /**
   * Defines the way the children list must be managed.
   * @useenum{TraversalMode}.  Default is NONE.
   */
  SoSFEnum traversalMode;

  /** Inheritance Mode enumerated values. */
  enum InheritMode
  {
    /** Current SoMultiSwitch node does not inherit its behavior from state value. */
    NO   = 1 << 0,
    /** #traversalMode field value is inherited from the state.*/
    MODE = 1 << 1,
    /** #whichChildren field value is inherited from the state. */
    LIST = 1 << 2,
    /** #whichChildren and #traversalMode fields values are inherited from the state. */
    ANY  = MODE | LIST
  };

  /**
   * Defines the way this node inherit values from the state.
   * @useenum{InheritMode}. Default is NO.
   */
  SoSFEnum inheritMode;

  /**
   * Creates a multi-switch node with default settings.
   */
  SoMultiSwitch();

  /**
   * Constructor that takes approximate number of children.
   */
  SoMultiSwitch(int nChildren);

  /**
   * Overrides method in SoNode to return FALSE if there is no
   * selected child or the selected child does not affect the state.
   */
   virtual SbBool affectsState() const;

 SoEXTENDER public:
  // Implement actions
  virtual void doAction(SoAction *action);
  virtual void callback(SoCallbackAction *action);
  virtual void GLRender(SoGLRenderAction *action);
  virtual void pick(SoPickAction *action);
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void handleEvent(SoHandleEventAction *action);
  virtual void getMatrix(SoGetMatrixAction *action);
  virtual void search(SoSearchAction *action);
  virtual void write(SoWriteAction *action);
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction *action);

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  virtual void notify(SoNotList *list);

  /**
   * Returns TRUE if a node has an effect on the traversal path. The default
   * method returns FALSE. Node classes such as SoSwitch that allow
   * to traverse scene in different ways returns TRUE.
   */
  virtual bool affectsPath() const;

 protected:
  /** Destructor. */
  virtual ~SoMultiSwitch();

  // Traverses correct child. If matchIndex >=0, child index must match
  virtual void doChild(SoAction *action, int matchIndex = -1);

private:
  void commonConstructor(int nbChildren = -1);
};

#endif // _SO_MULTI_SWITCH_


