/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_PERSPECTIVE_CAMERA_
#define  _SO_PERSPECTIVE_CAMERA_

#include <Inventor/SbPImpl.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/nodes/SoCamera.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoPerspectiveCamera
//
//  perspectiveCamera node.
//
//////////////////////////////////////////////////////////////////////////////

SO_PIMPL_PUBLIC_DECLARATION(SoPerspectiveCamera)

/**
 * Perspective camera node.
 * 
 * @ingroup CameraNodes
 * 
 * @DESCRIPTION
 *   A perspective camera defines a perspective projection from a viewpoint. The
 *   viewing volume for a perspective camera is a truncated right pyramid.
 *   
 *   By default, the camera is located at (0,0,1) and looks along the negative z-axis;
 *   the @B position@b and @B orientation@b fields can be used to change these
 *   values. The #heightAngle field defines the total vertical angle of the
 *   viewing volume; this and the @B aspectRatio@b field determine the horizontal
 *   angle.
 *
 *   See the base class SoCamera for more information about the inherited fields.
 *
 *   Useful algorithms for manipulating a camera are provided in the SoCameraInteractor class.
 * 
 * @FILE_FORMAT_DEFAULT
 *    PerspectiveCamera {
 *    @TABLE_FILE_FORMAT
 *       @TR viewportMapping       @TD ADJUST_CAMERA
 *       @TR position              @TD 0 0 1
 *       @TR orientation           @TD 0 0 1 0
 *       @TR aspectRatio           @TD 1
 *       @TR nearDistance          @TD 1
 *       @TR farDistance           @TD 10
 *       @TR focalDistance         @TD 5
 *       @TR blur                  @TD 0
 *       @TR exposureMode          @TD LINEAR
 *       @TR exposureCompensation  @TD 0
 *       @TR heightAngle           @TD 0.785398
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction,  SoGetBoundingBoxAction,  SoHandleEventAction,  SoRayPickAction @BR
 *        Sets the viewport and camera information in the state.
 *        Sets: SoFocalDistanceElement, SoProjectionMatrixElement, SoViewportRegionElement
 *              SoViewVolumeElement, SoViewingMatrixElement
 * 
 * @SEE_ALSO
 *    SbViewVolume,
 *    SoOrthographicCamera,
 *    SoCameraInteractor
 * 
 * 
 */
class INVENTOR_API SoPerspectiveCamera : public SoCamera {
  
  SO_NODE_HEADER(SoPerspectiveCamera);
  SO_PIMPL_PUBLIC_HEADER(SoPerspectiveCamera)
  
 public:

  /**
   * Vertical angle of the viewing volume.
   * 
   */
  SoSFFloat heightAngle;
  
  /**
   * Creates a perspective camera node with default settings.
   */
  SoPerspectiveCamera();
  
  /** 
   * Scales the height of the camera, in this case, the #heightAngle field.
   */
  virtual void scaleHeight(float scaleFactor);
  
  /**
   * Fills in a view volume structure, based on the camera. If the
   * useAspectRatio field is not 0.0 (the default), the camera uses
   * that ratio instead of the one it has.
   *
   * Note that in ADJUST_CAMERA mode, Inventor automatically modifies the
   * view volume based on the aspect ratio of the viewport.
   * However the view volume values are not changed, only the
   * values passed to OpenGL. In order to get the modified
   * values (i.e., the actual view volume used to render) you must
   * pass the actual viewport aspect ratio to getViewVolume.
   */
  virtual SbViewVolume getViewVolume(float useAspectRatio = 0.0) const;

  /** @copydoc SoCamera::getViewVolume(). */
  virtual SbViewVolume getViewVolume(float useAspectRatio,
                                     float nearDist,
                                     float farDist,
                                     SbVec3f position,
                                     SbRotation orientation) const;
 SoINTERNAL public:
  /** @copydoc SoCamera::initClass(). */
  static void initClass();
  /** @copydoc SoCamera::exitClass(). */
  static void exitClass();

 protected:

  /** @copydoc SoCamera::viewBoundingBox(). */
  virtual void viewBoundingBox(const SbBox3f &box, float aspect, float slack);

  /** Destructor. */
  virtual ~SoPerspectiveCamera();

private:
  void commonConstructor();
};

#endif /* _SO_PERSPECTIVE_CAMERA_ */

