#pragma once

/*============================================================================
**    THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),   **
**               AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.              **
**                                                                          **
** REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS  **
** SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR  **
** WRITTEN AUTHORIZATION OF FEI S.A.S.                                      **
**                                                                          **
**                         RESTRICTED RIGHTS LEGEND                         **
** USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS **
** WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN **
** SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT **
** CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN **
** TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.        **
**                                                                          **
**                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                  **
**                              BORDEAUX, FRANCE                            **
**                            ALL RIGHTS RESERVED                           **
============================================================================*/

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFColorRGBA.h>
#include <Inventor/fields/SoSFColor.h>

class SoAction;
class SoGLRenderAction;
class SoCallbackAction;
class SoWriteAction;
class SoRayPickAction;

/**
 Physically based surface material definition node.

 @ingroup LightNodes

 @DESCRIPTION
   This node defines the current physically based surface material properties
   for all subsequent shapes.
   The #baseColor, #emissiveColor, #specular, #roughness, and #metallic fields
   are taken into account during the rendering when the SoLighModel#model field
   is set to SoLightModel#PHYSICALLY_BASED.

   As described @ref SoLightModel-base-color-def "here", other property nodes
   can override the #baseColor of this class.

   If lighting is turned off (SoLightModel set to BASE_COLOR), only the
   baseColor color is used to render geometry.

# Override material

   To force all geometry following/below this node to use specific color and
   transparency values, call the setOverride() method with true.  Overriding
   the base color overrides the color and transparency values in other nodes
   including SoPackedColor and SoVertexProperty.  This can be useful,
   for example, to highlight a selected object.

   It is also possible to override only a subset of the SoPhysicalMaterial fields.
   If, for example, you only want to override the base color, but not the other
   values, call setIgnored(true) on the fields that you do not want to override.

# VolumeViz shapes

   For scalar (non-RGBA) volumes, the color and alpha value of a voxel is
   affected by two nodes. SoPhysicalMaterial's baseColor field specifies the
   "base" color and alpha values for all voxels. SoTransferFunction specifies
   color and alpha values based on the voxel value.  The final voxel color and
   alpha (before lighting and other effects) is computed by multiplying these
   two color and alpha values. The default material is (0.8, 0.8, 0.8, 1)
   (fully opaque gray). The 0.8 value for R, G and B allows lighting to
   increase the brightness of the voxel. For slice rendering without lighting,
   the application may want to set the material to 1, 1, 1, 1 so that only the
   SoTransferFunction affects the voxel color and alpha. Effectively the alpha
   value of baseColor (aka transparency) is a "global" multiplier that can 
   increase or decrease the transparency of all voxels in the volume.

 @FILE_FORMAT_DEFAULT
    PhysicalMaterial {
    @TABLE_FILE_FORMAT
       @TR baseColor      @TD 0.8 0.8 0.8 1
       @TR emissiveColor  @TD 0 0 0
       @TR specular       @TD 0
       @TR roughness      @TD 0.8
       @TR metallic       @TD 0
    @TABLE_END
    }

 @ACTION_BEHAVIOR
    SoGLRenderAction,  SoCallbackAction  
        Sets the base color, the emissive color, the specular coefficient,
        the roughness coefficient and the metallic coefficient of
        the current physically based material.
    Sets: SoMaterialElement

 @SEE_ALSO
    SoMaterial,
    SoBaseColor,
    SoLightModel,
    SoMaterialBinding,
    SoVertexProperty,

*/
class INVENTOR_API SoPhysicalMaterial : public SoNode
{
  SO_NODE_HEADER(SoPhysicalMaterial);

public:
  /**
   Base color of the surface.
   Default is 0.8 0.8 0.8 1.

   In physically based models, the base color mainly controls the diffuse reflectance.
   Diffuse reflectance plays the most important role in determining the appearance of an object.
   It is affected by the color of the incident light(s) and the angle of each incident light
   relative to the object's normal direction. (It is most intense where the incident light
   falls perpendicular to the surface.)

   As described @ref SoLightModel-base-color-def "here", this field is not the unique way to
   specify a base color in the scene graph.

  */
  SoSFColorRGBA baseColor;

  /**
   Emissive color of the surface.
   Default is 0 0 0.

   The emissive color is the color of the light emitted by the surface. It is not affected by
   the color of the incident light(s) or the angle of each incident light relative to the object's
   normal direction. It is used to simulate self-illuminated surfaces.

   @FIELD_SINCE_OIV 2025.1
  */
  SoSFColor emissiveColor;

  /**
   Specular coefficient of the surface. Values can range from 0.0 to 1.0.
   Default is 0.

   Specular reflection from an object produces highlights. Unlike diffuse reflection,
   the amount of specular reflection depends on the location of the camera - it is brightest
   along the direct angle of reflection. To see this, imagine looking at a metallic ball outdoors
   in the sunlight. As you move your head, the highlight created by the sunlight moves with you
   to some extent. However, if you move your head too much, you lose the highlight entirely.

   This field specifies the intensity of the reflected light.
   The #roughness field controls the size of the highlight.
  */
  SoSFFloat specular;

  /**
   Roughness coefficient of the surface. Values can range from 0.0 to 1.0.
   Default is 0.8.

   The roughness defines the quantity of irregularity on a surface, from a perfectly smooth surface
   (roughness = 0) to highly irregular micro-structure (roughness = 1). This value affects
   the  specular reflection from a mirror-like reflection (roughness = 0) to a scattered
   reflection (roughness = 1).
  */
  SoSFFloat roughness;

  /**
   Metallic coefficient of the surface. Values can range from 0.0 to 1.0.
   Default is 0.

   This parameter controls the metallic appearance of a surface. Most of the time for pure metallic
   surfaces the parameter should be 1, and 0 for other surfaces. For hybrid surfaces (e.g. corroded metals)
   or for styling reasons, you can choose a value between 0 and 1.
  */
  SoSFFloat metallic;

  SoPhysicalMaterial();

  /** @copydoc SoNode::setOverride() */
  inline virtual void setOverride(const SbBool state)
  {
    override.setValue(state);
  }

  /** @copydoc SoNode::isOverride() */
  inline virtual SbBool isOverride() const
  {
    return override.getValue();
  }

SoEXTENDER public:
  virtual void doAction(SoAction* action);
  virtual void GLRender(SoGLRenderAction* action);
  virtual void callback(SoCallbackAction* action);
  virtual void write(SoWriteAction* action);
  virtual void rayPick(SoRayPickAction* action);

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  SoSFBool override;

protected:
  virtual ~SoPhysicalMaterial();

private:
  float m_transparencyValue;
};
