/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/

#pragma once

#include <Inventor/nodes/SoLight.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTexture3.h>

#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/fields/SoSFRotation.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFBool.h>

class SoTextureUnit;
class SoShaderParameter1i;
class SoIndexedTriangleSet;
class SoSwitch;
class SoGLRenderAction;
class SoGetBoundingBoxAction;
class SoMaterial;

/**
 * Node representing a rectangular area light source.
 *
 * @ingroup LightNodes
 *
 * @DESCRIPTION
 *   This node defines a light source that covers a rectangular area.
 *
 *   As a light source node, this node affects subsequent shapes in the scene
 *   graph, depending on the current lighting model. The current transformation
 *   affects this node, and you can place it inside a SoSeparator node to
 *   avoid affecting any objects outside that SoSeparator.
 *
 *   You can also use a node kit to create a light; see the reference page for
 *   SoLightKit.
 *
 *   Unlike other light nodes, SoQuadAreaLight only supports the
 *   SoLightModel::PHYSICALLY_BASED lighting model. This node has no
 *   effect on the rendering if any another lighting model is simultaneously used.
 *
 *   Use the #color and #intensity fiels to control the color and intensity of
 *   the light.
 *
 *   The #location, #orientation, #width, and #height fields define the geometry
 *   of the rectangular area covered by the light.
 *
 *   The #twoSided field defines whether or not to use both sides of the rectangular
 *   area as a source of illumination.
 *
 * @FILE_FORMAT_DEFAULT
 *    QuadAreaLight {
 *    @TABLE_FILE_FORMAT
 *       @TR on                @TD TRUE
 *       @TR intensity         @TD 1
 *       @TR color             @TD 1 1 1
 *       @TR shadowIntensity   @TD 1
 *       @TR location          @TD 0 0 0
 *       @TR orientation       @TD 0 0 1 0
 *       @TR width             @TD 1
 *       @TR height            @TD 1
 *       @TR twoSided          @TD FALSE
 *    @TABLE_END
 *    }
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Activates this light (if so specified) during traversal. All shape nodes that
 *        come after this light in the scene graph are illuminated by this light. The
 *        current transformation affects the light's direction.
 *        Sets: SoLightElement, SoLightIdElement
 *
 * @SEE_ALSO
 *    SoPointLight,
 *    SoSpotLight,
 *    SoLightModel
 *
 *
 * @NODE_SINCE_OIV 2025.1
 */
class INVENTOR_API SoQuadAreaLight : public SoLight
{
  SO_NODE_HEADER(SoQuadAreaLight);

public:
  /**
   * Location of the source (center of the area).
   *
   * Default is (0, 0, 0).
   */
  SoSFVec3f location;

  /**
   * Principal orientation of illumination (normal vector of the area),
   * defined as a SoSFRotation object. The actual direction of illumination
   * is the vector obtained after applying this rotation to its default
   * (0, 0, -1) vector.
   *
   * See SoSFRotation and SbRotation for more details.
   */
  SoSFRotation orientation;

  /**
   * Width of the area.
   *
   * Default is 1.
   */
  SoSFFloat width;

  /**
   * Height of the area.
   *
   * Default is 1.
   */
  SoSFFloat height;

  /**
   * Two-Sided illumination.
   *
   * When TRUE, this enables illumination on both sides of the area.
   *
   * Default is FALSE
   */
  SoSFBool twoSided;

  /**
   * Creates an area light source node with default settings.
   */
  SoQuadAreaLight();

SoEXTENDER public:
  void GLRender(SoGLRenderAction* action) override;
  void getBoundingBox(SoGetBoundingBoxAction* action) override;

SoINTERNAL public:
  static void initClass();
  static void exitClass();

  /**
   * Display the area light as a shape.
   *
   * Default is FALSE
   */
  SoSFBool display;

  void getCameras(SoState* state, SoPath* scene, float visibilityLength, SoNodeList& lightCams) override;

  static SoTexture3* getLTCTexture();

protected:
  virtual ~SoQuadAreaLight();

private:
  void getUpAndRightVectors(SbVec3f& upVector, SbVec3f& rightVector) const;
  SoSeparator* getShapeSep();

  static SoRef<SoTexture3> s_ltcTexture;
  SoRef<SoSeparator> m_shapeSep;
  SoIndexedTriangleSet* m_shape;
  SoSwitch* m_displaySwitch;
  SoMaterial* m_shapeFrontMaterial;
  SoMaterial* m_shapeBackMaterial;
};
