/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#if !defined(SO_RENDERTOTARGET_H)
#define SO_RENDERTOTARGET_H

#include <Inventor/fields/SoMFEnum.h>
#include <Inventor/fields/SoMFNode.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFVec2i32.h>
#include <Inventor/fields/SoSFVec4f.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/SbPImpl.h>

SO_PIMPL_PUBLIC_DECLARATION(SoRenderToTarget)

//@TOBEWRAPPED
/** 
 * @VSGEXT Group node that renders its children to one or more "targets".
 * 
 * @ingroup GroupNodes
 * 
 * @DESCRIPTION
 *  This group node renders its children to one or more render targets.
 *  
 *  Multiple targets can be "attached" to the different outputs of the node.
 *  
 *  This node also provides antialiasing. When FrameBufferObjects (FBO) are used 
 *  (this is the default mode if they are available) multisampling can be configured 
 *  in order to perform antialiasing on the FBO. This MSAA feature is not related 
 *  to the viewer's FSAA. The quality factor is set between 0.f(default value) and 1.f. 
 *  The underlying system will setup the FBO with the correct number of samples according 
 *  to the quality value and according to the attachments configuration.
 *
 *  By default the node clears the targets when the rendering is started. The values used
 *  to clear the buffers can be specified by the fields #clearColorValue, #clearDepthValue
 *  and #clearStencilValue. It is also possible to disable this feature by setting the field
 *  #clearTargets to FALSE.
 *
 *  During rendering, the viewport from the Open Inventor state is used, which means that
 *  the current size of the render area is used. It is possible to change this default 
 *  behavior by setting the #size field. Setting it to (-1, -1) will switch back to the 
 *  default mode, other values will be used as the custom rendering size in pixels. 
 *  The targets are automatically resized to the correct size.
 *
 * The SoRenderToTextureProperty node can also be used to create a texture image
 * and is useful for simpler cases.
 *
 *  NOTES:
 *  - Performance:
 *    - Using the TARGET_COPY mode at each frame really reduces performance because a 
 *      transfer from GPU memory to CPU memory is performed to transfer the data to the 
 *      SoSFImage field of the texture.
 *    - Antialiasing (MSAA) can really reduce performance and increase memory consumption. 
 *      It can even be impossible to use antialiasing with large FLOAT color attachments.
 *  
 *  - Target limitations:
 *    - Only SoTexture2, SoTexture3 and SoTextureCubeMap nodes are supported. Using other
 *      types in the targets field will result in SoDebugErrors.
 * 
 *  - By default the textures are resized using the following rules:
 *    - RBGA uint8 for the color attachements. (Internal mode is RGBA8)
 *    - Luminance float32 for the depth attachment. (Internal mode is DEPTH_COMPONENT24).
 *    - Luminance uint8 for the stencil attachment.
 *      It is possible to override the default internal mode for the textures by using the internalFormat field of SoTexture.
 *
 *  - Rendering
 *    - SoRenderToTarget is an SoSeparator and can be added directly to the scene graph. In this
 *      case its children are rendered as part of the main scene graph and inherit properties
 *      (camera, light, material, etc) from the main scene graph. @BR @BR
 *    - SoRenderToTarget can also be used separate from the main scene graph and re-rendered
 *      as-needed by applying an SoGLRenderAction. In this case the children of the SoRenderToTarget
 *      will normally include a camera, light, etc node. But also note that, in this case, the
 *      OpenGL calls normally made by the viewer and/or SoSceneManager will not be done automatically.
 *      For example, the OpenGL depth test is not enabled. Use an SoDepthBuffer node to do this.
 *
 *  - Bounding box
 *    - WARNING: The 'boundingBoxIgnoring' default value is TRUE. @BR
 *      Usually the SoRenderToTarget scenegraph is not really part of the main scene and should
 *      not have any effect on the bounding box computed for the main scene. For this reason,
 *      SORTED_OBJECT_BLEND transparency is not supported for shapes that are children of this node.
 *      But in some cases, it can be useful to include the SoRenderToTarget scenegraph's bounding box,
 *      like when doing post-processing effects. In that case, manually set 'boundingBoxIgnoring' to FALSE.
 *
 *  - Hardware compatibility:
 *    - All the exposed features are supported by the FBO rendering subsystem. PBuffer and 
 *      pixmap might not support some of them. For example it is not possible to have multiple 
 *      color attachments in PBuffer mode. Those alternative modes are supported only for 
 *      compatibility with legacy hardware.
 *
 *
 * @NODE_SINCE_OIV 8.6
 *    
 * @FILE_FORMAT_DEFAULT
 *    RenderToTarget {
 *    @TABLE_FILE_FORMAT
 *       @TR modes                     @TD AUTO
 *       @TR targets                   @TD NULL
 *       @TR layers                    @TD 0
 *       @TR clearTargets              @TD TRUE
 *       @TR clearColorValue           @TD (0.F, 0.F, 0.F, 0.F)
 *       @TR clearDepthValue           @TD 1.F
 *       @TR clearStencilValue         @TD 0
 *       @TR antialiasingQuality       @TD 0.F
 *       @TR size                      @TD (-1, -1)
 *       @TR enableFragmentsQuery      @TD FALSE
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Performs an offscreen rendering using either FBO, PBuffer or pixmap.
 *        Sets: SoCacheElement
 * 
 * @SEE_ALSO
 *    SoTexture,
 *    SoTexture2.
 * 
 * 
 */
class INVENTOR_API SoRenderToTarget : public SoSeparator
{
  SO_NODE_HEADER( SoRenderToTarget );
  SO_PIMPL_PUBLIC_HEADER(SoRenderToTarget)

public:

  /** Attachment 
   *
   * This enum is used to specify which output of the rasterizer is considered.
   * Use these values as indices when setting the #targets and #modes fields.
   */
  enum Attachment
  {
    /** Default attachment used for rendering. */
    COLOR0,
    COLOR1,
    COLOR2,
    COLOR3,
    COLOR4,
    COLOR5,
    COLOR6,
    COLOR7,
    COLOR8,
    COLOR9,
    COLOR10,
    COLOR11,
    COLOR12,
    COLOR13,
    COLOR14,
    COLOR15,
    DEPTH,
    STENCIL,
    DEPTH_STENCIL,
    /** Not used, for security only */
    LAST_ENTRY
  };

  /**
   * This enum defines modifiers for the auto detection mechanism.
   */
  enum Mode
  {
    /** Default value, use OIV auto detection algorithm.
     * The detection algorithm checks the content of the #targets field.
     * NOTE: The node will use INTERNAL for DEPTH and COLOR0, NONE for the others.
     */
    AUTO,

    /** Use an internal buffer for the rendering.
     * Some buffers are useful only for shaders or for correct rendering (like depth buffer
     * for the depth tests) but they don't need to be attached to a specific target.
     */
    INTERNAL,

    /** 
     * After the rendering the rendered content is copied back to the node in CPU memory.
     */
    TARGET_COPY,

    /**
     * Do not use any target.
     */
    NONE
  };

  /** 
   * This defines the mode modifier for the targets.
   * For each target it is possible to specify a mode. Use enum #Mode. Default is AUTO.
   *
   * NOTE: This field is indexed using the Attachment enum.
   * For example (in pseudo-code):
   * - modes[COLOR0] = TARGET_AND_COPY;
   * - modes[STENCIL] = INTERNAL;
   */
  SoMFEnum modes;

  /**
   * This field allows to attach a texture to a specific attachment.
   * Supported target types are SoTexture2, SoTexture3 and SoTextureCubeMap.
   * Default is empty (no targets).
   *
   * NOTE: This field is indexed using the Attachment enum.
   * For example (in pseudo-code):
   * - targets[COLOR0] = rgbaTexture;
   * - targets[STENCIL] = stencilTexture;
   */
  SoMFNode targets;

  /**
   * This field is used when targets are SoTexture3 or SoTextureCubeMap.
   * It describes the layer where the color attachment must be written, along
   * the Z dimension for SoTexture3, or the cube map face for SoTextureCubeMap
   * (the cube map faces order is presented in the table below).
   * This field is indexed by color attachments (i.e. from COLOR0 to COLOR15)
   * and has no effect when targets are of type SoTexture2 or when target
   * attachments are DEPTH or STENCIL, which require a SoTexture2 Target.
   *
   * <table>
   *   <tr>
   *     <th>Layer number</th>
   *     <th>Cube Map Face</th>
   *   </tr>
   *   <tr>
   *     <td>0</td>
   *     <td>Positive X</td>
   *   </tr>
   *   <tr>
   *     <td>1</td>
   *     <td>Negative X</td>
   *   </tr>
   *   <tr>
   *     <td>2</td>
   *     <td>Positive Y</td>
   *   </tr>
   *   <tr>
   *     <td>3</td>
   *     <td>Negative Y</td>
   *   </tr>
   *   <tr>
   *     <td>4</td>
   *     <td>Positive Z</td>
   *   </tr>
   *   <tr>
   *     <td>5</td>
   *     <td>Negative Z</td>
   *   </tr>
   * </table>
   */
  SoMFInt32 layers;

  /**
   * If this field is set the targets are cleared before the rendering. Default is TRUE.
   */
  SoSFBool clearTargets;

  /**
   * Value used to clear the color buffer before the rendering. Default is (0.0, 0.0, 0.0, 0.0).
   */
  SoSFVec4f clearColorValue;

  /**
   * Value used to clear the depth buffer before the rendering. Default is 1.0.
   */
  SoSFFloat clearDepthValue;

  /**
   * Value used to clear the stencil buffer before the rendering. Default is 0.
   */
  SoSFInt32 clearStencilValue;

  /**
   * This field defines the antialiasing quality between 0.0 and 1.0 for the rendering.
   * The value 1.0 represents the maximun quality possible on this hardware.
   * Default is 0.
   *
   * NOTE:
   *   Hardware limitations: The result depends on the support for multisampling in FBO or the FSAA support for PBuffers.
   */
  SoSFFloat antialiasingQuality;

  /**
   * This field allows a custom rendering size for the render to texture.
   * When this field is set to the default value (-1, -1) the global viewport size from the viewer
   * is used. The size is in pixels.
   */
  SoSFVec2i32 size;

  /**
   * This field enables or disables the query counter used to count the number of fragments
   * rasterized during the render to texture operation. Default is FALSE.
   *
   * The method getRasterizedSamplesCount() can be used to get the result of the query.
   */
  SoSFBool enableFragmentsQuery;

  /**
   * Default constructor.
   */
  SoRenderToTarget();

  /**
   * Indicates if this node can be used on the actual hardware.
   * When using a debug build of Open Inventor, some "no context available"
   * warning messages may be generated. You can ignore them or see
   * SoGLExtension for an example of using SoGLContext to avoid them.
   */
  static SbBool isSupported(SoState* state = NULL);

  /**
   * Returns the number of samples generated by the rasterizer during the last GLRender action.
   *
   * To enable this feature the enableFragmentsQuery field must be set to TRUE.
   * Otherwise the result is undefined.
   */
  int getRasterizedSamplesCount() const;

SoINTERNAL public:

  /** @copydoc SoNode::initClass */
  static void initClass();

  /** @copydoc SoNode::exitClass */
  static void exitClass();
 
SoEXTENDER public:

  /** @copydoc SoSeparator::GLRenderBelowPath */
  virtual void GLRenderBelowPath(SoGLRenderAction *action);

  /** @copydoc SoSeparator::GLRenderInPath */
  virtual void GLRenderInPath(SoGLRenderAction *action);

  /** @copydoc SoSeparator::GLRenderOffPath */
  virtual void GLRenderOffPath(SoGLRenderAction *action);

protected:
  /** Destructor. */
  virtual ~SoRenderToTarget();

 private:

   void commonConstructor();
};

#endif // SO_RENDERTOTEXTURE_H


