/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

                                                             /* include files */
#ifndef _SO_SHADER_OBJECT_H_
#define _SO_SHADER_OBJECT_H_

#include <Inventor/SbBasic.h>
#include <Inventor/SbString.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSFFilePathString.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoMFUniformShaderParameter.h>
#include <Inventor/SbEnums.h>

#include <Inventor/STL/cassert>
#include <Inventor/STL/map>
#include <Inventor/helpers/SbConstCharMap.h>

class SoGLShaderContainer ;
class SoGLShaderObject ;
class SoFieldSensor ;
class SoNodeSensor ;
class SoFileSensor ;
class SoShaderParser;

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

/**
 * @VSGEXT Abstract node class which defines a shader object.
 *
 * @ingroup ShaderNodes
 *
 * @DESCRIPTION
 *   This abstract class is the parent of classes that define a programmable graphics pipeline stage
 *   that composes a shader program.
 *
 *   There are five types of shader objects that can be added to a shader program.
 *   Any of these stages can be user defined.
 *   - @B Vertex shader@b @BR
 *     The vertex shader is executed once for each vertex (usually in parallel).
 *     The main purpose is to transform each vertex's 3D position from model space to projection space.
 *     Vertex shaders can manipulate properties such as position, color and texture coordinate,
 *     but cannot create new vertices.
 *
 *   - @B Tessellation control shader @b @BR
 *     This shader accepts a list of vertices defined as a patch to control
 *     the amount of tessellation applied to the patch. Following the execution of this shader, a
 *     tessellator computes a set of triangles in a parametric space.
 *
 *   - @B Tessellation evaluation shader@b @BR
 *     This shader is executed at least once for each vertex that was created by the tesselator in the
 *     parametric space. The TES takes the parametric coordinate and the patch data output by the TCS
 *     to generate a final position for the surface.
 *
 *   - @B Geometry shader@b @BR
 *     The geometry shader acts on a complete primitive (triangle or line):
 *     it can modify existing primitives, it can insert (create) new primitives, it can remove (destroy) existing primitives.
 *
 *   - @B Fragment shader@b @BR
 *     Fragment shaders compute color and other attributes of each fragment.
 *
 *   Note: Compute shaders are not part of the rendering pipeline. They represent a shader stage used entirely
 *   for computing arbitrary information.
 *
 *   Shader object nodes cannot be inserted directly in a scene graph. They must
 *   be added to the field @link SoShaderProgram#shaderObject shaderObject @endlink of an SoShaderProgram node.
 *
 *   A shader object is defined by the following properties:
 *      - @B Source program@b, which is the shader's source code (see #sourceProgram field),
 *      - @B Uniform parameters @b set by the application (see #parameter field),
 *      - @B State @b (active or not) (see #isActive field).
 *
 *   The source program can be specified either by a string containing the program source
 *   code, or by a filename which contains the program source code. How the sourceProgram
 *   field is interpreted depends on the field #sourceType.
 *
 *   The supported shading language of the program source is @B OpenGL Shader Language (GLSL) @b.
 *   Furthermore, the Open Inventor shader API or VolumeViz shader API must be used to write
 *   any GLSL shader program. See @ref ShaderAPI for detail.
 *
 *   @I Uniform @i parameters can be set through the #parameter field.  Uniform means,
 *   in the case of a vertex or geometry program, a value which is the same for all vertices
 *   in a primitive, and, in the case of a fragment program, a value which is the same for all
 *   fragments created by a primitive.
 *   Each uniform parameter is represented by an instance of a specific subclass of
 *   SoUniformShaderParameter. For example, an SoShaderParameter1i holds a single integer value.
 *   A uniform parameter has no effect if it is not valid, that is, if there is
 *   no corresponding name (identifier) in the GLSL source program.
 *   An SoShaderParameter1i must be used for each texture sampler in order to specify the texture
 *   unit and texture sampler uniform parameter name pair.
 *
 *   A vertex shader can also use @I vertex @i parameters, which are per-vertex
 *   data passed from the application to the vertex shader. Vertex parameters are
 *   represented by an instance of a specific subclass of SoVertexShaderParameter.
 *   For example, an SoVertexShaderParameter1f holds a set of floating point values
 *   and an SoVertexShaderParameter3f holds a set of SbVec3f values. Vertex parameter
 *   nodes are property nodes (similar to materials or normals) and should be added
 *   directly in the scene graph, not in the shader object.
 *
 *   @B Tips@b:
 *   - Set the environment variable OIV_GLSL_DEBUG to get the GLSL compile/link output in the console.
 *
 *   - If you set the environment variable OIV_SHADER_CHECK_INTERVAL, the
 *     shader source file is checked for a change every @I n @i seconds, where @I n @i is
 *     the value specified by the variable.
 *     This allows you to edit a shader source file without needing to restart your
 *     application after each shader modification.
 *
 * @FILE_FORMAT_DEFAULT
 *    This is an abstract class. See the reference page of a derived class for the
 *    format and default values.
 *
 * @EXAMPLE
 *  Simple fragment shader with one uniform parameter:
 * \if_cpp
 * \code
 *   // First load the fragment shader code
 *   SoFragmentShader* fragmentShader = new SoFragmentShader();
 *   fragmentShader->sourceProgram = "filename.glsl";
 *
 *   // Set a shader parameter
 *   // The addShaderParameter1i method is equivalent to:
 *   //     SoShaderParameter1i *parameter = new SoShaderParameter1i;
 *   //     parameter->name  = "data1";
 *   //     parameter->value = 1;
 *   //     fragmentShader->parameter.set1Value(0, parameter);
 *   fragmentShader->addShaderParameter1i( "data1", 1 );
 *
 *   // Associate fragment shader with a shader program node
 *   SoShaderProgram* shaderProgram = new SoShaderProgram();
 *   shaderProgram->shaderObject.set1Value(0, fragmentShader);
 *   root->addChild( shaderProgram );
 * \endcode
 * \endif
 * \if_dotnet
 * \code
 *   // Simple fragment shader with one uniform parameter
 *   // First load the fragment shader code
 *   SoFragmentShader fragmentShader = new SoFragmentShader();
 *   fragmentShader.sourceProgram.Value = "filename.glsl";
 *
 *   // Set the shader parameter
 *   SoShaderParameter1i parameter = new SoShaderParameter1i();
 *   parameter.name.Value  = "data1";
 *   parameter.value.Value = 1;
 *   fragmentShader.parameter[0] = parameter;
 *
 *   // Associate fragment shader with a shader program node
 *   SoShaderProgram shaderProgram = new SoShaderProgram();
 *   shaderProgram.shaderObject[0] = fragmentShader;
 *   root.AddChild(shaderProgram);
 * \endcode
 * \endif
 * \if_java
 * \code
 *   // Simple fragment shader with one uniform parameter
 *   // First load the fragment shader code
 *   SoFragmentShader fragmentShader = new SoFragmentShader();
 *   fragmentShader.sourceProgram.setValue( "filename.glsl" );
 *
 *   // Set the shader parameter
 *   SoShaderParameter1i parameter = new SoShaderParameter1i();
 *   parameter.name.setValue( "data1" );
 *   parameter.value.setValue( 1 );
 *   fragmentShader.parameter.set1Value( 0, parameter );
 *
 *   // Associate fragment shader with a shader program node
 *   SoShaderProgram shaderProgram = new SoShaderProgram();
 *   shaderProgram.shaderObject.set1Value( 0, fragmentShader );
 *   root.addChild(shaderProgram);
 * \endcode
 * \endif
 *
 * @SEE_ALSO
 *    @ref ShaderAPI,
 *    SoVertexShader, SoGeometryShader, SoFragmentShader, SoShaderProgram, SoShaderParameter,
 *    SoUniformShaderParameter, SoVertexShaderParameter, SoTessellationControlShader, SoTessellationEvaluationShader
 */
class INVENTOR_API SoShaderObject : public SoNode
{

  SO_NODE_ABSTRACT_HEADER(SoShaderObject);

public:

  /** Shader Object source type possible values. */
  enum SourceType
  {
      /**
       * The source is an ARB vertex or fragment program.
       */
      SoDEPRECATED_ENUM_VALUE(10000,"ARB Support has been removed in OIV 10. This enum is kept for compatibility with previously serialized files.")
      ARB_PROGRAM,

      /**
       * The source is a CG program.
       */
      SoDEPRECATED_ENUM_VALUE(10000,"CG Support has been removed in OIV 10. This enum is kept for compatibility with previously serialized files.")
      CG_PROGRAM,

      /**
       * The source is an OpenGL Shading Language program.
       */
      GLSL_PROGRAM,

      /**
       * Only the name of the file containing the source is given (default).
       */
      FILENAME
  };

  /**
   * Specifies if the shader object is active or not.
   */
  SoSFBool isActive;

  /**
   * Specifies the shader object's source type.
   * The type of source can be either a filename containing the program (#FILENAME), or
   * a string containing the source program (#GLSL_PROGRAM).  Use enum #SourceType. Default is FILENAME.
   *
   * NOTE: The source type must be specified before the source program (#sourceProgram)
   * is specified.
   *
   */
  SoSFEnum sourceType;

  /**
   * Contains the shader object's source program, specified by a
   * filename (#sourceType set to #FILENAME) or by the string containing
   * the program (#sourceType set to #GLSL_PROGRAM).
   *
   * If the
   * filename is not an absolute path name, the list of directories maintained by
   * SoInput is searched. If the source program is not found in any of those directories,
   * then the file is searched for relative to the directory from which the
   * SoShaderObject node was read.
   *
   * NOTE: The source type (#sourceType) must be specified before the source program
   * is specified.
   */
  SoSFFilePathString sourceProgram;

  /**
   * Contains the shader's uniform parameters.
   *
   */
  SoMFUniformShaderParameter parameter;

  /**
  * Convenience method to create an SoShaderUniformParameter with the specified name and value
  * and add it to the given shader object.
  * UniformParamType is the type of parameter to add.
  * This function also generates the following helper methods not visible in the documentation:
  *
  * - UniformParamType* addShaderParameter{1,2,3,4}i(const SbString&, ParamValueType)
  * - UniformParamType* addShaderParameter{1,2,3,4}f(const SbString&, ParamValueType)
  * - UniformParamType* addShaderParameterArray{1,2,3,4}i(const SbString&, ParamValueType)
  * - UniformParamType* addShaderParameterArray{1,2,3,4}f(const SbString&, ParamValueType)
  * - UniformParamType* addShaderParameterMatrix
  *
  * These are template methods, so ParamValueType must have an operator= compatible with
  * the field value of the corresponding SoShaderParameter
  * (for example, addShaderParameter2f can take an SbVec2f as ParamValueType)
  */
  template<typename UniformParamType, typename ParamValueType>
  UniformParamType* addShaderParameter(const SbString& name, ParamValueType val);

  /** Type of the shader */
  enum ShaderType
  {
    /** The shader is a vertex shader. */
    VERTEX_SHADER = SbEnums::SHADER_TYPE_VERTEX,
    /** The shader is a geometry shader. */
    GEOMETRY_SHADER = SbEnums::SHADER_TYPE_GEOMETRY,
    /** The shader is a fragment shader. */
    FRAGMENT_SHADER = SbEnums::SHADER_TYPE_FRAGMENT,
    /** The shader is a tessellation control shader. */
    TESSELLATION_CONTROL_SHADER = SbEnums::SHADER_TYPE_TESS_CTRL,
    /** The shader is a tessellation evaluation shader. */
    TESSELLATION_EVALUATION_SHADER = SbEnums::SHADER_TYPE_TESS_EVAL,
    /** The shader is a compute shader. */
    COMPUTE_SHADER = SbEnums::SHADER_TYPE_COMPUTE,
  };

  /** Must be redefined by derived class */
  virtual ShaderType getShaderType() const = 0;

  /**
  * UniformParamType is the type of parameter to set.
  * This function also generates the following helper methods not visible in the documentation:
  *
  * - void setShaderParameter{1,2,3,4}i(const SbString&, ParamValueType)
  * - void setShaderParameter{1,2,3,4}f(const SbString&, ParamValueType)
  * - void setShaderParameterArray{1,2,3,4}i(const SbString&, ParamValueType)
  * - void setShaderParameterArray{1,2,3,4}f(const SbString&, ParamValueType)
  * - void setShaderParameterParameterMatrix
  *
  * These are template methods, so ParamValueType must have an operator= compatible with
  * the field value of the corresponding SoShaderParameter
  * (for example, setShaderParameter2f can take an SbVec2f as ParamValueType).
  */
  template<typename UniformParamType, typename ParamValueType>
  void setShaderParameter(const SbString& name, ParamValueType val);

#ifndef HIDDEN_FROM_DOC
  //Generate all setFragmentParameter* methods
#define MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(ParamType)\
  /** setShaderParameter[XXX] */ \
  template<typename ParamValueType>\
  void\
  setShaderParameter ## ParamType(const SbString& name, ParamValueType val)\
  {\
  setShaderParameter<SoShaderParameter ## ParamType>(name, val);\
  }
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(1i)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(2i)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(3i)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(4i)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(1f)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(2f)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(3f)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(4f)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(Array1f)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(Array2f)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(Array3f)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(Array4f)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(Array1i)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(Array2i)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(Array3i)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(Array4i)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(Matrix)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(Matrix3)
  MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER(MatrixArray)
#undef MAKE_HELPHER_FUNC_SET_SHADER_PARAMETER

//Generate all addShaderParameter* methods
#define MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(ParamType)\
  /** addShaderParameter[XXX] */ \
  template<typename ParamValueType>\
  SoShaderParameter ## ParamType *\
  addShaderParameter ## ParamType(const SbString& name, ParamValueType val)\
  {\
  return addShaderParameter<SoShaderParameter ## ParamType>(name, val);\
  }
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(1i)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(2i)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(3i)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(4i)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(1f)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(2f)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(3f)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(4f)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(Array1f)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(Array2f)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(Array3f)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(Array4f)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(Array1i)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(Array2i)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(Array3i)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(Array4i)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(Matrix)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(Matrix3)
  MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER(MatrixArray)
#undef MAKE_HELPHER_FUNC_ADD_SHADER_PARAMETER
#endif //HIDDEN_FROM_DOC

  /*----------------------------------------------------------------------------*/

SoINTERNAL public:

  /** Set needed infos used to decorate shaders source (glsl version, driver versions...) mostly used for cache */
  void setShaderObjectInfos(SoState* state, const SbString& shaderVersion, const SbString& shaderProfile);

  /** Get the source type. */
  SourceType getSourceType();

  static void initClass();
  static void exitClass() ;

  /**
   * Returns true if this shader is a VSG internal shader.
   * it is used to avoid serialization for iv files and ART scenario.
   */
  SbBool isInternal() const;

  // Returns TRUE if at least one parameter is of SoShaderStateMatrixParameter type
  SbBool containStateMatrixParameters() const
  { return (m_stateMatrixParameters.getLength() != 0) ; }

  const SbString& getSourceProgram() const
  { return m_sourceProgram;}

  /**
   * True if the source file has been found or if source type is not a file
   */
  inline bool isFileFound() const;

  /** True if shaderprogram must update this object*/
  inline bool needUpdate() const { return m_needUpdate; }

  /** Force shader object reconstruction */
  void invalidate();

  /** Define name with the given value */
  void setDefine(const char* name, const char* value);
  void setDefine(const char* name, const SbString& value)
  { setDefine(name,value.toLatin1()); }

  /** Remove name from the define list */
  void removeDefine(const char* name);

  /** cleanup existing object if invalid */
  void checkValidity();

  /** Set shader version */
  inline void setVersion(const SbString& version) { m_version = version; }

  /** Get shader version */
  const SbString& getVersion();

  /** Get shader profile (compatibility or core) */
  const SbString& getProfile();

  /** Set shader profile (compatibility or core) */
  inline void setProfile(const SbString& profile) { m_profile = profile; }

  /** return shader_object availability on current contex */
  static bool isShaderObjectAvailable(SoState* state);

  /** return GL_ARB_shading_language_100 availability on current context */
  static bool isGLSLLanguageAvailable(SoState* state);

  /** Force source code to be reloaded */
  void invalidateSource();

  /** From Internal data return the source code to load */
  void buildSourceToLoad(SbString &builtSource);

  /** Return absolute path of given filename */
  SbString getFileAbsolutePath(const SbString& filename);

  /** catch fields changes */
  virtual void notify(SoNotList *list);

  /** Add all used headers to filenames */
  void getHeaderFilenames( std::vector<SbString>& filenames );

  /** Sets whether or not this shader has been generated to be a fixed pipeline shader. */
  void setIsFixedPipeline(bool isFixedPipelineShader)
  {
    m_isFixedPipelineShader = isFixedPipelineShader;
  }

  /** @see setIsFixedPipeline() */
  bool isFixedPipeline() const
  {
    return m_isFixedPipelineShader;
  }

protected:

  friend class SoShaderProgram;
  /** A define and its value */
  typedef SbConstCharMap<SbString> DefineMap;
  typedef DefineMap::iterator DefineMapIt;

  /** A vector of SbString */
  typedef std::vector<SbString> SbStringVector;

  /** Define name with the given value */
  void setGlobalDefine(const DefineMap& map);

  /** Define name with the given value */
  void setDefine(DefineMap::value_type& p);

  /** Return true if define name has been set */
  bool hasDefine(const char* name) const;

  /** List of header source to include */
  typedef std::set<SbString> HeaderSet;
  typedef HeaderSet::iterator HeaderSetIt;

  /** Add a Header to include */
  void setGlobalHeader(const HeaderSet& map);

  // Extract the source from a given file if sourceType equals FILENAME.
  void readSource();

  /**
  * Constructor.
  */
  SoShaderObject();

  // Destructor
  virtual ~SoShaderObject();

private:
  SoShaderParser* m_parser;
  SbString          m_sourceProgram;

  /** Shader version */
  SbString m_version;
  SbString m_profile;
  /** Shader extension map */
  typedef std::map<std::string, std::string> ExtensionMap;
  ExtensionMap m_extensionList;

  typedef std::map<SbString, SbString> StrStrMap;

  /**
   * Get :
   * -the XXX of "#version XXX" in version and the rest of source code in source
   * -the profile mode of "#version XXX profile"
   * -this list of #extension xxx
   * -add all //!oiv_include "foo.h" found in the source code into m_includeSourceVector
   */
  void extractInfosFromSource(SbString& version, SbString& source,SbString& profile, ExtensionMap& extensionList);

  /** Return true is line contains space-like characters oly*/
  bool isEmptyLine(const std::string& line) const;

  /** Add "#extension p.first : p.second" at the beginning of source */
  static SbString definesExtension(const SbString& source, const ExtensionMap::value_type& p);

  /** Add "#define p.first p.second" at the beginning of source */
  static SbString definesAcc(const SbString& source, const DefineMap::value_type& p);

  /** Return all filenames used by the shader (including oiv_include filename).
      Empty if the file is in memory or there is no file. */
  void getSourceFilename(std::vector<SbString>& filenames) const;

  /** List of defines */
  DefineMap m_defineMap;

  /** Directly comes from the shader we are attached to */
  DefineMap m_defineGlobalMap;

  /** Directly comes from the shader we are attached to */
  HeaderSet m_headerGlobalSet;

  /** Source of all //!oiv_include found in the source*/
  SbStringVector m_includeSourceVector;

  /** Filename of all //!oiv_include found in the source*/
  SbStringVector m_includeFilenameVector;

  /** Directory where the file has been found */
  SbString m_directoryFound;

  // Update the list of state matrix parameters
  void updateStateMatrixList() ;

  // Store the list of state matrix parameters
  SoNodeList    m_stateMatrixParameters ;

  // True if the shader can't be read
  bool m_fileFound;

  /** True if shaderprogram must update this object*/
  bool m_needUpdate;

  /** True if shaderObject should reread data from disk */
  bool m_needRead;

  bool m_isFixedPipelineShader;

  /** Cache relative name to full path */
  static StrStrMap s_shadersPaths;
  static SbThreadRWMutex s_shadersPathsMutex;
  static bool s_debugCache;
};
/*----------------------------------------------------------------------------*/

/*******************************************************************************/
template<typename UniformParamType, typename ParamValueType>
UniformParamType*
SoShaderObject::addShaderParameter(const SbString& name, ParamValueType val)
{
  UniformParamType *param = dynamic_cast<UniformParamType *>(parameter.findUniformParameterByName(name));
  if (param)
  {
    // This parameter already exist => prefer to replace it
    param->value.setValue(val);
  }
  else
  {
    param = new UniformParamType;
    param->name = name;
    param->value.setValue(val);
    parameter.addShaderParameter(param);
  }

  return param;
}

/*****************************************************************************/
template<typename UniformParamType, typename ParamValueType>
void
SoShaderObject::setShaderParameter(const SbString& name, ParamValueType val)
{
  SoUniformShaderParameter* param = parameter.findUniformParameterByName(name);
  assert(param);
  assert(param->isOfType(UniformParamType::getClassTypeId()) == TRUE);
  UniformParamType* i = static_cast<UniformParamType*>(param);
  i->value = val;
}

/*****************************************************************************/
bool
SoShaderObject::isFileFound() const
{
  return m_fileFound;
}

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif /*_SO_SHADER_OBJECT_H_*/


