/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

                                                            /* include files */
#ifndef _SO_SHADER_PROGRAM_H_
#define _SO_SHADER_PROGRAM_H_

#include <Inventor/SbBasic.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoMFNode.h>
#include <Inventor/nodes/SoShaderParameter.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoVertexShader.h>
#include <Inventor/nodes/SoTessellationControlShader.h>
#include <Inventor/nodes/SoTessellationEvaluationShader.h>
#include <Inventor/nodes/SoTexture.h>
#include <Inventor/nodes/SoGeometryShader.h>
#include <Inventor/nodes/SoShaderParameterImage.h>
#include <Inventor/STL/vector>
#include <Inventor/STL/cassert>
#include <Inventor/helpers/SbConstCharMap.h>
#include <Inventor/elements/SoEnvironmentElement.h>
#include <memory>
#include <unordered_map>
#include <Inventor/renderer/RendererResourceMacro.h>
#include <Inventor/sensors/SoFileSensor.h>


#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

class SoGLRenderAction;
class SoGLShaderProgram ;
class SoGLShaderObject;
class SoFieldSensor ;
class SoLight;
class SoCache;
class SoShaderParameterBufferObject;
class SoComputeShader;

/*! \cond PRIVATE */
namespace inventor { namespace helper { class ShaderLibrary; } }
namespace inventor { namespace renderer { class ShaderStateUniforms; } }
/*! \endcond */

/*----------------------------------------------------------------------------*/

/**
 * @VSGEXT Shader program property node.
 *
 * @ingroup ShaderNodes
 *
 * @DESCRIPTION
 *   This property node defines the complete shader program for all subsequent shapes.
 *   A shader program may contain any graphics pipeline stage defined by the derivatives
 *   of a SoShaderObject. These stage programs are stored in the field shaderObject.
 *   Any Modifications that occur in the shader objects are dynamically detected and the state is updated.
 *
 *   The supported shading language of the program source is @B OpenGL Shader Language (GLSL) @b.
 *   Furthermore, the Open Inventor shader API or VolumeViz shader API must be used to write
 *   any GLSL shader program. See @ref ShaderAPI for detail.
 *
 *   @I Uniform @i shader parameters can be set per shader object and are available for all pipeline
 *   stages of the shader program. Uniform means a constant value during the shader execution.
 *   A uniform parameter is represented by an instance of a specific subclass of SoUniformShaderParameter.
 *   For example, an SoShaderParameter1i holds a single integer value.
 *
 *   @I Vertex @i parameters can be accessible from a vertex shader object. Vertex parameters are
 *   per-vertex data passed from the application to the vertex shader. A vertex parameter is
 *   represented by an instance of a specific subclass of SoVertexShaderParameter.
 *   For example, an SoVertexShaderParameter1f holds a set of floating point values
 *   and an SoVertexShaderParameter3f holds a set of SbVec3f values.
 *
 *   SoShaderProgram has specialized subclasses for volume visualization. If you are using
 *   the VolumeViz extension, see SoVolumeShader and SoVolumeRenderingQuality nodes.
 *
 * @B Notes@b:
 *   - For geometry shaders: the EXT_geometry_shader4 OpenGL extension must be supported by
 *     your graphics board.
 *
 *   - For tesselation shaders: the GL_ARB_tessellation_shader OpenGL extensions must be supported by
 *     your graphics board.
 *
 *   - When writing geometry shaders, the input and output primitive types must be specified in the shader code.
 *     See https://developer.openinventor.com/index.php/general-documentation/upgrade-oiv9-shader-code/#geomshader
 *     for more information.
 *
 *   - Transparency and fast editing are compatible with shader modified geometry, but most
 *     actions use the vertices of the basic shape (the vertices stored in the scene graph).
 *     This includes, for example, SoGetBoundingBoxAction, SoGetPrimitiveCountAction and SoRayPickAction.
 *
 * @B Tips@b:
 *  - To see the output from the GLSL compiler/linker, set the environment variable OIV_GLSL_DEBUG.
 *
 *  - If you set the environment variable OIV_SHADER_CHECK_INTERVAL, the
 *    shader source file is checked for a change every @I n @i seconds, where @I n @i is
 *    the value specified by the variable.
 *    This allows you to edit the shader source code without needing to restart your
 *    application after each shader modification.
 *
 *  - If your shader makes geometry transparent (sets alpha values < 1), you
 *    should set the #generateTransparency field to TRUE.  This ensures that Open Inventor
 *    knows to apply the correct handling for the current transparency mode.
 *
 *  - Since the GLSL specification doesn't currently provide a "#include" directive,
 *    Open Inventor provides this service through the comment directive "//!oiv_include". This provides greater
 *    flexibility in implementing complex GLSL shaders. Included files are loaded using SoInput
 *    and respect the same search path order.  For example:
 *    \par
 *    \code
 *      //!oiv_include <MyShaderDirectory/common.h>
 *    \endcode
 *
 *
 * @FILE_FORMAT_DEFAULT
 *    SoShaderProgram {
 *      @TABLE_FILE_FORMAT
 *        @TR #shaderObject          @TD NULL
 *        @TR #geometryInputType     @TD TRIANGLES_INPUT
 *        @TR #geometryOutputType    @TD TRIANGLE_STRIP_OUTPUT
 *        @TR #vertexProgramTwoSide  @TD FALSE
 *        @TR #shadowShader          @TD FALSE
 *        @TR #maxGeometryOutputVertices @TD -1
 *        @TR #generateTransparency  @TD FALSE
 *        @TR #patchLength           @TD 0
 *      @TABLE_END
 *    }
 *
 * @EXAMPLE
 *  Shader program with one fragment shader with one uniform parameter.
 * \if_cpp
 * \code
 *   // First load the fragment shader code
 *   SoFragmentShader* fragmentShader = new SoFragmentShader();
 *   fragmentShader->sourceProgram = "filename.glsl";
 *
 *   // Set a shader parameter
 *   // The addShaderParameter1i method is equivalent to:
 *   //     SoShaderParameter1i *parameter = new SoShaderParameter1i;
 *   //     parameter->name  = "data1";
 *   //     parameter->value = 1;
 *   //     fragmentShader->parameter.set1Value(0, parameter);
 *   fragmentShader->addShaderParameter1i( "data1", 1 );
 *
 *   // Associate fragment shader with a shader program node
 *   SoShaderProgram* shaderProgram = new SoShaderProgram();
 *   shaderProgram->shaderObject.set1Value(0, fragmentShader);
 *   root->addChild( shaderProgram );
 * \endcode
 * \endif
 * \if_dotnet
 * \code
 *   // Simple fragment shader with one uniform parameter
 *   // First load the fragment shader code
 *   SoFragmentShader fragmentShader = new SoFragmentShader();
 *   fragmentShader.sourceProgram.Value = "filename.glsl";
 *
 *   // Set the shader parameter
 *   SoShaderParameter1i parameter = new SoShaderParameter1i();
 *   parameter.name.Value  = "data1";
 *   parameter.value.Value = 1;
 *   fragmentShader.parameter[0] = parameter;
 *
 *   // Associate fragment shader with a shader program node
 *   SoShaderProgram shaderProgram = new SoShaderProgram();
 *   shaderProgram.shaderObject[0] = fragmentShader;
 *   root.AddChild(shaderProgram);
 * \endcode
 * \endif
 * \if_java
 * \code
 *   // Simple fragment shader with one uniform parameter
 *   // First load the fragment shader code
 *   SoFragmentShader fragmentShader = new SoFragmentShader();
 *   fragmentShader.sourceProgram.setValue( "filename.glsl" );
 *
 *   // Set the shader parameter
 *   SoShaderParameter1i parameter = new SoShaderParameter1i();
 *   parameter.name.setValue( "data1" );
 *   parameter.value.setValue( 1 );
 *   fragmentShader.parameter.set1Value( 0, parameter );
 *
 *   // Associate fragment shader with a shader program node
 *   SoShaderProgram shaderProgram = new SoShaderProgram();
 *   shaderProgram.shaderObject.set1Value( 0, fragmentShader );
 *   root.addChild(shaderProgram);
 * \endcode
 * \endif
 *
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Sets the Open Inventor state with the active shader program.
 *        Sets: SoGLShaderProgramElement
 *
 * @SEE_ALSO
 *    @ref ShaderAPI,
 *    SoFragmentShader,
 *    SoGeometryShader,
 *    SoShaderObject,
 *    SoTessellationControlShader,
 *    SoTessellationEvaluationShader,
 *    SoUniformShaderParameter,
 *    SoVertexShader,
 *    SoVertexShaderParameter
 *
 *
 */
class INVENTOR_API SoShaderProgram : public SoNode
{

  SO_NODE_HEADER( SoShaderProgram );
  RENDERER_RESOURCE(SoShaderProgram);

public:
  /** Geometry input type. Used with field #geometryInputType. */
  enum GeometryInputType
  {
    /**
     * The input geometry should be interpreted as points.
     * Geometry shaders that operate on points are valid only for the SoPointSet and
     * SoIndexedPointSet nodes. There is only a single vertex available
     * for each geometry shader invocation.
     */
    POINTS_INPUT = GL_POINTS,

    /**
     * The input geometry should be interpreted as lines.
     * Geometry shaders that operate on line segments are valid only for the
     * SoLineSet or SoIndexedLineSet nodes.  There are two vertices
     * available for each geometry shader invocation. The first vertex refers to
     * the vertex at the beginning of the line segment and the second vertex
     * refers to the vertex at the end of the line segment.
     */
    LINES_INPUT = GL_LINES,

    /**
     * The input geometry should be interpreted as triangles.
     * Geometry shaders that operate on triangles are valid only
     * for geometry nodes that generate triangles, for example, SoTriangleStripSet.
     * There are three vertices available for each program invocation. The first,
     * second and third vertices refer to attributes of the first, second and
     * third vertex of the triangle, respectively. Default.
     */
    TRIANGLES_INPUT = GL_TRIANGLES
  };

  /**
   * Geometry ouput type. Used with field #geometryOutputType.
   */
  enum GeometryOutputType
  {
    POINTS_OUTPUT = GL_POINTS,
    LINE_STRIP_OUTPUT = GL_LINE_STRIP,
    /**
     * Default.
     */
    TRIANGLE_STRIP_OUTPUT = GL_TRIANGLE_STRIP
  };

  /**
   * Specifies the list of shader objects (i.e., vertex shaders, geometry and fragment shaders)
   * which form the shader program. Be careful, with some languages (CG_PROGRAM or ARB_PROGRAM),
   * only one vertex program and one fragment program can be active at the same time.
   * In this case, only the first vertex shader and the first fragment shader are used.
   */
  SoMFNode shaderObject;

  /**
   * Specifies the input primitive type of the current geometry shader if any (not used otherwise).
   * Use enum #GeometryInputType. Default is TRIANGLES_INPUT.
   *
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFEnum geometryInputType;

  /**
   * Specifies the output primitive type of the current geometry shader if any (not used otherwise).
   * Use enum #GeometryOutputType. Default is TRIANGLE_STRIP_OUTPUT.
   *
   * @FIELD_SINCE_OIV 7.0
   */
  SoSFEnum geometryOutputType;

  /**
   * If set to TRUE, vertex shaders will operate in two-sided color mode.
   * Default is FALSE.
   *
   * @FIELD_SINCE_OIV 7.2
   */
  SoSFBool vertexProgramTwoSide;

  /**
   * Only used when an SoShadowGroup is active.
   * Default is FALSE.
   * If set to FALSE, a default shader will be used during
   * the shadowmap generation pass.
   * If TRUE, the shader will be used as is and must handle the
   * shadowmap pass correctly:
   * - If the uniform OivShadowPass is true, call OivGenerateShadowMap()
   * and output nothing into gl_FragColor, but discard fragments if needed.
   * - If the uniform OivShadowPass is false, follow the normal render path
   * If the shader doesn't modify the depth (with a discard or an alpha test),
   * Open Inventor will handle the shadowmap generation pass automatically.
   * Default is FALSE @BR
   * See also: SoShadowGroup
   *
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFBool shadowShader;

  /**
   * Set the maximum number of vertices the geometry shader will emit in one invocation.
   * Default is -1 which means it is set to the hardware limit.
   *
   * @FIELD_SINCE_OIV 8.1
   */
  SoSFInt32 maxGeometryOutputVertices;


  /**
   * If set to TRUE, then shapes affected by this shader will be considered transparent.
   * Otherwise, the shape transparency is deducted from the state.
   *
   * This allows Open Inventor to apply the correct handling for the current transparency mode.
   *
   * Default is FALSE.
   *
   * @FIELD_SINCE_OIV 9.0
   */
  SoSFBool generateTransparency;

  /**
   * Set the length of the fixed-size collection of vertices used by tessellation shaders.
   * Default is 0.
   *
   * @FIELD_SINCE_OIV 9.3
   */
  SoSFInt32 patchLength;

  /**
   * Specifies a list of SoShaderParameterBufferObject to use with this shader.
   * Default is empty.
   * @FIELD_SINCE_OIV 9.8
  */
  SoMFNode bufferObjects;

  /**
   * Specifies a list of SoShaderParameterImage nodes to use with this shader.
   * Default is empty.
   * @FIELD_SINCE_OIV 10.10
   */
  SoMFNode images;

  /**
   * Constructor
   */
  SoShaderProgram();

  /**
  * Returns the fragment shader at the specified position.
  */
  inline SoFragmentShader* getFragmentShader(int pos) const;

  /**
  * Returns the vertex shader at the specified position.
  */
  inline SoVertexShader* getVertexShader(int pos) const;

  /**
   *
   * Returns the geometry shader at the specified position.
   */
  inline SoGeometryShader* getGeometryShader(int pos) const;

  /**
   * Convenience method to create a fragment shader with the specified filename and
   * add it at the specified position. Return value is the new fragment shader.
   */
  virtual SoFragmentShader* setFragmentShader(int pos, const SbString& filenameOrSource,
                                              SoShaderObject::SourceType sourceType = SoShaderObject::FILENAME);

  /**
  * Convenience method to create a vertex shader with the specified filename and
  * add it at the specified position. Return value is the new vertex shader.
  */
  virtual SoVertexShader* setVertexShader(int pos, const SbString& filenameOrSource,
                                          SoShaderObject::SourceType sourceType = SoShaderObject::FILENAME);

  /**
   *
   * Convenience method to create a geometry shader with the specified filename and
   * add it at the specified position. Return value is the new geometry shader.
   */
  virtual SoGeometryShader* setGeometryShader(int pos, const SbString& filenameOrSource,
                                              SoShaderObject::SourceType sourceType = SoShaderObject::FILENAME);

  /**
   * Convenience method to create a compute shader with the specified filename and
   * add it at the specified position. Return value is the new compute shader.
   */
  virtual SoComputeShader* setComputeShader(int pos, const SbString& filenameOrSource,
                                            SoShaderObject::SourceType sourceType = SoShaderObject::FILENAME);

  /**
   *
   * Returns the number of reserved texture units.
   */
  static unsigned int getNumReservedTextures();

  /**
   * Returns the tessellation control shader at the specified position.
   */
  inline SoTessellationControlShader* getTessellationControlShader(int pos) const;

  /**
   * Returns the tessellation evaluation shader at the specified position.
   */
  inline SoTessellationEvaluationShader* getTessellationEvaluationShader(int pos) const;

  /**
   * Convenience method to create a tessellation control shader with the specified filename and
   * add it at the specified position. Return value is the new tessellation control shader.
   */
  virtual SoTessellationControlShader* setTessellationControlShader(int pos, const SbString& filenameOrSource,
                                              SoShaderObject::SourceType sourceType = SoShaderObject::FILENAME);

  /**
    * Convenience method to create a tessellation evaluation shader with the specified filename and
   * add it at the specified position. Return value is the new tessellation evaluation shader.
   */
  virtual SoTessellationEvaluationShader* setTessellationEvaluationShader(int pos, const SbString& filenameOrSource,
                                              SoShaderObject::SourceType sourceType = SoShaderObject::FILENAME);

  /*----------------------------------------------------------------------------*/

  /**
   * Convenience method to create an SoShaderParameterImage with the specified name and value
   * and add it to this shader program.
   */
  SoShaderParameterImage* addShaderParameterImage(const SbString& name, SoTexture* tex);

SoEXTENDER public:

  // GLRender
  virtual void doAction(SoAction *action);
  virtual void GLRender( SoGLRenderAction *action );
  virtual void getBoundingBox( SoGetBoundingBoxAction *action );
  virtual void pick( SoPickAction *action );

SoINTERNAL public:
  /** GPU type, used by getGPUVendorType() */
  enum GPUVendorType
  {
    GPU_TYPE_NOT_INITIALIZED = -1,
    GPU_NVIDIA = 0,
    GPU_ATI = 1,
    GPU_INTEL = 2,
    GPU_FIREPRO_MAC = 3,
    GPU_MESA = 4
  };

  // Constant, max number of texture image properties that can be set
  enum { MAX_IMG_UNITS = 8 };

  /** Returns the name of the main slot for the given ShaderType */
  static const char* getMainSlotName(SbEnums::ShaderType shaderType);

  /** Return true if shader has a specified type of shader in its private shader list. */
  bool hasPrivateMainShaderType(SbEnums::ShaderType shaderType) const;

  /** Return true if shader has a specified type of shader in its public shader list. */
  bool hasPublicMainShaderType(SbEnums::ShaderType shaderType) const;

  /** Return true if shader has a specified type of shader in its shader list. */
  bool hasMainShaderType(SbEnums::ShaderType shaderType) const;

  /** Returns whether or not this shader contains a fixed pipeline generated main of the given type. */
  bool hasFixedPipelineMainShaderType(SbEnums::ShaderType shaderType) const;

  /** Returns whether or not this shader contains a fixed pipeline generated main. */
  bool hasFixedPipelineMain() const;

  /** Fill shaderParameters with all uniform parameters (hidden included) */
  void getAllParameters(SoState* state, std::vector<SoUniformShaderParameter*>& shaderParameters);

  /** Fill vector with all needed shader object (hidden&co) */
  void getShaderObjects(std::vector<SoShaderObject*>& shaderObjectsList) const;

  /** return all linked shader object filenames of a specific type */
  SbString getLinkedShaderFileNames(SbEnums::ShaderType shaderType) const;

  /** display all linked shader object filenames */
  void displayLinkedFileNames() const;

  //Return true if its an internal Oiv/Vviz name
  static bool isOivReservedName(const std::string &paramName);

  /** Handle field change */
  virtual void notify(SoNotList *list);

  /** Notifications Policies (enable or disable) */
  enum NotificationsPolicy
  {
    DISABLE_NOTIFICATIONS,
    ENABLE_NOTIFICATIONS,
  };

  /** Define name with the given value */
  void setDefine(const char* name, const char* value, NotificationsPolicy notifPolicy = DISABLE_NOTIFICATIONS);

  void setDefine(const char* name, const SbString& value, NotificationsPolicy notifPolicy = DISABLE_NOTIFICATIONS)
  {
    setDefine(name, value.toLatin1(), notifPolicy);
  }

  /** Return the value of specified define. Return false if define is not set.
   * Note that this method can return true and value may be empty. In this case, the define
   * is set but is empty. */
  bool getDefine(const char* name, SbString& value) const;

  /** Remove name from the define list */
  void removeDefine(const char* name);

  /** Add header with the given source */
  void setHeader(const SbString& filename);

  /** Remove name from the header list */
  void removeHeader(const SbString& name);

  /**
   * Convert the offset into a GL_TEXTUREX
   */
  static GLint offsetToGLTextureUnit(SoGLRenderAction* action, const char* offset);

  /**
   * Return the texture unit number (the X of GL_TEXTUREX)
   */
  static int offsetToTextureUnit(SoGLRenderAction* action, const char* offset);

  // Initializes the classes.
  static void initClass();
  static void exitClass();

  /** Force shader program reconstruction */
  void invalidate();

  /** Return true if it's a GLSL shader */
  bool isGlslProgram() const;

  /** Returns true if this program contains a Compute Shader */
  bool isComputeShader() const;

  /** Return true if shader used for shadowmap generation */
  bool isShadowShader() const;

  /** Return shader used during shadowmap pass */
  SoShaderProgram* getShadowPassShader() const;

  /** Return true if shader used for shadowmap generation */
  void setShadowShader(bool flag);

  /** This object will be added to the shader program but won't be displayed.
   * It also increases its ref count */
  void setHiddenShaderObject(const char* objName, SoShaderObject* obj);

  /** Remove the given object and decrease its ref count */
  void removeHiddenShaderObject(const char* objName);

  /** Get the given object */
  SoShaderObject* getHiddenShaderObject(const char* objName) const;

  /** Get the map of hidden shader objects */
  const SbConstCharMap<SoShaderObject*>& getHiddenShaderObjects() const;

  /** @return GPUVendorType (ATI, NVIDIA, INTEL, ...) */
  static GPUVendorType getGPUVendorType();


  /**
  * setup private shader stage hiddenName to shaderSource if needed.
  */
  template<typename T>
  T* setupPrivateShaderStage(const char* hiddenName, const SbString& shaderSource, SoShaderObject::SourceType sourceType = SoShaderObject::FILENAME)
  {
    // check if it is already allocated
    T* fp = (T*)getHiddenShaderObject(hiddenName);
    if (!fp || (fp->sourceProgram.getValue() != shaderSource))
    {
      fp = new T;
      fp->sourceProgram.setValue(shaderSource);
      fp->sourceType = sourceType;
      setHiddenShaderObject(hiddenName, fp);
    }
    return fp;
  }

  /** Remove all shader of specified type in private shader list */
  void removeAllPrivateShaderType(SbEnums::ShaderType shaderType);

  /** State of a shader library for this program */
  enum ShaderLibraryState
  {
    LIBRARY_NOT_INSTALLED,
    LIBRARY_INSTALLED_EMPTY,
    LIBRARY_INSTALLED,
  };

  /** Returns the current state of the given shader library for this program */
  ShaderLibraryState getLibraryState( inventor::helper::ShaderLibrary* library ) const;

  /** Sets the state of the given shader library for this program */
  void setLibraryState( inventor::helper::ShaderLibrary* library, ShaderLibraryState libraryState );

  bool getPreviousNeedDepthPeelingLibrary() const
  {
    return m_previousNeedDepthPeelingLibrary;
  }

  /** Sets whether or not this shader has been generated to be a fixed pipeline shader. */
  void setIsFixedPipeline(bool isFixedPipelineShader)
  {
    m_isFixedPipelineShader = isFixedPipelineShader;
  }

  /** @see setIsFixedPipeline() */
  bool isFixedPipeline() const
  {
    return m_isFixedPipelineShader;
  }

  /** Set vertexProgramTwoSide according to state */
  void setVertexProgramTwoSide(SoState* state, bool allowVertexTwoSideLighting = true);

  /**
   * Set to TRUE if this shader program needs to manage multisampled input or target textures.
   *
   * When TRUE, this allows OpenInventor to setup the necessary defines.
   *
   * Default is FALSE.
   *
   * @FIELD_SINCE_OIV 2024.2.1
   */
  SoSFBool multiSampling;

protected:
  typedef std::vector<SoShaderObject*> ShaderObjectVector;
  typedef SbConstCharMap<SoShaderObject*> ShaderObjectMap;
  /** Just for gcc 3.1 compat */
  typedef SoShaderObject::DefineMap DefineMap;
  typedef SoShaderObject::HeaderSet HeaderSet;

  /** member variables */
  struct Members
  {
    Members();
    ~Members();
    void unref();

    /** True if this shader is for shadowmap rendering only */
    bool m_isShadowShader;

    /** List of shaders used for shadowing */
    ShaderObjectMap m_hiddenShaderObjects;

    // Contains the list of vertex and fragment shaders,
    // before modification.
    // Enable to track add/remove/change or shader objects.
    std::vector<SoShaderObject*> m_prevShaderObject;
    std::vector< SoNode * > m_toUnrefShaderObjects;

    /** Shader used to generate the shadowmap */
    SoShaderProgram* m_shadowPassShader;

    /** List of defines */
    DefineMap m_defineMap;

    /** List of headers */
    HeaderSet m_headerMap;

    /** Keep track of validity only for debug purpose...*/
    bool m_isValid;
  };
  Members m_members;

  // Destructor
  virtual ~SoShaderProgram();

  // Check if all shader objects are valid
  SbBool isValidShaderObjects() const;

  // Check if at least one of the shader objects has
  // its "isActive" field set to true
  SbBool isOneShaderObjectActive();

  /** Return members  */
  static SoShaderProgram::Members* getMembers(const SoShaderProgram* prog);

private:
  // Required for compat with some old IV files. (cf ebug#3765)
  SoMFNode prevShaderObject;

  /** Add a global defines to the shader object*/
  void addDefines(SoShaderObject* obj);

  /** Add a global headers to the shader */
  void addHeaders(SoShaderObject* obj);

  /** Add hidden shaders to the given list  */
  void addHiddenShaderObjects(ShaderObjectVector& shaderObjectsList);

  /** return true if obj1's version < obj2's version */
  static bool lessVersion(SoShaderObject* obj1, SoShaderObject* obj2);

  /** Return the max version of the shader object list */
  static SbString getMaxVersion(const ShaderObjectVector& objList);

  /** Return true if obj1's profile "core" and obj2 is "compatibility"*/
  static bool lessProfile(SoShaderObject* obj1, SoShaderObject* obj2);

  /** Return "compatibility", "core" or "" if a shader has one of this mode in this order
   * of importance */
  static SbString getLoosestProfile(const ShaderObjectVector& objList);

  /** Return true if shaderObjects field has changed */
  bool hasShaderObjectsChanged() const;

  /** Copy shaderObjects into  m_prevShaderObject */
  void updatePrevShaderObject();

  /** Helper method to invalidate a all objects in a ShaderObjectMap */
  static void invalidate(ShaderObjectMap::value_type& p);

  /** Add OIV_USE_ATI define if running on ATI or add OIV_USE_INTEL define if running on INTEL */
  void addGPUVendorDefine();

  /** Send buffer storage to GL */
  void setShaderBufferObject(SoGLRenderAction* action);

  /** Send texture images to GL action */
  void setShaderTextureImages(SoGLRenderAction* action) const;

  /** 
   * Update file sensor and add/remove shader API objects if needed 
   * @param hiddenShaderObjectName If adding a hidden shader, string ID of the hidden shader otherwise NULL
   */
  void notifyAddedShaderObject(const char* hiddenShaderObjectName);
  
  /** Handle source code change */
  static void fileSensorCB(void *data, SoSensor *) ;

  /** Update list of monitored file for live editing */
  void updateFileSensor();

  /** Check for shader source change*/
  std::unique_ptr<SoFileSensor> m_fileSensor;

  /** Used to invalidate the shaders when we switch to depth peeling */
  bool m_previousNeedDepthPeelingLibrary;

  typedef std::unordered_map<inventor::helper::ShaderLibrary*, ShaderLibraryState> ShaderLibraryStateMap;
  ShaderLibraryStateMap m_shaderLibrariesStates;

  /** see enum GPUVendorType, init to GPU_TYPE_NOT_INITIALIZED */
  static GPUVendorType s_gpuVendorType;

  /** First texture unit used by internal shaders*/
  static int s_firstUsedTextureUnit;

  /** Use to enable or disable cache debug output*/
  static bool s_debugCache;


  bool m_isFixedPipelineShader;

  friend class SoUniformShaderParameter;
};
/*----------------------------------------------------------------------------*/

/*******************************************************************************/
SoFragmentShader*
SoShaderProgram::getFragmentShader(int pos) const
{
  assert(pos >= 0 && pos < shaderObject.getNum());
  SoShaderObject* obj = static_cast<SoShaderObject*>(shaderObject[pos]);

  if ( !obj )
    return NULL;

  assert(obj->getTypeId() == SoFragmentShader::getClassTypeId());

  return static_cast<SoFragmentShader*>(obj);
}

/*******************************************************************************/
SoVertexShader*
SoShaderProgram::getVertexShader(int pos) const
{
  assert(pos >= 0 && pos < shaderObject.getNum());
  SoShaderObject* obj = static_cast<SoShaderObject*>(shaderObject[pos]);

  if ( !obj )
    return NULL;

  assert(obj->getTypeId() == SoVertexShader::getClassTypeId());

  return static_cast<SoVertexShader*>(obj);
}

/*******************************************************************************/
SoGeometryShader*
SoShaderProgram::getGeometryShader(int pos) const
{
  assert(pos >= 0 && pos < shaderObject.getNum());
  SoShaderObject* obj = static_cast<SoShaderObject*>(shaderObject[pos]);

  if ( !obj )
    return NULL;

  assert(obj->getTypeId() == SoGeometryShader::getClassTypeId());

  return static_cast<SoGeometryShader*>(obj);
}

/*******************************************************************************/
SoTessellationControlShader*
SoShaderProgram::getTessellationControlShader(int pos) const
{
  assert(pos >= 0 && pos < shaderObject.getNum());
  SoShaderObject* obj = static_cast<SoShaderObject*>(shaderObject[pos]);

  if ( !obj )
    return NULL;

  assert(obj->getTypeId() == SoTessellationControlShader::getClassTypeId());

  return static_cast<SoTessellationControlShader*>(obj);
}

/*******************************************************************************/
SoTessellationEvaluationShader*
SoShaderProgram::getTessellationEvaluationShader(int pos) const
{
  assert(pos >= 0 && pos < shaderObject.getNum());
  SoShaderObject* obj = static_cast<SoShaderObject*>(shaderObject[pos]);

  if ( !obj )
    return NULL;

  assert(obj->getTypeId() == SoTessellationEvaluationShader::getClassTypeId());

  return static_cast<SoTessellationEvaluationShader*>(obj);
}

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif /*_SO_SHADER_PROGRAM_H_*/


