/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : SGI (MMM YYYY)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : Nicolas DAGUISE (Oct 2004)
**=======================================================================*/


#ifndef  _SO_TEXT_2_
#define  _SO_TEXT_2_

#include <Inventor/fields/SoMFString.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoMFVec3f.h>
#include <Inventor/nodes/SoShape.h>

class SoAnnoText3;
class SoFontServer;

SO_PIMPL_PUBLIC_DECLARATION( SoText2 );

/**
 * Screen-aligned 2D text shape node.
 * 
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This node defines one or more strings of 2D text. The text is always 
 *   horizontal, aligned with the screen and does not change size with distance
 *   from the camera. However the position of the text is specified in 3D units
 *   and is affected by viewing and modeling transformations. 2D text cannot be
 *   rotated, but because it always faces the user, it is often used for
 *   annotation, for example labels on axes.
 *
 *   The text position is always at the local origin (0,0,0). Rotations and scales
 *   have no effect on the orientation or size of 2D text, just the location. 
 *
 *   Note: A transform node, e.g. SoTranslation, must be used to position the text in 3D space.
 *
 *   Text supports both single byte Latin1 (ISO8859-1) characters and multi-byte 
 *   UNICODE characters.  ISO8859-1 is an 8-bit character set which is a superset
 *   of the older 7-bit ASCII character set. See the SbString class for more information.
 *   
 *   @B Rendering: @b
 *
 *   The typeface and size (default = 10) are specified using an SoFont node.
 *   Note that for SoText2 the size is interpreted as a value in pixels, not 3D units.
 *
 *   Setting the SoFont @I renderStyle @i field to TEXTURE is recommended for best
 *   appearance and performance. (antialiasing is applied).  
 *
 *   Additional rendering styles can be applied using an SoTextProperty node, 
 *   including underline, overline, strikethrough, background and background outline.
 *
 *   SoText2 is drawn using the diffuse color of the current material
 *   (SoMaterial). Lighting is not applied, regardless of the lighting model. 
 *   Textures do not affect SoText2 (note this is independent of the fact
 *   that textures are used internally to render SoText2). 
 *   Complexity (SoComplexity) does not affect SoText2.
 *   Drawing styles (SoDrawStyle) other than INVISIBLE do not affect SoText2.
 *
 *   @B Justification: @b
 *
 *   Horizontal alignment of the text can be specified using the #justification
 *   field.  Both horizontal and vertical alignment can be controlled using an 
 *   SoTextProperty node, however the justification setting in the SoText2 node
 *   overrides the horizontal setting unless it is explicitly set to INHERITED.
 *
 *   @B Clipping: @b
 *
 *   If the transformed position (origin point) of the SoText2 is clipped, then
 *   the entire string is clipped, even if some part of the string should
 *   conceptually be visible.  If the origin point is inside the view volume
 *   then the string is clipped similar to other geometry.
 *   
 *   @B Bounding box: @b
 *
 *   Because 2D text is screen-aligned, it has some unusual characteristics. For
 *   example, the 3D bounding box surrounding a 2D text string depends on the current
 *   camera and the current viewport size, since changing the field of view or the
 *   mapping onto the window changes the relative size of the text with respect to
 *   the rest of the scene. 
 *
 *   When computing the bounding box of a 2D text node (see SoGetBoundingBoxAction), 
 *   be careful to:
 *   - Initialize the bounding box action with the correct viewport region. @BR
 *     Typically this information is queried from the viewer.
 *   - Make sure the action will traverse the camera node. @BR
 *     Apply the action to a node above the camera or create a path to the 2D text
 *     node from the root of the scene graph (e.g. using SoSearchAction).
 *
 *   @B Picking: @b
 *
 *   Text can be picked (SoRayPickAction) like any other geometry. If a text
 *   node has been picked then SoPickedPoint can return an SoTextDetail object.
 *   This object contains information about which string (in a multi-string text
 *   node) was picked and which character within the string was picked.
 *
 *   @B Render caching: @b
 *
 *   If your Open Inventor version is older than 9.2, then SoText2 nodes using 
 *   CENTER or RIGHT justification cannot be render cached
 *   because the bounding box and justification offset depend on the camera.  As a
 *   result, no SoSeparator above the SoText2 will be able to build a render cache
 *   or a bounding box cache and this can reduce performance. If possible put other 
 *   geometry, that can be cached, under its own SoSeparator. SoText2 nodes using 
 *   LEFT justification (the default) can be cached.  Since Open Inventor 9.2,
 *   SoText2 nodes can be render cached in all cases.
 *
 *   Antialiasing: @BR
 *   Shape Antialiasing type is SoShape::TEXT.
 *
 *   @B Export limitations:@b
 *
 *   - SoText2 can be exported to PDF3D/U3D (SoToPDFAction), but... @BR
 *     due to limitations of that file format, the text will not be "screen locked".
 *
 *   - SoText2 can be exported to various Hardcopy formats (SoVectorizeAction), but... @BR
 *     native fonts are used, so the requested font name must be available on
 *     the rendering device.
 * 
 * @FILE_FORMAT_DEFAULT
 *    Text2 {
 *    @TABLE_FILE_FORMAT
 *       @TR string          @TD ""
 *       @TR spacing         @TD 1
 *       @TR justification   @TD LEFT
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction @BR
 *        Draws text based on the current font, at a location based on the current
 *        transformation.
 * 
 *    SoRayPickAction @BR
 *        Performs a pick on the text. Text will be picked if the picking ray intersects
 *        the bounding box of the strings. The string index and character position are
 *        available from the SoTextDetail.
 * 
 *    SoGetBoundingBoxAction @BR
 *        Computes the bounding box that encloses the text.
 * 
 * @SEE_ALSO
 *    SoFont,
 *    SoFullSceneAntialiasing,
 *    SoText3,
 *    SoTextDetail,
 *    SoTextProperty
 */
class INVENTOR_API SoText2 : public SoShape {

  SO_PIMPL_PUBLIC_HEADER( SoText2 );

  SO_NODE_HEADER(SoText2);

public:
   /** Justification types. */
  enum Justification {
    /**
     *  Left edges of all strings are aligned 
     */
    LEFT   = 0x01,
    /**
     *  Centers of all strings are aligned 
     */
    CENTER = 0x03,
    /**
     *  Right edges of all strings are aligned 
     */
    RIGHT  = 0x02,
    /** 
     *  The text justification is inherited from the current SoTextProperty::alignmentH value.
     */
    INHERITED = 0x04
  };

  /**
   * The text string(s) to display. Each string will appear on its own line.
   * 
   */
  SoMFString string;
  /**
   * Defines the distance (in the negative y direction) between the base points of
   * successive strings, measured with respect to the current font height. A value of
   * 1 indicates single spacing, a value of 2 indicates double spacing, and so on.
   * 
   */
  SoSFFloat spacing;
  /**
   * Specifies horizontal alignment of strings. With LEFT justification, the left
   * edge of the first line is at the (transformed) origin, and all left edges are
   * aligned. RIGHT justification is similar. CENTER justification places the center
   * of the first string at the (transformed) origin, with the centers of all
   * remaining strings aligned under it.
   * Use enum #Justification. Default is LEFT.
   *
   * Both horizontal and vertical alignment can be specified using an SoTextProperty
   * node. However the justification setting in this node overrides the horizontal
   * setting in SoTextProperty unless this field is set to INHERITED.
   */
  SoSFEnum justification;

  /**
   * Creates a 2D text node with default settings.
   */
  SoText2();

SoEXTENDER public:
  virtual void GLRender(SoGLRenderAction* action);
  virtual void rayPick(SoRayPickAction* action);
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction* action);
  void testRayPick(SoRayPickAction* action, SoNode* node);
  
  // Computes bounding box of text
  virtual void computeBBox(SoAction* action, SbBox3f& box, SbVec3f& center);

  /**
   * Returns an alternative representation of this node 
   * for PDF and U3D exports.
   */
  virtual SoNode* getAlternateRep( SoAction* action );

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  void setRenderZoom(bool zoom);
  bool isRenderZoom();

  virtual void notify(SoNotList *);

protected:
  // This method is a no-op for SoText2, since there are no
  // primitives it can generate
  virtual void generatePrimitives( SoAction* action );

  virtual ~SoText2();

  friend class SoAnnoText3;
  friend class SoFontServer;
private:
  void commonConstructor();
};

#endif // _SO_TEXT_2_

/**/


