/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_TRANSFORM_
#define  _SO_TRANSFORM_

#include <Inventor/fields/SoSFRotation.h>
#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/nodes/SoTransformation.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoTransform
//
//  Geometric transformation node.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * General 3D geometric transformation node.
 * 
 * @ingroup TransformNodes
 * 
 * @DESCRIPTION
 *   This node defines a geometric 3D transformation consisting of (in order) a
 *   (possibly) non-uniform scale about an arbitrary point, a rotation about an
 *   arbitrary point and axis, and a translation. (While the transformations can be
 *   thought of as being applied in that order, matrices are actually premultiplied
 *   in the opposite order. Therefore, the operations are listed in the reverse order
 *   throughout this reference page.)
 * 
 * @FILE_FORMAT_DEFAULT
 *    Transform {
 *    @TABLE_FILE_FORMAT
 *       @TR translation        @TD 0 0 0
 *       @TR rotation           @TD 0 0 1 0
 *       @TR scaleFactor        @TD 1 1 1
 *       @TR scaleOrientation   @TD 0 0 1 0
 *       @TR center             @TD 0 0 0
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction,  SoGetBoundingBoxAction,  SoRayPickAction @BR
 *        Accumulates transformation into the current transformation.
 *        Sets: SoModelMatrixElement
 * 
 *    SoGetMatrixAction @BR
 *        Returns the matrix corresponding to the total transformation.
 * 
 * 
 * @SEE_ALSO
 *    SoMatrixTransform,
 *    SoResetTransform,
 *    SoRotation,
 *    SoRotationXYZ,
 *    SoScale,
 *    SoTransformManip,
 *    SoTransformSeparator,
 *    SoTranslation
 * 
 * 
 */
class INVENTOR_API SoTransform : public SoTransformation {

  SO_NODE_HEADER(SoTransform);

 public:
  // Fields
  /**
   * Translation vector.
   * 
   */
  SoSFVec3f           translation;
  /**
   * Rotation specification.
   * 
   */
  SoSFRotation        rotation;
  /**
   * Scale factors.
   * Negative and null values may cause undesirable results.
   */
  SoSFVec3f           scaleFactor;
  /**
   * Rotational orientation for scale.
   * 
   */
  SoSFRotation        scaleOrientation;
  /**
   * Origin for scale and rotation.
   * 
   */
  SoSFVec3f           center;

  /**
   * Creates a transformation node with default settings.
   */
  SoTransform();

  /**
   * Sets the node to translate the origin to the @B fromPoint @b and rotate the
   * negative z-axis (0,0,-1) to lie on the vector from @B fromPoint @b to
   * @B toPoint @b. This always tries to keep the "up" direction the positive
   * y-axis, unless that is impossible. All current field values in the node are
   * replaced.
   */
  void                pointAt(const SbVec3f &fromPoint,
                              const SbVec3f &toPoint);

  /**
   * Returns composite matrix that transforms from object space to
   * scale space (the space after the scale).
   */
  void                getScaleSpaceMatrix(SbMatrix &mat, SbMatrix &inv) const;
  /**
   * Returns composite matrix that transforms from object space to
   * rotation space (the space after the scale and rotation).
   */
  void                getRotationSpaceMatrix(SbMatrix &mat,
                                             SbMatrix &inv) const;
  /**
   * Returns composite matrix that transforms from object space to
   * translation space (the space after scale, rotation, and translation).
   */
  void                getTranslationSpaceMatrix(SbMatrix &mat,
                                                SbMatrix &inv) const;

  /**
   * Convenience function that combines the effects of a matrix
   * transformation into the current transformation stored in this node. This
   * method premultiplies the transformation.
   */
  void                multLeft(const SbMatrix &mat);
  /**
   * Convenience function that combines the effects of a matrix
   * transformation into the current transformation stored in this node. This
   * method postmultiplies the transformation.
   */
  void                multRight(const SbMatrix &mat);

  /**
   * Convenience function that combines the effects of another
   * transformation node into the current transformation stored in this node. This
   * method premultiplies the transformation.
   */
  void                combineLeft(SoTransformation *nodeOnLeft);
  /**
   * Convenience function that combines the effects of another
   * transformation node into the current transformation stored in this node. This
   * method postmultiplies the transformation.
   */
  void                combineRight(SoTransformation *nodeOnRight);

  /**
   * Sets the fields in the node to implement the transformation represented by the
   * given matrix. Note that invalid matrices (such as singular ones) have undefined
   * results.
   */
  void                setMatrix(const SbMatrix &mat);

  /**
   * Changes the center of the transformation to the given point without affecting
   * the overall effect of the transformation.
   */
  void                recenter(const SbVec3f &newCenter);

 SoEXTENDER public:
  virtual void        doAction(SoAction *action);
  virtual void        callback(SoCallbackAction *action);
  virtual void        GLRender(SoGLRenderAction *action);
  virtual void        getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void        getMatrix(SoGetMatrixAction *action);
  virtual void        pick(SoPickAction *action);
  virtual void        getPrimitiveCount(SoGetPrimitiveCountAction *action);

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

 protected:
  virtual ~SoTransform();

};

#endif /* _SO_TRANSFORM_ */

