/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Alan Norton (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/



#ifndef  _SO_VERTEX_PROPERTY
#define  _SO_VERTEX_PROPERTY

#include <Inventor/SbVertexAttributeBinding.h>

#include <Inventor/fields/SoMFUInt32.h>
#include <Inventor/fields/SoMFVec3f.h>
#include <Inventor/fields/SoMFVec2f.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/nodes/SoNode.h>

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

class SoColorPacker;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoVertexProperty
//
//  SoNode class that manages arrays of data for GLVertex Array Extension.
//  Data arrays include: vertices, normals, tangents, colors, texture coordinates.
//  Also has NormalBinding, TangentBinding and MaterialBinding fields
//
//////////////////////////////////////////////////////////////////////////////

/*! \cond PRIVATE */
namespace inventor
{
  namespace cache
  {
    class VertexProperty;
  }
}
/*! \endcond */

/**
 * Vertex property node.
 * 
 * @ingroup PropertyNodes
 * 
 * @DESCRIPTION
 *   This property node is be used to efficiently specify coordinates, normals,
 *   tangents, texture coordinates, colors, transparency values, material binding,
 *   normal and tangent binding for vertex-based shapes, i.e., shapes derived from SoVertexShape.
 *
 *   An SoVertexProperty node can be used as a child of a group node in a scene graph,
 *   in which case the properties it specifies are inherited by subsequent shape
 *   nodes in the graph. It can also be directly referenced using the
 *   @B vertexProperty @b field of a vertex-based shape, bypassing scene graph
 *   inheritance. Direct referencing is the recommended practice.
 *   
 *   When directly referenced by a @B VertexProperty @b field of a vertex-based
 *   shape, the SoVertexProperty node is the most efficient way of specifying
 *   vertex-based shapes. Use of the directly referenced SoVertexProperty node
 *   results in significantly faster scene rendering than scene graph inheritance of
 *   vertex properties, provided all required vertex properties are specified in the
 *   SoVertexProperty node.
 *   
 *   Because the class SoVertexProperty is derived from SoNode, a vertex property node
 *   can be inserted as a child node in a scene graph. When inserted as a node in a
 *   scene graph, the SoVertexProperty node is traversed as any other property node
 *   and the properties it specifies are inherited by subsequent shape nodes in the
 *   scene graph. It specifies the current material, normal and tangent bindings, and
 *   can be used to specify the current 3D coordinates, the current normals, the current
 *   tangents, the current texture coordinates, the current diffuse colors, and the
 *   current transparencies.
 *   
 *   All multiple-valued fields in the SoVertexProperty node are optional. If a field
 *   is not present (i.e. if it has 0 values), then shapes that require the missing
 *   information are required to obtain it from the current traversal state. However,
 *   developers are cautioned that, for optimal performance, the vertex property node
 *   should be referenced as the @B VertexProperty @b field of an SoVertexShape, and
 *   should specify in its fields all values required to render that shape.
 *
 *   Fields:
 *   \par
 *   The various fields in a vertex property node can be used in place of
 *   corresponding fields in other property nodes, as follows:
 *     - The #vertex field contains 3D coordinates, as in the @B point @b field of
 *       an SoCoordinate3 node. @BR @BR
 *     - The #normal field contains normal vectors, as in the @B vector @b field of
 *       the SoNormal node. @BR Normal vectors are used (for example) to compute the
 *       effect of lighting on the shape when lighting is enabled. @BR @BR
 *     - The #tangent field contains tangent vectors. @BR When not using any custom shader,
 *       tangents are used by line shapes to compute the effect of lighting on the shape
 *       when lighting is enabled. See SoLineSet and SoIndexedLineSet for more details. @BR @BR
 *     - The #orderedRGBA field contains packed color+transparency values. @BR Colors are specified in the hexadecimal format
 *       @B 0xrrggbbaa @b, where @B rr @b is the red value (between 00 and 0xFF hex),
 *       @B gg @b is the green value (between 00 and 0xFF hex),
 *       @B bb @b is the blue value (between 00 and 0xFF hex),
 *       @B aa @b is the alpha value (between 00 = transparent and 0xFF = opaque).
 *       Packed colors use less memory than SoMaterial colors and can be sent directly to
 *       the GPU. A packed color value can be queried from an SbColor or SbColorRGBA
 *       object using the getPackedValue() method.
 *       The packed colors are equivalent to an SoPackedColor node, and provide values
 *       for both diffuse color and transparency. @BR @BR
 *     - The #texCoord and #texCoord3 fields contain texture coordinates, as in the
 *        @B point @b field of the SoTextureCoordinate2 and SoTextureCoordinate3 nodes.
 *   \par
 *   If one of these nodes appears higher in the scene graph and setOverride(true) was
 *   called on that node, those values are used instead of the corresponding field of
 *   the vertex property node. @BR
 *   Note: Calling setIgnored(true) on the fields of this node has no effect on
 *   its behavior.
 *
 *   Color:
 *   \par
 *   If setOverride() was called on an SoMaterial node higher in the scene graph
 *   and either the @I diffuseColor@i field or the @I transparency@i field is not
 *   ignored (setIgnored()), then the @I color and transparency@i values from the
 *   SoMaterial node are used instead of the values in the SoVertexProperty. It
 *   is not currently possible to override just the color or just the transparency.
 *
 *   Transparency:
 *   \par
 *   The default transparency algorithm is NO_SORT.  To get a nice
 *   appearance for transparent objects you must change this to another value, for
 *   example, OPAQUE_FIRST or SORTED_PIXEL, using the setTransparencyType method in the
 *   viewer or renderAction class. See SoGLRenderAction for a discussion of transparency
 *   algorithms and rendering order.
 *
 *   Material binding:
 *   \par
 *   The #materialBinding field replaces the @B value @b field of the
 *   SoMaterialBinding node. The #materialBinding field in a directly
 *   referenced SoVertexProperty node has no effect unless there is a nonempty
 *   #orderedRGBA field, in which case the material binding specifies the
 *   assignment of diffuse colors and alpha values to the shape.
 *   
 *   Normal binding:
 *   \par
 *   The #normalBinding field replaces the @B value @b field of the
 *   SoNormalBinding node. The #normalBinding field of a directly referenced
 *   SoVertexProperty node has no effect unless there is a nonempty #normal
 *   field, in which case the normal binding specifies the assignment of normal
 *   vectors to the shape.
 *
 *   Tangent binding:
 *   \par
 *   The #tangentBinding field of a directly referenced SoVertexProperty node
 *   has no effect unless there is a nonempty #tangent field, in which case
 *   the tangent binding specifies the assignment of tangent vectors to the shape.
 * 
 * @FILE_FORMAT_DEFAULT
 *    VertexProperty {
 *    @TABLE_FILE_FORMAT
 *       @TR vertex            @TD [ ]
 *       @TR normal            @TD [ ]
 *       @TR tangent           @TD [ ]
 *       @TR texCoord          @TD [ ]
 *       @TR texCoord3         @TD [ ]
 *       @TR orderedRGBA       @TD [ ]
 *       @TR materialBinding   @TD OVERALL
 *       @TR normalBinding     @TD PER_VERTEX_INDEXED
 *       @TR tangentBinding    @TD PER_VERTEX_INDEXED
 *       @TR forceSending   @TD FALSE
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction,  SoPickAction @BR
 *        When traversed in a scene graph, sets coordinates, normals, tangents, texture
 *        coordinates, diffuse colors, transparency, normal binding, tangent binding and material binding
 *        in current traversal state. If not traversed, has no effect on current traversal
 *        state associated with action. The normalBinding field has no effect if there are
 *        no normals. The tangentBinding field has no effect if there are no tangents.
 *        The materialBinding has no effect if there are no packed colors.
 *        Sets: SoCoordinateElement, SoNormalElement, SoNormalBindingElement, SoMaterialElement
 *              SoMaterialBindingElement, SoTextureCoordinateElement, SoTangentElement,
 *              SoTangentBindingElement
 * 
 *    SoGetBoundingBoxAction @BR
 *        When traversed in a scene graph, sets coordinates in current traversal state.
 *        If not traversed, has no effect on current traversal state associated with
 *        action. Sets: SoCoordinateElement
 * 
 * 
 * @SEE_ALSO
 *    SoIndexedTriangleStripSet,
 *    SoIndexedFaceSet,
 *    SoIndexedLineSet,
 *    SoTriangleStripSet,
 *    SoLineSet,
 *    SoFaceSet,
 *    SoPointSet,
 *    SoQuadMesh,
 *    SoVertexShape,
 *    SoIndexedShape,
 *    SoNonIndexedShape
 * 
 * 
 */ 

class INVENTOR_API SoVertexProperty : public SoNode {

  SO_NODE_HEADER(SoVertexProperty);

 public:

  /**
   * Vertex coordinate(s). Default is empty.
   */
  SoMFVec3f vertex;
  /**
   * 2D texture coordinate(s). Default is empty.
   */
  SoMFVec2f texCoord;
  /**
   * 3D texture coordinate(s). Default is empty.
   */
  SoMFVec3f texCoord3;

  /**
   * Forces the texture coordinates to be sent to the GPU even if no texture is bound.
   */
  SoSFBool forceSending; 

  /**
   * Normal vector(s). Default is empty.
   */
  SoMFVec3f normal;

  /**
   * Tangent vector(s). Default is empty.
   * @FIELD_SINCE_OIV 10.2
   */
  SoMFVec3f tangent;

  /**
   * Normal binding.
   * @useenum{Binding}. Default is PER_VERTEX_INDEXED.
   */
  SoSFEnum normalBinding;

  /**
   * Tangent binding.
   * @useenum{Binding}. Default is PER_VERTEX_INDEXED.
   * @FIELD_SINCE_OIV 10.2
   */
  SoSFEnum tangentBinding;

  /**
   * Packed color(s), including transparencies. Default is empty.
   */
  SoMFUInt32 orderedRGBA;

  /**
   * Material binding.
   * @useenum{Binding}. Default is OVERALL.
   */
  SoSFEnum materialBinding;

  /** Binding.
   *  Note: The meaning of some binding values depends on the shape class.
   *  For example, PER_FACE and PER_PART are the same for some shapes. */
  enum Binding {
    /**
     *  Whole object has same material/normal/tangent
     */
    OVERALL = SbVertexAttributeBinding::OVERALL,
    /**
     *  One material/normal/tangent for each part of object
     */
    PER_PART = SbVertexAttributeBinding::PER_PART,
    /**
     *  One material/normal/tangent for each part, indexed
     */
    PER_PART_INDEXED = SbVertexAttributeBinding::PER_PART_INDEXED,
    /**
     *  One material/normal/tangent for each face of object
     */
    PER_FACE = SbVertexAttributeBinding::PER_FACE,
    /**
     *  One material/normal/tangent for each face, indexed
     */
    PER_FACE_INDEXED = SbVertexAttributeBinding::PER_FACE_INDEXED,
    /**
     *  One material/normal/tangent for each vertex of object
     */
    PER_VERTEX = SbVertexAttributeBinding::PER_VERTEX,
    /**
     *  One material/normal/tangent for each vertex, indexed
     */
    PER_VERTEX_INDEXED = SbVertexAttributeBinding::PER_VERTEX_INDEXED
  };

  /**
   * Creates an SoVertexProperty node with default settings.
   */
  SoVertexProperty();

  /**
   * Set the state of the override field.
   * see SoNode::setOverride doc.
   */
  inline virtual void setOverride(const SbBool state)
  { override.setValue(state); }

  /**
   * Returns the state of the override field.
   */
  inline virtual SbBool isOverride() const
  { return override.getValue(); }

 SoEXTENDER public:
  virtual void doAction(SoAction *action);
  virtual void GLRender(SoGLRenderAction *action);
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void callback(SoCallbackAction *action);
  virtual void pick(SoPickAction *action);
  virtual void getPrimitiveCount(SoGetPrimitiveCountAction *action);

  SbBool isTransparent();

 SoINTERNAL public:

  /** Helper structure used to return status of various vertex properties 
   * A bit set to true mean the attribute is valid
   */
  class ValidityBits
  { 
	public:
		/** WARNING: if a member is added, NUM_FIELDS must be updated */
		enum { NUM_FIELDS = 6 };

		union Data
		{
		    /** [OIV-WRAPPER-NO-WRAP] */
			struct foo
			{
				bool coordinates : 1;
				bool normals : 1;
				bool tangents : 1;
				bool colors : 1;
				bool textureCoords : 1;
				bool attributes : 1;
				bool indices : 1;
        bool picks : 1;
			} attr;

			/** 
			 * Allow to initialize/test all members in one shot 
			 * [OIV-WRAPPER-NO-WRAP]
			 */
			unsigned char bits;

		} m_data;

		/** Initialize as invalidate */
		ValidityBits()
		{
			invalidate();
		}

		/** Mark all properties as valid */
		inline void reset()
		{
		  m_data.bits = 0xFF;
		}

		/** Mark all properties as invalid */
		inline void invalidate()
		{
			m_data.bits = 0;
		}

		/** Return true if everything is invalid */
		inline bool isInvalid() const
		{
      return m_data.bits == 0;
		}

		/** Return true if the bits are valid */
		inline bool isValid() const
		{
      return (m_data.bits&((1 << NUM_FIELDS) - 1)) == ((1 << NUM_FIELDS) - 1);
		}
  };

  static void         initClass();
  static void         exitClass();

  SoSFBool override;

  // set NormalElement, LazyElement and TextureElement
  void setupElements(SoState* state);

  SoColorPacker *getColorPacker2() const;
  enum Transparent {
    YES,
    NO,
    UNKNOWN
  };

  inline uint32_t getVextexTimeStamp() const
  { return m_vertexTimeStamp; }

  // check for transparency when field changes.
  virtual void        notify(SoNotList *list);

 protected:
  virtual ~SoVertexProperty();

 private:
  // store whether transparent or not
  Transparent transparent;
  uint32_t m_vertexTimeStamp;
  mutable SoColorPacker *m_colorPacker2;
};


#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif /* _SO_VERTEX_PROPERTY */

