/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : David Mott (MMM yyyy)
** Modified by : Jim Kent (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_WWWANCHOR_
#define  _SO_WWWANCHOR_

#include <Inventor/fields/SoSFString.h>
#include <Inventor/fields/SoSFFilePathString.h>
#include <Inventor/nodes/SoLocateHighlight.h>

class SoWWWAnchor;

/** This is the format for callbacks during anchor activation (left mouse click)
 * and highlight (move over the anchor).
 *
 * @memberof SoWWWAnchor
 *
 * The URL for the anchor is passed, with additional information as described by the
 * map field tagged onto the URL.
 * [OIV-WRAPPER NAME{URLCallback}]
 */
typedef void SoWWWAnchorCB(const SbString &url, void *userData, SoWWWAnchor *node);


//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoWWWAnchor
//
//////////////////////////////////////////////////////////////////////////////

/**
 * @VSGEXT Separator group node with a URL hyperlink.
 * 
 * @ingroup GroupNodes
 * 
 * @DESCRIPTION
 *   This node defines a separator group which has a URL (Universal Resource Locator)
 *   hyperlink. When clicked on, this node invokes an application \if_dotnet delegate \else callback \endif to fetch
 *   the URL. The application may fetch the data itself, or call a web browser to load the URL.
 * 
 * @FILE_FORMAT_DEFAULT
 *    WWWAnchor {
 *    @TABLE_FILE_FORMAT
 *       @TR boundingBoxCaching   @TD AUTO
 *       @TR renderCulling        @TD AUTO
 *       @TR pickCulling          @TD AUTO
 *       @TR fastEditing          @TD DISABLE
 *       @TR mode                 @TD AUTO
 *       @TR style                @TD EMISSIVE
 *       @TR color                @TD 0.3 0.3 0.3
 *       @TR name                 @TD "<Undefined URL>"
 *       @TR description          @TD ""
 *       @TR map                  @TD NONE
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoHandleEventAction  @BR
 *        When the left mouse is clicked over a child of this anchor node, the
 *        application is called back to fetch the URL (see setFetchURLCallBack()).
 * 
 * 
 * @SEE_ALSO
 *    SoWWWInline
 * 
 * 
 */
class INVENTOR_API SoWWWAnchor : public SoLocateHighlight {

  SO_NODE_HEADER(SoWWWAnchor);

 public:
  /**
   * Creates an anchor node with default settings.
   */
  SoWWWAnchor();
 
    /** Mapping */
  enum Mapping { 
    /** Do no add information to the URL  */
    NONE,
    /** Add object-space coordinates to URL */
    POINT
  };

  /**
   * URL which the application will be called back to fetch when this node is
   * activated by a left mouse click (e.g., "https://www.openinventor.com/").
   * 
   */
  SoSFFilePathString  name;
  /**
   * A description of the URL which may make more sense to users than the URL itself
   * (e.g., "The VSG Home Page").
   * 
   */
  SoSFString  description;
  /**
   * Specifies what additional information should be added to the end of the URL when
   * passed back to the application for fetching. If set to POINT, then the x,y,z
   * location of the object space intersection point when the mouse was clicked will
   * be suffixed to the URL. Since the comma character does not transmit through some
   * web browsers, it is written in hex form (e.g., if the intersection point is
   * 17,4.5,1 the URL passed to the application \if_dotnet delegate \else callback \endif would be
   * "https://www.openinventor.com/?17\%2c4.5\%2c1").
   * @useenum{Mapping}. Default is NONE.
   */
  SoSFEnum map;
    
  /**
   * If the name field contains a relative URL (e.g., "foo.wrl" instead of
   * "http://bogus.com/foo.wrl"), the anchor cannot resolve the URL reference. This
   * method allows the application to tell the anchor what its full URL should be.
   * getFullURLName() returns the fullURL set here, or if not set, returns the
   * contents of the name field.
   */
  void setFullURLName(const SbString &url) { fullURL = url; }
  /**
   * Returns the full URL set by setFullURLName(), or if not set, 
   * returns the contents of the name field.
   */
  const SbString & getFullURLName();

  /**
   * Application callback invoked when the mouse is clicked on an anchor so that the
   * application can fetch the URL.
   */
  static void setFetchURLCallBack( SoWWWAnchorCB *f, void *userData); 

  /**
   * Application callbacks invoked when the mouse is moved over an anchor so that the
   * app can provide feedback such as changing the cursor, or displaying the anchor
   * description and name.
   */
  static void setHighlightURLCallBack( SoWWWAnchorCB *f, void *userData);

  // Need to enable or disable event handling so the geometry
  // contained therein can be picked, manipulated, etc.
  static void enableEvents(SbBool OnOffFlag);

 SoEXTENDER public:
  // Override handleEvent to look for left-mouse clicks, to do a
  // pick:
  virtual void handleEvent(SoHandleEventAction *action);

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

 protected:
  virtual ~SoWWWAnchor();
    
  // redefine this to also invoke the app callbacks...
  // This is called when the locate highlight status of this node changes.
  virtual void redrawHighlighted(SoAction *act, SbBool isNowHighlighting);

  static SoWWWAnchorCB           *fetchURLcb;
  static void                    *fetchURLdata;
  static SoWWWAnchorCB           *highlightURLcb;
  static void                    *highlightURLdata;
  static SbString                emptyString;
  
 private:  
  SbString fullURL;
  static SbBool enableEventsFlag;

};

#endif /* _SO_WWWANCHOR_ */

