/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Nick Thompson (MMM yyyy)
** Modified by : Paul Strauss (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_SENSOR_
#define  _SO_SENSOR_

#include <Inventor/SbBasic.h>
#include <Inventor/threads/SbThreadSpinlock.h>

class SoField;
class SoMField;
class SoSensor;

/** This typedef defines the calling sequence for all callbacks from sensors
 *
 * @memberof SoSensor
 *
 * [OIV-WRAPPER NAME{SensorCB}]
 */
typedef void SoSensorCB(void *data, SoSensor *sensor);

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoSensor
//
//  Abstract base class for all sensors. Defines basic callback
//  definition (explicit or in constructor) and scheduling and
//  triggering methods.
//
//////////////////////////////////////////////////////////////////////////////
/**
 * Abstract base class for Open Inventor sensors.
 * 
 * @ingroup sensors
 * 
 * @DESCRIPTION
 *   Sensors detect changes either to time or to Open Inventor objects in a scene
 *   graph, and call a user-defined \if_dotnet delegate \else callback \endif function. Sensors are @B scheduled @b
 *   when the thing they are attached to changes, and sometime after they are
 *   scheduled they are @B triggered @b, calling the user's \if_dotnet delegate \else callback \endif function.
 * 
 * @SEE_ALSO
 *    SoAlarmSensor,
 *    SoDataSensor,
 *    SoFieldSensor,
 *    SoIdleSensor,
 *    SoNodeSensor,
 *    SoPathSensor
 * 
 * 
 */
class INVENTOR_API SoSensor {

 public:

  /**
   * Constructor.
   */
  SoSensor()
    {
      func = NULL;
      funcData = NULL;
#ifdef _WIN32
      dwThreadId=GetCurrentThreadId();
#endif
      m_schedule = FALSE;
    }
  /**
   * Constructor that takes standard \if_dotnet delegate \else callback \endif function and data.
   * [OIV-WRAPPER-NO-WRAP]
   */
  SoSensor(SoSensorCB *f, void *d)
    {
      func = f;
      funcData = d;
#ifdef _WIN32
      dwThreadId=GetCurrentThreadId();
#endif
      m_schedule = FALSE;
    }

  // Virtual destructor so that subclasses are deleted properly
#ifndef HIDDEN_FROM_DOC
  virtual ~SoSensor();
#endif // HIDDEN_FROM_DOC

  /**
   * [OIV-WRAPPER EVENT_NAME{Action}]
   */
  inline void setFunction(SoSensorCB *f, void *userData);

  /**
   * Sets the callback function that is called when the sensor is triggered. The
   * function must take two arguments - user-supplied callback data (of type void *)
   * and a pointer to the sensor that is triggering the function (of type SoSensor
   * *).
   * [OIV-WRAPPER-NO-WRAP]
   */
  void                setFunction(SoSensorCB *f)
    {
      func = f;
#ifdef _WIN32
      dwThreadId=GetCurrentThreadId();
#endif
      
    }
    
  /**
   * Sets the callback data passed to the callback function.
   * [OIV-WRAPPER-NO-WRAP]
   */
  void                setData(void *d)                { funcData = d; }

  /**
   * Returns the callback that will be called when the sensor is triggered.
   * [OIV-WRAPPER-NO-WRAP]
   */
  SoSensorCB *        getFunction() const             { return func; }
    
  /**
   * Returns the user-supplied pointer that will be passed to the callback function.
   * [OIV-WRAPPER-NO-WRAP]
   */
  void *              getData() const                 { return funcData; }

  // Schedules the sensor for triggering at the appropriate time
  virtual void        schedule() = 0;

  // Unschedules sensor to keep it from being triggered
  virtual void        unschedule() = 0;

  /**
   * Returns TRUE if this sensor has been scheduled and is waiting in a sensor queue
   * to be triggered. Sensors are removed from the queue before their \if_dotnet delegate \else callback \endif
   * function is triggered.
   */
  virtual SbBool      isScheduled() const
  { return m_schedule; }

 SoINTERNAL public:

  // take critical section ownership
  inline void lock()
  { m_mutex.lock(); }

  // release critical section ownership
  inline void unlock()
  { m_mutex.unlock(); }


  // This must be called only by the SensorManager when it add/remove the sensor from its queues.
  // with the queue locked (in Multithread) to be ensure that this flag reflect the real state 
  // of the sensor.
  inline void setScheduleFlag(SbBool flag)
  { m_schedule = flag; };

#ifdef _WIN32
  DWORD getThreadId() { return dwThreadId;};
  void setThreadId(DWORD id) { dwThreadId=id;};
#endif

  // Initialize static members, etc.
  static void         initClass();

  // Triggers the sensor, calling its callback function
  virtual void        trigger();

  // This returns TRUE if this sensor should precede sensor s in
  // whichever queue this sensor would be in.
  virtual SbBool      isBefore(const SoSensor *s) const = 0;

  // Sets/returns the next sensor in whichever queue the sensor is in
  void                setNextInQueue(SoSensor *next) { nextInQueue = next; }
  SoSensor *          getNextInQueue() const         { return nextInQueue; }

 protected:
  SoSensorCB *        func;           // Callback function
  void *              funcData;       // Data to pass to callback

#ifdef _WIN32
  DWORD           dwThreadId; // thread to which this sensor belongs

 protected:
#else
 private:
#endif
  SoSensor            *nextInQueue;   // Next sensor in queue
 private:
  SbBool m_schedule;  // Whether sensor is scheduled in a queue or not.
    SbThreadSpinlock m_mutex;
};

void SoSensor::setFunction(SoSensorCB *f, void *userData)
{ 
  setFunction(f);
  setData(userData);
}

#endif /* _SO_SENSOR_ */


