/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Nick Thompson (MMM yyyy)
** Modified by : Paul Strauss (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_TIMER_SENSOR_
#define  _SO_TIMER_SENSOR_

#include <Inventor/sensors/SoTimerQueueSensor.h>

/**
 * Sensor that triggers repeatedly at regular intervals.
 *
 * @ingroup sensors
 *
 * @DESCRIPTION
 *   Timer sensors trigger their function at regular intervals.
 *   A base time may be specified, which causes the sensor to be triggered at multiples
 *   of the interval starting from the base time. The base time is an absolute time, as returned by
 *   SbTime::getTimeOfDay(). For example, a timer might be set up to call its 
 *   function every second on the second by setting the base time to SbTime@B
 *   (0.0) @b and the interval to SbTime@B (1.0) @b. Timers are guaranteed to be triggered only once
 *   when the timer queue is processed, so if the application only processes the timer queue once
 *   every 5 seconds (because it is busy doing something else) the once-a-second sensor's
 *   function will only be called once every 5 seconds.
 *
 *   If no base time is set, then the function will be
 *   rescheduled from the present time using the interval value.
 *
 *   When a base time has been specified, SoTimers always schedule themselves to be triggered the
 *   next multiple of the interval time after the base time. For example, if the once-a-second
 *   sensor is triggered at time 2.9 (because the application way busy doing something at time 2.0
 *   and didn't get around to processing the sensor queue for a while) it will reschedule itself to
 *   go off at time 3.0, not at time 3.9.  On the other hand, if the base time had never been set,
 *   then the sensor would be scheduled for time 3.9.
 *
 * @SEE_ALSO
 *    SoOneShotSensor,
 *    SoAlarmSensor,
 *    SoTimerQueueSensor,
 *    SbTime
 *
 *
 */
class INVENTOR_API SoTimerSensor : public SoTimerQueueSensor {

 public:
  /**
   * Default constructor.
   */
  SoTimerSensor();
  /**
   * Constructor that takes the function and data to be
   * called when the sensor is triggered.
   * [OIVJAVA-WRAPPER CUSTOM_CODE]
   */
  SoTimerSensor(SoSensorCB *func, void *data);

  /**
   * Destroys the sensor, freeing up any memory associated with it after unscheduling
   * it.
   */
  virtual ~SoTimerSensor();

  /**
   * Sets the base time. The default base time is the time when
   * the sensor is scheduled or rescheduled.
   */
  void                setBaseTime(const SbTime &base)  { baseTime = base;
  baseTimeSet = TRUE;}
  /**
   * Sets the interval. The default interval is 1/30th of a
   * second.
   */
  void                setInterval(const SbTime &intvl) { interval = intvl; }
  /**
   * Gets the base time.
   */
  const SbTime &      getBaseTime() const              { return baseTime; }
  /**
   * Gets the interval.
   */
  const SbTime &      getInterval() const              { return interval;  }

  // Overrides the regular schedule() method because we have to set
  // up the trigger time first.
  virtual void        schedule();

  // Overrides the regular unschedule() because the timer could be
  // in either the timer queue or the waiting-to-be-rescheduled
  // queue, depending on the state of the 'triggering' flag.
  virtual void        unschedule();

 SoINTERNAL public:

  // Reschedule this timer relative to the given time
  void                reschedule(const SbTime &);

protected:
  // Triggers the sensor, calling its callback function. This
  // overrides the method in SoSensor because it has to reschedule
  // itself for the next interval after triggering.
  virtual void        trigger();

  SbBool              triggering;     // True if in trigger() code

 private:
  SbTime              baseTime;       // Base time for scheduling triggering
  SbTime              interval;       // Interval between triggering
  SbBool              baseTimeSet;    // Whether base time was set by user

};

#endif  /* _SO_TIMER_SENSOR_ */

