/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Feb 2008)
**=======================================================================*/

#ifndef _SB_THREAD_AUTO_LOCK_H_
#define _SB_THREAD_AUTO_LOCK_H_

#include <Inventor/SbBase.h>
#include <Inventor/threads/SbThreadMutex.h>

class SbThreadRWMutex;

/**
 * @VSGEXT Provides reliable unlocking of a mutex.
 * 
 * @ingroup Threads
 * 
 * @DESCRIPTION
 *   This class provides reliable unlocking of a mutex.
 *   
 *   An SbThreadMutex object must be created separately. The address of the mutex
 *   object is passed to the constructor, which locks the mutex and saves a reference
 *   to the mutex object. The autolock object should be created on the stack, so each
 *   thread will have its own autolock. The destructor automatically unlocks the
 *   mutex. 
 *   
 *   Because the autolock object is created on the stack, it avoids the need to
 *   explicitly unlock the mutex at every point of exit from the function (a common
 *   source of errors). The mutex will be unlocked when the autolock goes out of
 *   scope.
 *   
 *   Example:
 *   \code
 *      // myClass contains a Mutex object named m_mutex
 *      void myClass::someFunction() 
 *      {
 *         // m_mutex is a ptr to an existing SbThreadMutex object
 *         SbThreadAutoLock dummyName( m_mutex );
 *         ... other code ...
 *      } // mutex automatically unlocked
 *   \endcode
 *    
 * @SEE_ALSO
 *    SbThread,
 *    SbThreadAutoReadLock,
 *    SbThreadAutoWriteLock,
 *    SbThreadBarrier,
 *    SbThreadMutex,
 *    SbThreadRWMutex
 * 
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
class INVENTORBASE_API SbThreadAutoLock {
 public:
  /**
   * Creates a mutex "autolock" object and automatically calls the mutex lock method.
   * The mutex will be automatically unlocked when the autolock object is destroyed.
   */
  SbThreadAutoLock(SbThreadMutex *mutex)
  {
    m_mutex = mutex;
    if ( m_mutex )
      m_mutex->lock();
  }

  SbThreadAutoLock( SbThreadMutex &mutex )
  {
    m_mutex = &mutex;
    m_mutex->lock();
  }

#ifndef HIDDEN_FROM_DOC
  ~SbThreadAutoLock()
  {
    if ( m_mutex )
      m_mutex->unlock();
  }
#endif // HIDDEN_FROM_DOC

 private:
  SbThreadMutex *m_mutex;
};

/**
 * @VSGEXT Provides reliable unlocking of a read-only mutex.
 * 
 * @ingroup Threads
 * 
 * @DESCRIPTION
 *   This class provides reliable unlocking of a read-only mutex.
 *   
 *   An SbThreadRWMutex object must be created separately. The address of the mutex
 *   object is passed to the constructor, which locks the mutex and saves a reference
 *   to the mutex object. The autolock object should be created on the stack, so each
 *   thread will have its own autolock. The destructor automatically unlocks the
 *   mutex.
 *   
 *   Because the autolock object is created on the stack, it avoids the need to
 *   explicitly unlock the mutex at every point of exit from the function (a common
 *   source of errors). The mutex will be unlocked when the autolock goes out of
 *   scope.
 *   
 *   The unlock() and relock() methods are provided for complex cases where
 *   the lock needs to be temporarily released. If the autolock is explicitly
 *   unlocked, the destructor does nothing.
 *   
 *   Example:
 *   \code 
 *      myClass contains an RWMutex object named m_mutex
 *      void myClass::someFunction()
 *      {
 *         SbThreadAutoReadLock( m_mutex );
 *         ... other code ...
 *      } // mutex automatically unlocked
 *   \endcode
 * 
 * @SEE_ALSO
 *    SbThread,
 *    SbThreadAutoLock,
 *    SbThreadAutoWriteLock,
 *    SbThreadBarrier,
 *    SbThreadMutex,
 *    SbThreadRWMutex
 * 
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
class INVENTORBASE_API SbThreadAutoReadLock {
 public:
  /**
   * Creates a mutex "autolock" object and automatically calls the mutex readlock
   * method. The mutex will be automatically unlocked when the autolock object is
   * destroyed.
   */
  SbThreadAutoReadLock( SbThreadRWMutex *mutex );
  SbThreadAutoReadLock( SbThreadRWMutex &mutex );

#ifndef HIDDEN_FROM_DOC
  /**
   * Destructor.
   */
  ~SbThreadAutoReadLock();
#endif

  /**
   * Explicitly unlock the associated mutex. Do not unlock or lock the
   * mutex directly.
   */
  int unlock();
  /**
   * Explicitly relock the associated mutex. Do not unlock or lock the
   * mutex directly.
   */
  int relock();

 private:
  SbThreadRWMutex *m_mutex;
  SbBool           m_isLocked;

};

/**
 * @VSGEXT Provides reliable unlocking of a write-only mutex.
 * 
 * @ingroup Threads
 * 
 * @DESCRIPTION
 *   This class provides reliable unlocking of a write-only mutex.
 *   
 *   An SbThreadRWMutex object must be created separately. The address of the mutex
 *   object is passed to the constructor, which locks the mutex and saves a reference
 *   to the mutex object. The autolock object should be created on the stack, so each
 *   thread will have its own autolock. The destructor automatically unlocks the
 *   mutex. 
 *   
 *   Because the autolock object is created on the stack, it avoids the need to
 *   explicitly unlock the mutex at every point of exit from the function (a common
 *   source of errors). The mutex will be unlocked when the autolock goes out of
 *   scope.
 *   
 *   The unlock() and relock() methods are provided for complex cases where
 *   the lock needs to be temporarily released. If the autolock is explicitly
 *   unlocked, the destructor does nothing. 
 *   
 *   Example:
 *   \code
 *      // myClass contains an RWMutex object named m_mutex
 *      void myClass::someFunction()
 *      {
 *         // m_mutex is a ptr to an existing SbThreadRWMutex object
 *         SbThreadAutoWriteLock dummyName( m_mutex );
 *         ... other code ...
 *      } // mutex automatically unlocked
 *   \endcode
 *    
 * 
 * @SEE_ALSO
 *    SbThread,
 *    SbThreadAutoLock,
 *    SbThreadAutoReadLock,
 *    SbThreadBarrier,
 *    SbThreadMutex,
 *    SbThreadRWMutex
 * 
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
class INVENTORBASE_API SbThreadAutoWriteLock {
 public:
  /**
   * Creates a mutex "autolock" object and automatically calls the mutex writelock
   * method. The mutex will be automatically unlocked when the autolock object is
   * destroyed.
   */
  SbThreadAutoWriteLock( SbThreadRWMutex *mutex );
  SbThreadAutoWriteLock( SbThreadRWMutex &mutex );

#ifndef HIDDEN_FROM_DOC
  /**
   * Destructor.
   */
  ~SbThreadAutoWriteLock();
#endif

  /**
   * Explicitly unlock the associated mutex. Do not unlock or lock the
   * mutex directly.
   */
  int unlock();
  /**
   * Explicitly relock the associated mutex. Do not unlock or lock the
   * mutex directly.
   */
  int relock();

 private:
  SbThreadRWMutex *m_mutex;
  SbBool           m_isLocked;

};

#endif //_SB_THREAD_AUTO_LOCK_H_


