/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Nicolas DAGUISE (Nov 2008)
**=======================================================================*/



#ifndef _SB_THREAD_STORAGE_H_
#define _SB_THREAD_STORAGE_H_

#include <Inventor/SoInventorBase.h>

#include <Inventor/threads/SbThreadStorageBase.h>
#include <Inventor/STL/functional>

/**
 * @VSGEXT Template class for per instance thread Local Storage.
 *
 * This class provides a way to declare a class member as a thread Specific variables.
 *
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
template <class T>
class SbThreadStorage : public SbThreadStorageBase
{
public:
  /**
   * Default constructor.
   */
  SbThreadStorage<T>() { }

  /*
   * Copy constructor.
   */
  SbThreadStorage<T>( const T t )
  {
    Register( t );
  }

  /**
   * Storage destructor:
   * Find current storage in the global storage map,
   * Delete thread dependent entry,
   * And remove each entry.
   */
  virtual ~SbThreadStorage()
  { erase(); }

  /**
   * Override operator=
   * Set value of the storage for current thread.
   * If current thread storage was not existing, create it.
   */
  SbThreadStorage<T>& operator=(T t)
  {
    Register( t );
    return *this;
  }

  /**
   * Override operator->
   * Gets value of the storage for current thread.
   * Returns NULL if the storage does not exist for current thread.
   */
  T operator->() const
  {
#if defined(_DEBUG)
    T t = static_cast<T>(getValue());
    if ( t == NULL && SoInventorBase::checkMultithread() )
      fprintf( stderr, "MT ERROR: in ::operator-> , object not initialized for current thread." );
    return t;
#else
    return static_cast<T>(getValue());
#endif
  }

  /**
   * Override operator T*() 
   * Destroying objects from delete operator is not allowed.
   */
  T operator*() const 
  {
    return static_cast<T>(getValue());
  }

  /**
   * Override operator bool() 
   * Comparing templated objects work as comparing direct object.
   * for example, for: SbThreadStorage< SoCache* > m_cache;
   * doing:  m_cache == NULL
   * run as doing 
   */
  operator bool() const
  {
    return ((static_cast<T>(getValue())) != NULL);
  }

  /** Calls the passed function to all registered threads object. */
  template< typename Func>
  void call( Func f )
  {
    s_threadStorageGlobalMutex.lock();
    SbThreadStorageGlobalList::iterator it_i = s_threadStorageGlobalList.begin();
    while(it_i != s_threadStorageGlobalList.end())
    {
      // For each registered map, search for this threadStorage and apply function call.
      void* value=get(it_i);
      if (value!=NULL)
        std::mem_fn(f)(static_cast<T>(value));
      ++it_i;
    }
    s_threadStorageGlobalMutex.unlock();
  }

  /** Calls the passed function to all registered threads object. */
  template< typename Func, typename Param>
  void call( Func f, Param p )
  {
    s_threadStorageGlobalMutex.lock();
    SbThreadStorageGlobalList::iterator it_i = s_threadStorageGlobalList.begin();
    while(it_i != s_threadStorageGlobalList.end())
    {
      // For each registered map, search for this threadStorage and apply function call.
      void* value=get(it_i);
      if (value!=NULL)
          std::bind(std::mem_fn((Func)(f)), std::placeholders::_1, p)(static_cast<T>(value));
      ++it_i;
    }
    s_threadStorageGlobalMutex.unlock();
  }


protected:
  /**
  * release memory of a single value.
  * by default object must have a ->unref() method to be able to be deleted correctly.
  */
  virtual void deleteStorage(void* p)
  {
    T val = (static_cast<T>(p));
    if (val)
      val->unref();
  }

private:
  // Disable several operators
  void* operator new[] (size_t) throw() { return 0; }
  void operator delete[] (void*) { }
  void* operator new( size_t, char ) throw() { return 0; }
};

/**
 * @VSGEXT Template class specialization for bool type instance thread Local Storage.
 *
 * This class provides a way to declare a class member of type bool as a thread Specific variables.
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
template <>
class SbThreadStorage<bool> : public SbThreadStorageBase
{
public:
  /** Destructor */
  virtual ~SbThreadStorage<bool>()
  { erase(); }

  /**
   * Override default <<= operator
   * Set the specified value for the current thread.
   */
  SbThreadStorage<bool>& operator <<=(bool t)
  {
    void* pt = reinterpret_cast<void*>(static_cast<uintptr_t>(t));
    Register( pt );
    return (*this);
  }

  /**
   * Override default >>= operator
   * Set the specified value for all threads.
   */
  SbThreadStorage<bool>& operator >>=(bool t)
  {
    void *pt = reinterpret_cast<void*>(static_cast<uintptr_t>(t));
    Register( pt );
    setAll(pt);
    return *this;
  }

  /**
   * Redefine operator bool to return the value itself
   */
  operator bool() const
  { return ((static_cast<bool>(getValue())) == true); }

protected:
  /** redefined to avoid unref usage wich is no needed for basic type like bool */
  virtual void  deleteStorage(void*)
  {}

private:
  // Disable several operators
  void* operator new[] (size_t) throw() { return 0; }
  void operator delete[] (void*) { }
  void* operator new( size_t , char  ) throw() { return 0; }
  SbThreadStorage<bool>& operator=(bool) { return *this; }
};

#endif //_SB_THREAD_STORAGE_H_

/**/


