/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef LDM_DATA_ACCESS_H
#define LDM_DATA_ACCESS_H

#include <Inventor/SbLinear.h>
#include <Inventor/SbBox.h>
#include <LDM/SoLDMTileID.h>

#include <Inventor/STL/vector>
#include <Inventor/STL/map>

#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoCpuBufferObject.h>

#include <LDM/nodes/SoLDMResourceParameters.h>

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

class SoDataSet;
class SoLDMMediator;
class SoPerfCounter;
class SoPerfCounterManager;
class SoCpuBufferObject;
class SbThread;
class SbThreadBarrier;
class SoLDMAlgorithms;


/**
 *
 * @LDMEXT Volume data access methods.
 *
 * @ingroup LDM
 *
 * @DESCRIPTION
 * The Data Access API provides methods to conveniently extract data from a 
 * volume. This is useful, for example, to extract data for computation, for 
 * segmentation or for display using other primitives. The data is accessible 
 * whether the SoVolumeData is part of a scene graph or not. These methods are 
 * only valid in LDM mode (which is the default mode). The Data Access API
 * automatically takes advantage of LDM multi-threaded data loading when
 * multiple tiles (not already in cache) are needed to complete the request.
 *
 * The Data Access API automatically loads all data required to satisfy the 
 * request and can be invoked asynchronously to allow simultaneous loading and 
 * computation.  The application can request data at any resolution level, e.g. 
 * full resolution (level 0) data, independent of the resolution level currently 
 * being used for rendering.  For some requests, e.g. line and plane, the application 
 * can also specify a subvolume (region of interest) that limits the extent of the data request.
 *
 * SoLDMDataAccess provides the following data requests:
 * - Subvolume: The set of voxels inside a specified subvolume.
 * - Plane: The set of voxels intersecting an arbitrary plane.
 * - Line: The set of voxels intersecting an arbitrary line.
 * - Trace: A column of axis aligned voxels (e.g. a seismic trace).
 * - Polyline: The set of voxels intersecting an arbitrary polyline.
 * - Tile: Direct access to the tile containing a specified voxel.
 *   (This is the best way to query individual voxels. See the method for more information.)
 *
 * Each data request returns information struct specific to the request type. For
 * example, #SoLDMDataAccess::DataInfoBox is returned by the subvolume request. All these structs
 * contain an @I errorFlag @i member containing the result of the request and a
 * @I bufferSize @i member containing the amount of data (in bytes) returned.
 * The application should always check these values.
 * Note that even when errorFlag is CORRECT, bufferSize may be zero,
 * meaning no data was returned.  This happens, for example, if the specified plane is
 * completely outside the volume.
 *
 * The application is responsible for allocating and freeing the memory to store 
 * the requested data. Calling the data request method with a null bufferObject will 
 * return the size of the memory needed, in bytes, in the @I bufferSize @i member of
 * the information struct. Once the buffer is allocated, a second call 
 * will extract the data from the volume. 
 * Since Open Inventor 8.0 we recommend using the methods that return data
 * into an SoBufferObject.  Because this class abstracts access to data stored
 * on different devices it is possible, for example, to conveniently fetch
 * data into CPU memory or directly into GPU memory.
 *
 * Each data request can be either synchronous or asynchronous. Synchronous means 
 * that all the data will be loaded before the function returns. Asynchronous means 
 * that the function will return immediately and you will be notified when the data 
 * is actually available. This allows the application to continue to work while the 
 * data is being loaded, for example to overlap data loading and computation. 
 * Synchronous access is done using the @B getData @b methods.
 *
 * To do asynchronous access, you must create a new class derived from
 * SoLDMDataAccess and implement your own version of the #endRequest method.
 * Begin a request by calling the appropriate @B requestData @b method.
 * This method will return a requestId.
 * - If the requestID is positive, this value will be passed to the #endRequest 
 *   method when the requested data is ready, and can then be used with
 *   the appropriate @B getRequestedData @b method to complete the transaction.
 * - If the requestID is negative, the data is already in memory.
 *   In this case #endRequest will not be called, and -requestID (a positive value)
 *   should be used with the getRequestedData method.
 * The data inside the destBuffer will be aligned as in the original volume. 
 * The info structure contains additional information about the current request
 * including error status, coordinates (for line and polyline requests), tile id
 * (for tile requests), etc.
 *
 * Three different modes are available to control how data is requested from
 * the volume reader when the necessary tiles are not already present in LDM
 * cache memory:
 *
 * - CACHE: When data is requested, the data is copied from the LDM cache (tiles in 
 *   memory) if possible. If any required tiles are not in the LDM cache, LDM will 
 *   load those tiles into cache, then copy the data from the cache. This is the default behavior.
 *
 * - DIRECT: When data is requested, the data is copied from the LDM cache (tiles in 
 *   memory) if possible. If any required tiles are not in the LDM cache, LDM will 
 *   request the data directly from the volume reader without loading any tiles into 
 *   the LDM cache memory. This could potentially be slower than the previous mode, 
 *   but uses less I/O and less memory.  It allows getting data without loading the 
 *   whole tile into memory, for example when the requested data is a slice.
 *
 * - DIRECT_AND_PREFETCH: This mode is a mix of the previous modes. When data is 
 *   requested, the data is copied from the LDM cache if possible. If any required 
 *   tiles are not in the LDM cache, LDM will request the data directly from the 
 *   volume reader to avoid reading complete tiles immediately, but will also schedule 
 *   the relevant tiles to be loaded into the LDM cache memory asynchronously.  
 *   This may improve performance overall if the application is likely to load 
 *   additional data from these tiles.
 *
 * Note that DIRECT access requires that specific methods are implemented in the
 * data set reader, for example readXSliceInTile (see SoLDMReader and SoVolumeReader).
 * The default LDM reader implements these methods for the standard Open Inventor LDM
 * file format. Custom LDM readers may need to be enhanced.
 *
 * @EXAMPLE
 * The following example shows how to extract a subvolume from a given volume.
 * \par
 * \if_cpp
 *   \code
 *     // Create a volume data object and attach to a data file
 *     SoVolumeData* volData = new SoVolumeData();
 *     volData->fileName = "<path-to-filename>";
 *
 *     // Call to get size of data
 *     int      resolution = 0; // Full resolution data
 *     SbBox3i32 subvolume = SbBox3i32(0,0,0,63,63,63);
 *     SoLDMDataAccess& access = volData->getLdmDataAccess();
 *     SoLDMDataAccess::DataInfoBox info = access.getData( resolution, subvolume );
 *
 *     if (info.errorFlag == SoLDMDataAccess::CORRECT) {
 *       // Create and size a buffer to store the data
 *       SoRef<SoCpuBufferObject> buffer = new SoCpuBufferObject;
 *       buffer->setSize( info.bufferSize );
 *
 *       // Call to get the actual data
 *       info = access.getData( resolution, subvolume, buffer.ptr() );
 *
 *       // Access the data then unmap the buffer
 *       unsigned int* pData = (unsigned int*)buffer->map(SoBufferObject::READ_ONLY);
 *       unsigned int value = pData[0];
 *       . . .
 *       buffer->unmap();
 *     }
 *   \endcode
 * See $OIVHOME/examples/source/VolumeViz/examples/getLDMData/asyncGetDataBox.cxx for a
 * more complete example.
 *
 * \endif
 * \if_dotnet
 *   \code
 *     // Create a volume data object and attach to a data file
 *     SoVolumeData volData = new SoVolumeData();
 *     volData.fileName.Value = "<path-to-filename>";
 *
 *     // Call with null pointer to get size of data
 *     int      resolution = 0; // Full resolution data
 *     SbBox3i32 subvolume = new SbBox3i32( 0, 0, 0, 63, 63, 63 );
 *     SoLDMDataAccess access = volData.GetLdmDataAccess();
 *     SoLDMDataAccess.DataInfoBox info = access.GetData( resolution, subvolume );
 *
 *     if (info.ErrorFlag == SoLDMDataAccess.ErrorValues.CORRECT)
 *     {
 *         // Create and size a buffer to store the data
 *         SoCpuBufferObject buffer = new SoCpuBufferObject();
 *         buffer.SetSize((uint)info.BufferSize);
 *
 *         // Call to get the actual data
 *         info = access.GetData(resolution, subvolume, buffer);
 *
 *         // Access the data then unmap the buffer
 *         SbNativeArray<uint> data = (SbNativeArray<uint>)buffer.Map( SoBufferObject.AccessModes.READ_ONLY );
 *         uint value = data[0];
 *         . . .
 *         buffer.Unmap();
 *     }
 *   \endcode
 * \endif
 * \if_java
 *   \code
 *     // Create a volume data object and attach to a data file
 *     SoVolumeData volData = new SoVolumeData();
 *     volData.fileName.setValue( "<path-to-filename>" );
 *
 *     // Call with null to get size of data
 *     int      resolution = 0; // Full resolution data
 *     SbBox3i32 subvolume = new SbBox3i32(0,0,0, 63,63,63);
 *     SoLDMDataAccess access = volData.getLdmDataAccess();
 *     SoLDMDataAccess.DataInfoBox info;
 *     info = access.getData( resolution, subvolume, (SoBufferObject)null );
 *
 *     if (info.errorFlag == SoLDMDataAccess.ErrorValues.CORRECT.getValue()) {
 *         // Create a buffer and set size large enough for returned data
 *         ByteBuffer buffer = ByteBuffer.allocateDirect( info.bufferSize );
 *         buffer.order( ByteOrder.nativeOrder() );
 *
 *         // Call to get the actual data
 *         info = access.getData( resolution, subvolume, buffer );
 *
 *         // Access the data
 *         IntBuffer ibuffer = buffer.asIntBuffer();
 *         int value = ibuffer.get(0);            
 *     }
 *   \endcode
 * \endif
 *
 *
 */

class LDM_API SoLDMDataAccess {

public:

  /**
   * Constructor. @BR
   * Application must call the #setDataSet method before using any other methods of this object.
   * [OIVJAVA-WRAPPER CUSTOM_CODE] 
   */
  SoLDMDataAccess();

  /**
   * Destructor.
   */
  virtual ~SoLDMDataAccess();

  /**
   * Set dataset to fetch data from.
   */
  void setDataSet(SoDataSet* v);

  /**
   * Returns the associated dataset.
   */
  inline SoDataSet* getDataSet();

  /** Error Code values */
  enum ErrorValue
  {
    /** Request was successfully completed. */
    CORRECT,
    /** The specified resolution is lower than the lowest resolution level of the data. */
    INCORRECT_RESOLUTION,
    /** The specified subvolume does not intersect the data volume. */
    INCORRECT_SUBVOLUME,
    /** The specified line does not intersect the specified subvolume. */
    INCORRECT_LINE,
    /** The specified polyline does not intersect the specified subvolume. */
    INCORRECT_POLYLINE,
    /** The specified plane does not intersect the specified subvolume. */
    INCORRECT_PLANE,
    /** The specified position is outside the specified subvolume. */
    INCORRECT_POSITION,
    /** The trace at the specified coordinate does not intersect the specified subvolume */
    INCORRECT_COORDINATE,
    /** No more memory is available. */
    MEMORY_FULL,
    /** Requesting data from an unfinished asynchronous request. */
    REQUEST_NOT_COMPLETED,
    /** Request is unknown. */
    REQUEST_NOT_FOUND,
    /** Application buffer to copy data into is invalid.
     * Most common cases are :
     * - buffer size does not take in account the datasize 
     * - request size parameter is inclusive and app forget to remove (1,1,1) to it.
     */
    INCORRECT_BUFFER
  };

  /** 
  * Specifies the way data is requested from the volume reader when the
  * necessary tiles are not already present in LDM cache memory.
  * See the class description for more info.  See setGetDataMode() method.
  */
  enum GetDataMode
  {
    /** Request data by tile (default) */
    CACHE,
    /** Request data directly.  This means to only request, for example,
     *  the specific slice of data, not the complete tiles. */
    DIRECT,
    /** Same as Direct, but also schedule the relevant tiles to be
     *  loaded for faster future access. */
    DIRECT_AND_PREFETCH
  };

  /**
   * Information about data returned for an arbitrary box (subvolume). 
   * Used with the SoLDMDataAccess::getData(int, const SbBox3i32&, SoBufferObject*) method.
   *
   * The following figure shows the information that is returned.
   * @IMAGE subvolumeboxdetail.jpg
   *
   * [OIVJAVA-WRAPPER-CLASS NO_WRAP]
   */
  struct DataInfoBox
  {
    /** Error indicator. @useenum{ErrorValue}. 
     *  Note that even when errorFlag is CORRECT, bufferSize may be zero meaning no data was returned. 
     */
    ErrorValue errorFlag;
    /** Size of the requested data in bytes. Used by the application to allocate buffer.*/
    int64_t       bufferSize;
    /** Dimensions of the 3D data (in voxels). It depends on the resolution.
     * It is equal to the subvolume if resolution = 0.
     */
    SbVec3i32    bufferDimension;
  };

  /**
   * Information about data returned for an arbitrary trace (column of voxels).
   * Used with the SoLDMDataAccess::getData(int, const SbBox3i32&, const SbVec2i32, SoBufferObject*) method.
   *
   * The following figure shows the information that is returned.
   * @IMAGE subvolumetracedetail.jpg
   *
   * [OIVJAVA-WRAPPER-CLASS NO_WRAP]
   */
  struct DataInfoTrace
  {
    /** Error indicator. @useenum{ErrorValue}. 
     *  Note that even when errorFlag is CORRECT, bufferSize may be zero meaning no data was returned. 
     */
    ErrorValue errorFlag;
    /** Size of the requested data in bytes. Used by the application to allocate buffer.*/
    int64_t        bufferSize;
    /** Dimension of the 1D array of data (the number of values returned). 
     *  The dimension depends on the requested resolution level. */
    int        bufferDimension;
  };

  /**
   * Information about data returned for an arbitrary line.
   * Used with the SoLDMDataAccess::getData(int, const SbBox3i32&, const SbLine&, SoBufferObject*) method.
   *
   * The following figure shows the information that is returned.
   * @IMAGE subvolumelinedetail.jpg
   *
   * [OIVJAVA-WRAPPER-CLASS NO_WRAP]
   */
  struct DataInfoLine
  {
    /** Error indicator. @useenum{ErrorValue}. 
     *  Note that even when errorFlag is CORRECT, bufferSize may be zero meaning no data was returned. 
     */
    ErrorValue errorFlag;
    /** Size of the requested data in bytes. Used by the application to allocate buffer.*/
    int64_t        bufferSize;
    /** Dimension of the 1D array of data (in voxels). The dimension depends on the resolution. */
    int        bufferDimension;
    /** First and last point of the segment of data in voxel coordinates. */
    SbVec3i32    lineCoord[2];
  };
  /**
   *
   * Information about data returned for an arbitrary polyline.
   * Used with the SoLDMDataAccess::getData(DataInfoPolyLine&, int, const SbBox3i32&, int, const SbVec3i32*, SoBufferObject*) method.
   *
   * The following figure shows the information that is returned.
   * @IMAGE subvolumepolylinedetail.jpg
   *
   * [OIVJAVA-WRAPPER-CLASS NO_WRAP]
   */
  class LDM_API DataInfoPolyLine
  {
  public:

    /** Constructor of a dataInfoPolyline. The number of points must be passed for the structure allocation.*/
    DataInfoPolyLine(int numPoint);
    virtual ~DataInfoPolyLine();

    /** Copy constructor */
    DataInfoPolyLine(const DataInfoPolyLine& A);
    void operator=(const DataInfoPolyLine& A);

    /** Error indicator. @useenum{ErrorValue}. 
     *  Note that even when errorFlag is CORRECT, bufferSize may be zero meaning no data was returned. 
     */
    ErrorValue errorFlag;

    /** Size of the requested data in bytes. Used by the application to allocate buffer.*/
    int64_t         bufferSize;

    /** Number of segments in the polyline. */
    int numSegment;

    /** Dimension of buffer for each segment. This array of ints contains #numSegment elements.
        A null value means the segment does not intersect the data.*/
    int*  segmentBufferDim;

    /** Size of buffer for each segment (size = dimension*datatype).
        This array of ints contains #numSegment of non null dimension value elements.*/
    int*  segmentBufferSize;

    /** Segment coordinates in the data.
        Each segment of non null dimension has two associated SbVec3i32 coordinates (start and end point).
        This array of SbVec3i32 contains #numSegment of non null dimension value elements.
        [OIV-WRAPPER-CUSTOM-CODE] 
    */
    SbVec3i32**   segmentCoord;

    SoINTERNAL public:
    private:

    DataInfoPolyLine();
    void free();
    void commonConstructor();
    void allocate(int numPoint);
  };

  /**
   * Information about data returned for an arbitrary plane.
   * Used with the SoLDMDataAccess::getData(int, const SbBox3i32&, const SbPlane&, SoBufferObject*) method.
   *
   * The following figure shows the information that is returned.
   * @IMAGE subvolumeplanedetail.jpg
   *
   * [OIVJAVA-WRAPPER-CLASS NO_WRAP]
   */
  struct DataInfoPlane
  {
    /** Error indicator. @useenum{ErrorValue}. 
     *  Note that even when errorFlag is CORRECT, bufferSize may be zero meaning no data was returned. 
     */
    ErrorValue errorFlag;
    /** Size of the requested data in bytes. Used by the application to allocate buffer.*/
    int64_t        bufferSize;
    /** Dimension of the 2D data matrix (in voxels). The dimension depends on the resolution. */
    SbVec2i32  bufferDimension;
    /** Number of points of the polygon intersecting subvolume.
     * The polygon will have 3, 4, 5, or 6 vertices depending on how the plane
     * intersects the subvolume.
     */
    int        numPolygonPoints;
    /** Polygon intersecting subvolume in voxel coordinates. */
    SbVec3i32  polygonCoord[6];
    /** Quad corresponding to the returned data in voxel coordinates. */
    SbVec3i32  quadCoord[4];
    /** Axis corresponding to the first dimension of the returned array.
     * 0 = x, 1 = y, 2 = z.
     */
    int        uAxis;
    /** Axis corresponding to the second dimension of the returned array.
     * 0 = x, 1 = y, 2 = z.
     */
    int        vAxis;
  };

  /**
   * Information about returned data when directly accessing the data rather than copying the data.
   * Used with the SoLDMDataAccess::getData(int, const SbVec3i32&) method.
   *
   * [OIVJAVA-WRAPPER-CLASS NO_WRAP]
   */
  struct DataInfo
  {
    /** Error indicator. @useenum{ErrorValue}. 
     *  Note that even when errorFlag is CORRECT, bufferSize may be zero meaning no data was returned. 
     */
    ErrorValue errorFlag;
    /** LDM tile ID of the tile that contains the data at #tilePosition. */
    SoLDMTileID tileID;
    /** Dimensions of the data tile in voxels. */
    SbVec3i32 tileDimension;
    /** Position of the data tile in voxel coordinates (min = lower left back corner,
     * max = upper right front corner). */
    SbBox3i32 tilePosition;
    /** Size of the data tile in bytes. */
    int64_t bufferSize;
    /** \if_cpp Pointer \else Reference \endif to the data tile, valid until the #releaseData() method of SoLDMDataAccess
    * is called or the associated SoVolumeData is destroyed
    * [OIV-WRAPPER-CUSTOM-CODE]
    */
    void* tileData;
  };

  /**
   * Given a subvolume in voxel coordinates, copies the associated data
   * into an application buffer. @BR
   * The resolution is the power of 2 of the desired subsampling level
   * (0:1/1, 1:1/2, 2:1/4, ...). NOTE: The data is copied.
   *
   * All information is returned in the struct DataInfoBox. Call this method with buffer = NULL
   * to get the required size of the application buffer (in the bufferSize member
   * of DataInfoBox).
   *
   * @IMAGE subvolumebox.jpg
   */
  DataInfoBox getData(int resolution, const SbBox3i32& subVolume0, SoBufferObject* bufferObj = NULL);

  /**
   * Given a subvolume in voxel coordinates and a plane, copies the data intersecting the plane and
   * the subvolume into an application buffer. @BR
   * The resolution is the power of 2 of the desired subsampling level (0:1/1, 1:1/2, 2:1/4, ...)
   * NOTE: The data is copied.
   *
   * All information is returned in the struct DataInfoPlane. Call this method with buffer = NULL
   * to get the required size of the application buffer (in the bufferSize member
   * of DataInfoPlane).
   *
   *  @IMAGE subvolumeplane.jpg
   */
  DataInfoPlane  getData(int resolution, const SbBox3i32& subVolume0, const SbPlane& plane, 
                        SoBufferObject* bufferObj = NULL);

  /**
   * Given a subvolume in voxel coordinates and a line, copies the data intersecting the line and
   * the subvolume into an application buffer. @BR
   * The resolution is the power of 2 of the desired subsampling level (0:1/1, 1:1/2, 2:1/4, ...)
   * NOTE: The data is copied.
   *
   * All information is returned in the struct DataInfoLine. Call this method with buffer = NULL
   * to get the required size of the application buffer (in the bufferSize member
   * of DataInfoLine).
   *
   *  @IMAGE subvolumeline.jpg
   */
  DataInfoLine getData(int resolution, const SbBox3i32& subVolume0, const SbLine&  line , SoBufferObject* bufferObj = NULL) ;

  /**
   * Data values in a single seismic trace (a row of voxels along the volume X axis) are
   * copied into an application buffer. @BR  The trace is identified by a YZ voxel coordinate.
   * The range of values returned is the intersection of the trace with the specified
   * subvolume.
   * The resolution is the power of 2 of the desired subsampling level (0:1/1, 1:1/2, 2:1/4, ...)
   * NOTE: The data is copied.
   *
   * Returns a DataInfoTrace struct containing errorFlag, bufferSize (number of bytes) and
   * bufferDimension (number of values).
   * Call this method with buffer = NULL to get the required size of the application buffer
   * (in the bufferSize member of DataInfoTrace).
   * @IMAGE subvolumetrace.jpg
   */
  DataInfoTrace getData( int resolution, const SbBox3i32& subVolume, const SbVec2i32 coord, 
                       SoBufferObject* bufferObj = NULL);

  /**
   * Given a subvolume in voxel coordinates and a polyline (set of connected line segments), 
   * copies the data intersecting each line segment with the subvolume into an application buffer. @BR
   * The resolution is the power of 2 of the desired subsampling level (0:1/1, 1:1/2, 2:1/4, ...)
   * NOTE: The data is copied.
   *
   * All information is returned in the struct DataInfoPolyline. Call this method with buffer = NULL
   * to get the required size of the application buffer (in the bufferSize member
   * of DataInfoLine).
   * @IMAGE subvolumepolyline.jpg
   * [OIV-WRAPPER-ARG OUT,IN,IN,NO_WRAP{(polyline != NULL? polyline->Length: 0)},ARRAY,IN]
   */
  void getData(DataInfoPolyLine& infoPolyline, int resolution, const SbBox3i32& subVolume, int numPoints,
               const SbVec3i32* polyline, SoBufferObject* bufferObj = NULL);

  /**
   * Returns a \if_cpp pointer \else reference \endif to the block of data (LDM tile) which
   * contains the voxel at @B dataPosition@b.
   *
   * The resolution is the power of 2 of the desired subsampling level (0:1/1, 1:1/2, 2:1/4, ...).
   * In most cases you will pass 0 to request full resolution data.
   *
   * The data is not copied. The \if_cpp pointer \else reference \endif is valid until #releaseData() is
   * called with the corresponding tile id or until the associated volumeData node is destroyed.
   *
   * The LDM tile is locked in CPU memory until #releaseData() is called.  The application may lock
   * multiple tiles in memory using this method.  However these locked tiles count against the
   * maximum CPU memory allowed for the volume.  Locking too many tiles may reduce the rendered
   * image quality because LDM does not have enough memory to load other tiles.  A released tile
   * will remain in memory (allowing quick re-access) for some period of time, depending on whether
   * it is needed for rendering and (if not) on the value of SoLDMResourceParameters::tileHalfLife.
   *
   * This method is a good solution when the application needs to get the values of multiple voxels
   * that are not part of a line, plane or box.  After locking a tile, for each subsequent voxel, check if
   * the voxel position is contained in the currently locked tile.  You can use the intersect() method
   * on the SbBox3i32 tilePosition member of the DataInfo struct for this test.  If the voxel is in
   * the current tile, compute the offset into the data buffer and get its value.  If not, release the
   * current tile (using #releaseData) and get the appropriate tile by calling getData() again.
   *
   * \if_cpp
   * \par
   * \code
   * // Compute the address of a voxel value in a tile.
   * // We know the position of the desired voxel in absolute IJK coords and we
   * // know the location of the tile in absolute IJK coords, so we can compute
   * // the voxel position relative to the tile.  But note that tiles on a "face"
   * // of the volume are normally partial tiles, so we have to consider both the
   * // min and max corners of the tile.
   * SbVec3i32 voxelPos; // Absolute IJK position of desired voxel
   * DataInfo  dataInfo; // Tile info returned by getData() method
   *
   * SbVec3i32 tileSize = dataInfo.tileDimension;
   * SbVec3i32 tilePosMin, tilePosMax;
   * dataInfo.tilePosition.getBounds( tilePosMin, tilePosMax );
   *
   * SbVec3i32 relPos; // Relative IJK position of desired voxel
   * relPos[0] = (voxelPos[0] - tilePosMin[0]) * tileSize[0] / (tilePosMax[0] - tilePosMin[0] + 1);
   * relPos[1] = (voxelPos[1] - tilePosMin[1]) * tileSize[1] / (tilePosMax[1] - tilePosMin[1] + 1);
   * relPos[2] = (voxelPos[2] - tilePosMin[2]) * tileSize[2] / (tilePosMax[2] - tilePosMin[2] + 1);
   *
   * // Offset into tile data. First in voxels, then in bytes.
   * int64_t offset = (relPos[2]*tileSize[1] + relPos[1])*tileSize[0] + relPos[0];
   * offset *= volumeData->getDatumSize();
   *
   * // Address of voxel value
   * unsigned char* voxelPtr = ((unsigned char *)dataInfo.tileData) + offset;
   * \endcode
   * \endif
   */
  DataInfo getData(int resolution, const SbVec3i32& dataPosition);

  /**
   *
   * Tells VolumeViz that this block of data (returned by getData above)
   * is no longer in use by the application.
   */
  void releaseData(SoLDMTileID tileID);

  /**
   * Returns TRUE if the tile containing the data located at position dataPosition (IJK voxel
   * coordinates) at the specified resolution is uniform (all voxels have the same value).
   * Returns FALSE if not or if accessing the tile failed (e.g. invalid coordinates).
   * If TRUE then uniformValue contains the uniform value of the tile, else it is undefined.
   *
   * @M_SINCE 9.1
   */
  SbBool isTileUniform(int resolution, const SbVec3i32& dataPosition, double& uniformValue);

  ////////////////////////////
  // asynchronous data access
  ////////////////////////////

  /**
   * Given a subvolume in voxel coordinates, asynchronously copies the associated data
   * into an application buffer. @BR
   * The resolution is the power of 2 of the desired subsampling level
   * (0:1/1, 1:1/2, 2:1/4, ...). NOTE: The data is copied.
   *
   * Returns a request ID.  If requestID is positive, this value will be passed to the 
   * #endRequest method when the requested data is ready, and can then be used with
   * the @B getRequestedData @b method to complete the transaction.  If requestID is
   * negative, the data is already in memory, so endRequest will not be called,
   * and -requestID (a positive value) should be used with the getRequestedData method.
   *
   * Call this method with buffer = NULL
   * to get the required size of the application buffer.
   *
   * @B Limitations @b :
   *    - it is not currently possible to query the value of a single voxel using this method.
   * The smallest query possible is 8 voxels (a subvolume 2 x 2 x 2).
   *    - it is not possible to perform an asynchronous data request within  GLRender action
   * @IMAGE subvolumebox.jpg
   */
  int requestData(int resolution, const SbBox3i32& box, SoBufferObject* bufferObj);

  /**
   * Returns the data associated with @B requestID@b into @B infoBox@b.
   */
  void getRequestedData(int requestId, DataInfoBox& infoBox);

  /**
   * Given a subvolume in voxel coordinates and a plane, asynchronously copies the data intersecting the plane and
   * the subvolume into an application buffer. @BR
   * The resolution is the power of 2 of the desired subsampling level
   * (0:1/1, 1:1/2, 2:1/4, ...). NOTE: The data is copied.
   *
   * Returns a request ID.  If requestID is positive, this value will be passed to the 
   * #endRequest method when the requested data is ready, and can then be used with
   * the @B getRequestedData @b method to complete the transaction.  If requestID is
   * negative, the data is already in memory, so endRequest will not be called,
   * and -requestID (a positive value) should be used with the getRequestedData method.
   *
   * Call this method with buffer = NULL
   * to get the required size of the application buffer (in the bufferSize member
   * of DataInfoPlane).
   * @B Limitations @b :
   *    - it is not possible to perform an asynchronous data request within  GLRender action
   * @IMAGE subvolumeplane.jpg
   */
  int requestData(int resolution, const SbBox3i32& subVolume, const SbPlane& plane, SoBufferObject* bufferObj);

  /**
   * Returns the data associated with @B requestID@b into @B infoPlane@b.
   */
  void getRequestedData(int requestId, DataInfoPlane& infoPlane);

  /**
   * Given a subvolume in voxel coordinates and a line, asynchronously copies the data intersecting the line and
   * the subvolume into an application buffer. @BR
   * The resolution is the power of 2 of the desired subsampling level
   * (0:1/1, 1:1/2, 2:1/4, ...). NOTE: The data is copied.
   *
   * Returns a request ID.  If requestID is positive, this value will be passed to the 
   * #endRequest method when the requested data is ready, and can then be used with
   * the @B getRequestedData @b method to complete the transaction.  If requestID is
   * negative, the data is already in memory, so endRequest will not be called,
   * and -requestID (a positive value) should be used with the getRequestedData method.
   *
   * Call this method with buffer = NULL
   * to get the required size of the application buffer (in the bufferSize member
   * of DataInfoLine).
   * @B Limitations @b :
   *    -  it is not possible to perform an asynchronous data request within  GLRender action
   * @IMAGE subvolumeline.jpg
   */
  int requestData(int resolution, const SbBox3i32& subVolume0, const SbLine& line , SoBufferObject* bufferObj);

  /**
   * Returns the data associated with @B requestID@b into @B infoLine@b.
   */
  void getRequestedData(int requestId, DataInfoLine& infoLine);

  /**
   * Given a subvolume in voxel coordinates and a stack of line, asynchronously copies the data intersecting each line and
   * the subvolume into an application buffer. @BR
   * The resolution is the power of 2 of the desired subsampling level
   * (0:1/1, 1:1/2, 2:1/4, ...). NOTE: The data is copied.
   *
   * Returns a request ID.  If requestID is positive, this value will be passed to the 
   * #endRequest method when the requested data is ready, and can then be used with
   * the @B getRequestedData @b method to complete the transaction.  If requestID is
   * negative, the data is already in memory, so endRequest will not be called,
   * and -requestID (a positive value) should be used with the getRequestedData method.
   *
   * Call this method with buffer = NULL
   * to get the required size of the application buffer (in the bufferSize member
   * of DataInfoLine).
   * @B Limitations @b :
   *    - it is not possible to perform an asynchronous data request within  GLRender action
   * @IMAGE subvolumepolyline.jpg
   * [OIV-WRAPPER-ARG IN,IN,NO_WRAP{(polyline != NULL? polyline->Length: 0)},ARRAY,IN]
   */
  int requestData(int resolution, const SbBox3i32& subVolume, int numPoints, const SbVec3i32* polyline, SoBufferObject* bufferObj);

  /**
   * Returns the data associated with @B requestID@b into @B infoPolyline@b.
   */
  void getRequestedData(int requestId, DataInfoPolyLine& infoPolyline);

  /**
   * Data values in a single seismic trace (a row of voxels along the volume X axis) are
   * asynchronously copied into an application buffer.   @BR
   * The trace is identified by a YZ voxel coordinate.
   * The range of values returned is the intersection of the trace with the specified
   * subvolume (which may be the full dimensions of the volume).
   * The resolution is the power of 2 of the desired subsampling level 
   * (0:1/1, 1:1/2, 2:1/4, ...). NOTE: The data is copied.
   *
   * Returns a request ID.  If requestID is positive, this value will be passed to the 
   * #endRequest method when the requested data is ready, and can then be used with
   * the @B getRequestedData @b method to complete the transaction.  If requestID is
   * negative, the data is already in memory, so endRequest will not be called,
   * and -requestID (a positive value) should be used with the getRequestedData method.
   *
   * Call this method with buffer = NULL
   * to get the required size of the application buffer (in the bufferSize member
   * of DataInfoTrace).
   * @B Limitations @b :
   *    - it is not possible to perform an asynchronous data request within  GLRender action
   * @IMAGE subvolumetrace.jpg
   */
  int requestData(int resolution, const SbBox3i32& subVolume0, const SbVec2i32&  coord, SoBufferObject* bufferObj);

  /**
   * Returns the data associated with @B requestID @b into @B infoTrace @b.
   */
  void getRequestedData(int requestId, DataInfoTrace& infoTrace);

  /**
   * This method is called each time a data request is finished, meaning that the
   * data requested asynchronously is now available. This method should be overloaded
   * in a derived class in order to handle loaded data.
   * [OIVJAVA-WRAPPER CUSTOM_CODE] 
   */
  virtual void endRequest(int requestId);

  /**
   * Converts a voxel coordinate (I,J,K) to a geometry coordinate (X,Y,Z).
   *
   * The geometric coordinates are expressed in "extent" space, where voxel
   * coordinates are mapped to the box defined by the volume extent.
   *
   * Correctly converts coordinates that are outside the volume extent,
   * but the resulting voxel coordinate is outside the volume dimensions.
   *
   * @see voxelToXYZ(const SbBox3i32&) const
   * @see XYZToVoxel(const SbVec3f&) const
   * @see XYZToVoxel(const SbBox3f&) const
   */
  SbVec3f voxelToXYZ(const SbVec3i32 &dataPosition);

  /**
   * Converts the specified box in voxel coordinates (I,J,K) to geometric coordinates (X,Y,Z).
   *
   * The geometric coordinates are expressed in "extent" space, where voxel
   * coordinates are mapped to the box defined by the volume extent.
   *
   * Correctly converts coordinates that are outside the volume extent,
   * but the resulting voxel coordinate is outside the volume dimensions.
   *
   * @see voxelToXYZ(const SbVec3i32&) const
   * @see XYZToVoxel(const SbVec3f&) const
   * @see XYZToVoxel(const SbBox3f&) const
   *
   * @M_SINCE 10.8
   */
  SbBox3f voxelToXYZ( const SbBox3i32& boxIJK );

  /**
   * Converts a geometry coordinate (X,Y,Z) to a voxel coordinate (I,J,K).
   *
   * The geometric coordinates are expressed in "extent" space, where voxel
   * coordinates are mapped to the box defined by the volume extent.
   *
   * Correctly converts coordinates that are outside the volume extent,
   * but the resulting voxel coordinate is outside the volume dimensions.
   *
   * @see voxelToXYZ(const SbVec3i32&) const
   * @see voxelToXYZ(const SbBox3i32&) const
   * @see XYZToVoxel(const SbBox3f&) const
   */
  SbVec3i32 XYZToVoxel(const SbVec3f &dataPosition);

  /**
   * Converts the specified box in geometric coordinates (X,Y,Z) to voxel coordinates (I,J,K).
   *
   * The geometric coordinates are expressed in "extent" space, where voxel
   * coordinates are mapped to the box defined by the volume extent.
   *
   * Correctly converts coordinates that are outside the volume extent,
   * but the resulting voxel coordinate is outside the volume dimensions.
   *
   * @see voxelToXYZ(const SbVec3i32&) const
   * @see voxelToXYZ(const SbBox3i32&) const
   * @see XYZToVoxel(const SbVec3f&) const
   *
   * @M_SINCE 10.8
   */
  SbBox3i32 XYZToVoxel( const SbBox3f& boxXYZ );

  /**
   * Set the GetDataMode. @BR
   * This controls how data is requested from the volume reader when the necessary
   * tiles are not already present in LDM cache memory. Default is CACHE.
   */
  bool setGetDataMode (const GetDataMode getDataMode);

  /**
   * Returns the GetDataMode (see setGetDataMode).
   */
  GetDataMode getGetDataMode();


#if SoDEPRECATED_BEGIN(8000)

  /**
   * [OIV-WRAPPER-ARG IN,IN,ARRAY]
   */
  SoDEPRECATED_METHOD(8000,"No longer used. Use getData(int, const SbBox3i32& , SoBufferObject*) instead.")
  DataInfoBox getData(int resolution, const SbBox3i32& subVolume, void* buffer);

  /**
   * [OIV-WRAPPER-ARG IN,IN,IN,ARRAY]
   */
  SoDEPRECATED_METHOD(8000,"No longer used. Use getData(int, const SbBox3i32&, const SbPlane&, SoBufferObject*) instead.")
  DataInfoPlane getData(int resolution, const SbBox3i32& subVolume, const SbPlane& plane,
                        void* buffer);

  /**
   * [OIV-WRAPPER-ARG IN,IN,IN,ARRAY]
   */
  SoDEPRECATED_METHOD(8000,"No longer used. Use getData(int, const SbBox3i32&, const SbLine&, SoBufferObject* ) instead.")
  DataInfoLine getData(int resolution, const SbBox3i32& subVolume, const SbLine&  line , void* buffer);

  /**
   * [OIV-WRAPPER-ARG IN,IN,IN,ARRAY]
   */
  SoDEPRECATED_METHOD(8000,"No longer used. Use getData(int, const SbBox3i32&, const SbVec2i32, SoBufferObject*) instead.")
  DataInfoTrace getData(int resolution, const SbBox3i32& subVolume, const SbVec2i32 coord,
                       void* buffer );

  /**
   * [OIV-WRAPPER-ARG OUT,IN,IN,NO_WRAP{(polyline != NULL? polyline->Length: 0)},ARRAY,ARRAY]
   */
  SoDEPRECATED_METHOD(8000,"No longer used. Use getData(int, const SbBox3i32&, const SbVec2i32, SoBufferObject* ) instead.")
  void getData(DataInfoPolyLine& infoPolyline, int resolution, const SbBox3i32& subVolume, int numPoints,
               const SbVec3i32* polyline, void* buffer);

  /**
   * [OIV-WRAPPER-ARG IN,IN,ARRAY]
   */
  SoDEPRECATED_METHOD(8000,"No longer used. Use requestData(int, const SbBox3i32&, SoBufferObject*) instead.")
  int requestData(int resolution, const SbBox3i32& subVolume, void* buffer);

  /**
   * [OIV-WRAPPER-ARG IN,IN,IN,ARRAY]
   */
  SoDEPRECATED_METHOD(8000,"No longer used. Use requestData(int, const SbBox3i32&, const SbPlane&, SoBufferObject*) instead.")
  int requestData(int resolution, const SbBox3i32& subVolume, const SbPlane& plane, void* buffer);

  /**
   * [OIV-WRAPPER-ARG IN,IN,IN,ARRAY]
   */
  SoDEPRECATED_METHOD(8000,"No longer used. Use requestData(int, const SbBox3i32&, const SbLine&, SoBufferObject*) instead.")
  int requestData(int resolution, const SbBox3i32& subVolume, const SbLine& line, void* buffer);

  /**
   * [OIV-WRAPPER-ARG IN,IN,NO_WRAP{(polyline != NULL? polyline->Length: 0)},ARRAY,ARRAY]
   */
  SoDEPRECATED_METHOD(8000,"No longer used. Use requestData(int, const SbBox3i32&, const SbLine&, SoBufferObject*) instead.")
  int requestData(int resolution, const SbBox3i32& subVolume, int numPoints, const SbVec3i32* polyline, void* buffer);

  /**
   * [OIV-WRAPPER-ARG IN,IN,IN,ARRAY]
   */
  SoDEPRECATED_METHOD(8000,"No longer used. Use requestData(int, const SbBox3i32&, const SbVec2i32&, SoBufferObject*) instead.")
  int requestData(int resolution, const SbBox3i32& subVolume, const SbVec2i32& coord, void* buffer);

#endif /** @DEPRECATED_END */

SoINTERNAL public:

  void initPerfCounters();
  void resetPerfCounters();
  bool intersectWith(const SbBox3i32 &, const SbBox3i32 &, SbBox3i32 &);
  int  intersectWith(const SbBox3i32 &, const SbPlane &, int, SbVec3i32 [], SbVec3i32 []) const;
  bool intersectWith(const SbBox3i32 &, const SbLine &, SbVec3i32 []) const;
  int  getZaxis(const SbVec3f &) const;
  void copyData(unsigned short dataSetId, SoLDMTileID tile, SoBufferObject* tileBufferObj, int tileIndex);
  void copyData(unsigned short dataSetId, SoLDMTileID tile, SoBufferObject* tileBufferObj, int tileIndex, int requestId, bool canRelease=true);
  void requestNotCompleted(int requestId);
  std::vector<SoLDMTileID> getRequestedTiles(int requestId);
  SoDataSet* getRequestingDataSet(int requestId);
  bool isTileToPrefecth(const SoLDMTileID &tileId);
  void removeTileFromPrefetch(const SoLDMTileID &tileId);
  SbThreadBarrier* getDataAccessBarrier();
  SbThreadBarrier* getReaderThreadBarrier();
  
  //info needed when copyData is called
  enum CopyWhat{
    COPY_BOX,
    COPY_PLANE,
    COPY_LINE,
    COPY_SINGLE,
    COPY_TRACE
  };

  CopyWhat getTypeOfCopy () { return m_copyInfo.copyWhat;};

protected:
  int  clipSegment(const SbVec3i32 &, const SbVec3i32 &, SbVec3i32 *, int, const SbVec3i32 &, const SbVec3i32 &) const;

  //check if box is flat and readjust
  void      checkBox(int& _max, int& _min, int volBox);
  SbBox3i32 getBox(SbVec3i32 p1, SbVec3i32 p2, SbBox3i32 volBox);

  // TODO: this class should not be defined in a public header
  // and should be cleanup
  struct CopyInfo {
  public:
    CopyInfo()
    { reset(); }

    // reset all members to a well define default value
    void reset()
    {
      destBufferObj = NULL;
      userCpuObj = NULL;
      vmin = SbVec3i32(0,0,0);
      vmax = SbVec3i32(0,0,0);
      tileDim = -1;
      dataSize = -1;
      subVolume.makeEmpty();
      resolution = -1;
      copyWhat = COPY_SINGLE;
      qmin = SbVec2i32(0,0);
      qmax = SbVec2i32(0,0);
      normal = SbVec3f(0,0,0);
      distance = -1;
      Zaxis = -1;
      voxelsToCopy = NULL;

      // asynchronous access
      tiles.clear();
      tilesFromDisk.clear();
      tilesToPrefetch.clear();
      tilesInfos.clear();
      tilesIndex.clear();
      ldmDataAccess = NULL;
      dataSet = NULL;
      loadPolicy = SoLDMResourceParameters::NEVER;
      requestSize = -1;
      requestId = -1;
      shouldCallback = false;
      dataInfoPolyline = NULL;
      dataInfoLine = NULL;
      dataInfoPlane = NULL;
      dataInfoBox = NULL;
      dataInfoTrace = NULL;
    }

  public:
    SoRef<SoBufferObject> destBufferObj;
    SoRef<SoCpuBufferObject> userCpuObj;
    SbVec3i32 vmin;
    SbVec3i32 vmax;
    int tileDim;
    int dataSize;
    SbBox3i32 subVolume;
    int resolution;
    CopyWhat copyWhat;
    SbVec2i32 qmin;
    SbVec2i32 qmax;
    SbVec3f normal;
    float distance;
    SbPlane plane;
    int Zaxis;
    SbVec2i32 bufDim;
    std::vector< std::vector<SbVec2i32> >* voxelsToCopy;
    // asynchronous access
    std::vector<SoLDMTileID> tiles;
    std::vector<SoLDMTileID> tilesFromDisk;
    std::vector<SoLDMTileID> tilesToPrefetch;
    std::map<int64_t, SbVec3i32> tilesInfos;
    std::map<int64_t, int> tilesIndex;
    SoLDMDataAccess* ldmDataAccess;
    SoDataSet* dataSet;
    SoLDMResourceParameters::LoadPolicy loadPolicy;
    int64_t requestSize;
    int requestId;
    bool shouldCallback;
    DataInfoPolyLine* dataInfoPolyline;
    DataInfoLine* dataInfoLine;
    DataInfoPlane* dataInfoPlane;
    DataInfoBox* dataInfoBox;
    DataInfoTrace* dataInfoTrace;
  } m_copyInfo;


private:
  enum threadType
  { NONE = 0,
    PLANE,
    BOX,
    TRACES,
    LINE,
    POLYLINE
  };
  
  /**
  * Forbid recopy constructors
  */
  SoLDMDataAccess& operator=(const SoLDMDataAccess& ldmDa);
  SoLDMDataAccess(const SoLDMDataAccess& );

  SoDataSet      *m_dataSet;
  SoLDMMediator  *m_mediator;

  std::map<int, CopyInfo> m_requests;

  //bool getRequestInfo(int resolution, const SbBox3i32& subVolume0, void *buffer, DataInfoBox& info, CopyInfo& copyInfo);
  bool getRequestInfo(int resolution, const SbBox3i32& subVolume0, SoBufferObject *bufferObj, DataInfoBox& info, CopyInfo& copyInfo);

  bool getRequestInfo(int resolution, const SbBox3i32& subVolume0, const SbPlane& plane, SoBufferObject *bufferObj,
                      SoLDMDataAccess::DataInfoPlane& info, SoLDMDataAccess::CopyInfo& copyInfo);
  
  bool getRequestInfo(int resolution, const SbBox3i32& subVolume0, const SbLine& line, SoBufferObject *bufferObj,
                      SoLDMDataAccess::DataInfoLine& info, SoLDMDataAccess::CopyInfo& copyInfo);
  
  bool getRequestInfo(int resolution, const SbBox3i32& subVolume0, const SbVec2i32 coord, SoBufferObject *bufferObj,
                      SoLDMDataAccess::DataInfoTrace& info, SoLDMDataAccess::CopyInfo& copyInfo);

  //void copyData(unsigned short dataSetId, SoLDMTileID tile, void* bufferTile, int tileIndex, CopyInfo& copyInfo);
  void copyData(unsigned short , SoLDMTileID tile, SoBufferObject* tileBufferObj, int tileIndex, SoLDMDataAccess::CopyInfo& copyInfo);

  static void *readTraceFromDiskThreadRoutine(void* userData);
  static void *readSliceFromDiskThreadRoutine(void* userData);
  static void *readTilesFromDiskThreadRoutine(void* userData);
  void readSliceInTile(SoLDMDataAccess* dataAccess, int fileID, SbBox3i32& tilePos, int& sliceNumber, SoBufferObject* destBuffer );
  void buildPlane(SoLDMDataAccess* dataAccess, int firstAxis, int secondAxis, int sliceNumber);
  bool createThreadReadDisk(SoLDMDataAccess::threadType threadType);
  void updateTilesLists(SoLDMDataAccess::CopyInfo& copyInfo);
  void checkGetDataMode (const SoLDMDataAccess::CopyWhat copyWhat);

  enum RequestState {
    NOT_FOUND,
    OK
  };

  RequestState queryRequestState(int requestId);

  int m_nextRequestId;
  int getNextRequestId();

  SoPerfCounterManager* m_perf;

  // Perf counter used for data copy timing
  SoPerfCounter* m_time_copy_data_trace;
  SoPerfCounter* m_time_copy_data_vol;
  SoPerfCounter* m_time_copy_data_plane;

  // Perf counter used for tile select timing
  SoPerfCounter* m_time_tile_select_trace;
  SoPerfCounter* m_time_tile_select_vol;
  SoPerfCounter* m_time_tile_select_plane;

  SoLDMAlgorithms* m_algorithms;

  GetDataMode m_getDataMode;

  SbThreadBarrier *m_readerBarrier;
  SbThreadBarrier *m_diskReadThreadBarrier;
  SbThread *m_diskReadThread;
  bool m_threadMustStop;
  threadType m_threadType;

  static void checkBuffer(SoBufferObject* buffer);

SoINTERNAL public:
     CopyInfo& getCopyInfo();
     bool threadMustStop();

};// END SoLDMDataAccess definition


inline SoDataSet*
SoLDMDataAccess::getDataSet()
{
  return m_dataSet; 
}

inline void
SoLDMDataAccess::endRequest(int)
{
  SoDebugError::post("SoLDMDataAccess::endRequest", "An asynchronous access has been made using first class SoLDMDataAccess. Please implement your own, overloading endRequest method.");
}

inline SoLDMDataAccess::GetDataMode
SoLDMDataAccess::getGetDataMode()
{
  return m_getDataMode;
}

inline SoLDMDataAccess::CopyInfo&
SoLDMDataAccess::getCopyInfo()
{
  return m_copyInfo;
}

inline bool
SoLDMDataAccess::threadMustStop()
{
  return m_threadMustStop;
}

inline SbThreadBarrier* 
SoLDMDataAccess::getDataAccessBarrier()
{
  return m_readerBarrier;
}

inline SbThreadBarrier* 
SoLDMDataAccess::getReaderThreadBarrier()
{
 return m_diskReadThreadBarrier;
}

/***************************************************************************/
inline SoLDMDataAccess&
SoLDMDataAccess::operator=(const SoLDMDataAccess& )
{
  assert(0);
  return *this;
}

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif /* LDM_DATA_ACCESS_H */


