/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef  _SO_LDM_GLOBAL_RESOURCE_PARAM_
#define  _SO_LDM_GLOBAL_RESOURCE_PARAM_

class SoLDMTileManager;

#include <LDM/SoLDM.h>
#include <Inventor/nodes/SoSubNode.h>


/**
 *
 * Loading callback for LDM mode. Called with TRUE means the process starts loading.
 * FALSE means the process is done loading.
 *
 * @memberof SoLDMGlobalResourceParameters
 *
 * [OIV-WRAPPER NAME{LDMLoadCB}]
 */
typedef void SoLDMLoadCB( SbBool startLoading, void *userData );

/**
 *
 * Monitoring callback for LDM mode. Reports how much time is needed to load a given
 * amount of data. The amount of data is specified when setting the CB.
 *
 * @memberof SoLDMGlobalResourceParameters
 *
 * [OIV-WRAPPER NAME{LDMMonitoringCB}]
 */
typedef void SoLDMMonitoringCB(SoDataSet *dataSet, double elapsedTime, double memLoaded, void *userData);

/**
 * @LDMEXT Global resource parameters of VolumeViz.
 *
 * @ingroup LDM
 *
 * @DESCRIPTION
 *   This class defines functions for controlling global resource parameters of VolumeViz.
 *
 *   These parameters can have a significant impact on resource usage (primarily memory)
 *   data loading and rendering performance.
 *
 *   **Memory management**
 *
 *   This class allows the application to specify how much CPU and GPU memory VolumeViz can
 *   use to load data and textures.  By default the main memory and texture memory limits are set to a
 *   percentage of the total available system and GPU memory (see the #setMaxMainMemory(),
 *   #setMaxTexMemory() and #setMax2DTexMemory() methods). Applications can use the SoCpuDevice and SoGLDevice
 *   classes to query the total and available memory on the CPU and GPU. How VolumeViz uses CPU
 *   memory depends on the data source (tiled or not tiled), and the selected tile cache policy
 *   (see the field SoLDMResourceParameters::tileCachePolicy).
 *
 *   **Tiles:**  
 *   VolumeViz uses a data manager called LDM to manage volume data.  LDM manages
 *   volume data in chunks called "tiles".  If the volume size is larger than the tile
 *   size, then there are two classes of tile: full-resolution tiles that contain actual
 *   data values and low-resolution tiles that contain sub-sampled data values.
 *   The application may have native data that is already tiled or the data may be
 *   pre-processed to create tiles and store them in the LDM file format. In an LDM format
 *   file the tiles may optionally be compressed to save disk space and memory, but in this
 *   case, the tiles must be expanded before sending to the GPU. If the data is not tiled,
 *   for example disk formats like DICOM and SEGY or an in-memory volume, then tiles must
 *   be created in memory at runtime.
 *
 *   **Tile size:**  
 *   Note that the default tile size is small and
 *   generally you will get better performance with non-tiled data by explicitly setting a
 *   larger tile size (see the ldmResourceParameters field and SoLDMResourceParameters::tileDimension).
 *   If the volume data fits in CPU and GPU memories, the tile size may be set to the volume dimension
 *   to mimic non-tiled volume data behavior (i.e., without multi-resolution).
 *
 *   **Caches:**  
 *   VolumeViz allocates a significant amount of memory on the CPU to store tile data and
 *   on the GPU to process the rendering. The chosen amounts of memory are a trade off between
 *   efficient data loading and rendering performance.
 *   If the volume is larger than the available or allowed memory, the *most important*
 *   tiles are kept in memory. The importance is determined by various heuristics set
 *   in this class and in SoLDMResourceParameters.
 *
 *   - **First level CPU tile cache** (controlled by *max main memory*)  
 *     VolumeViz uses this amount of CPU memory as a first level cache for LDM data tiles.
 *     The main memory limit is set in megabytes using the #setMaxMainMemory() method.
 *     If the data set is small enough to fit in system memory, then you should generally set the memory limit 
 *     high enough to allow this.  If the data set is larger than system memory,
 *     then set the memory limit as high as possible without impacting the application's
 *     other needs for memory.
 *     For example, if the tile size is 64 and each voxel is a float value, then each tile
 *     requires approximately 1 MB of system memory. However the CPU memory footprint
 *     can be reduced for compressed tiles (stored compressed) depending on the
 *     selected SoLDMResourceParameters#tileCachePolicy.
 *
 *     When there are multiple data sets, by default each data set receives
 *     getMaxMainMemory() divided by the number of datasets.  Each time a new data set is created
 *     the per-dataset value is recomputed and some memory is taken away from existing data sets.
 *     **Note:** This default way to allocate memory (when handling multiple data sets) is not optimal
 *     if the data sets have
 *     significantly different sizes.  For example, the memory required for a height field is
 *     normally much less than for a volume.  In this case the application can explicitly
 *     allocate memory to each data set using the field SoLDMResourceParameters#maxMainMemory
 *     associated with the data set (see the SoDataSet::ldmResourceParameters field).
 *
     - **Second level CPU tile cache**  
 *     VolumeViz also uses a second cache of tiles in CPU memory that is used when some tiles
 *     must be created or uncompressed on the fly. This second level cache is normally used 
 *     when a tile must be sent to the GPU but it is not used when the SoLDMResourceParameters#tileCachePolicy is ALL.
 *     This cache has a short life cycle thus the tiles might only be temporarily stored in this cache.
 *
 *     **Warning:**  
 *     This second cache is separate from the first level CPU cache and its
 *     footprint is not included in the limit specified by maxMainMemory
 *     (neither #setMaxMainMemory nor SoLDMResourceParameters#maxMainMemory).
 *     Even if the variable OIV_BUFFER_OBJECT_CACHE_SIZE is deprecated, it can be set as a workaround.
 *     It defines the maximum number of tiles that can be stored in the second level cache.
 *
 *   - **GPU Tile Cache** (controlled by *max texture memory*)  
 *     VolumeViz uses this amount of GPU memory to cache LDM data tiles on the GPU.
 *     The texture memory limit is set in megabytes using the #setMaxTexMemory() method for
 *     for 3D tiles and by the #setMax2DTexMemory() method for the 2D tiles.
 *     You can set these values separately for each data set using SoLDMResourceParameters.
 *
 *     **Warning:**  
 *     Some systems integrate GPU memory with CPU memory. In this case, if you allocate 4GB of CPU
 *     memory *and* 4GB of GPU memory, the system may see a total of 8GB private memory used by
 *     the process, leading to out-of-memory errors or swapping issues.
 *     If you face such issues, you should ensure that the max CPU memory + max GPU memory is less than
 *     the total available CPU memory for your machine.
 *
 *   \if_cpp
 *   Notes:  
 *   SoLDMGlobalResourceParameters values are initialized by SoLDMLargeDataManagement::init(),
 *     so SoLDMGlobalResourceParameters methods must be called after calling
 *     SoLDMLargeDataManagement::init() or SoVolumeRendering::init()
 *   \endif
 *
 * @SEE_ALSO
 *    SoDataSet,
 *    SoVolumeData,
 *    SoLDMResourceParameters
 *
 */
class LDM_API SoLDMGlobalResourceParameters
{

public:
  ////////////////////////////////////////////////////////////////////////////////
  // Memory Control Parameters section

  /**
   * Sets the maximum main (CPU) memory, in MB, to use for all VolumeViz data sets. @BR
   * You can also set the maximum main memory for each data set separately using the maxMainMemory
   * field of the data set's SoLDMResourceParameters object (see SoDataSet::ldmResourceParameters).
   * The sum of all per data set values must be less than or equal to the value set here.
   *
   * Default (since Open Inventor 8.1) is 50% of total CPU memory up to 70% of free memory.
   * See SoCpuDevice to query available CPU memory.
   */
  static void setMaxMainMemory( int maxMainMemory );

  /**
   * Returns the maximum main memory in MB for all the VolumeViz data sets.
   * See #setMaxMainMemory.
   */
  static int  getMaxMainMemory();

  /**
   * Sets the maximum texture (GPU) memory, in MB, for all VolumeViz data sets. @BR
   * You can also set the maximum texture memory for each data set separately using the
   * maxTexMemory field of the data set's SoLDMResourceParameters object
   * (see SoDataSet::ldmResourceParameters).  The sum of all per data set values must
   * be less than or equal to the value set here.
   *
   * Default is 75% of GPU memory available at application startup.
   * See SoGLDevice to query available GPU memory.
   */
  static void setMaxTexMemory( int maxTexMemory );

  /**
   * Returns the maximum texture memory in MB for all the VolumeViz data sets.
   * See #setMaxTexMemory.
   */
  static int  getMaxTexMemory();

  /**
   * Sets the maximum number of 3D textures allowed to be downloaded into texture (GPU) memory,
   * per frame (render traversal), for all VolumeViz data sets. @BR
   * You can set this parameter separately for each data set using field
   * SoLDMResourceParameter::tex3LoadRate.
   * Default is 10 3D textures (effectively 3 tiles) per render traversal.
   *
   * Each tile needed for volume rendering (SoVolumeRender) and volume geometry must be 
   * transfered to GPU memory as a 3D texture.
   * So, for example, transfering 1000 tiles at the default loadRate (10) requires 100 frames.
   * (How many seconds this takes depends on the frames per second possible, which in turn
   * depends on many factors including the complexity of the scene graph.)  Increasing the
   * loadRate value reduces the number of frames required to reach maximum resolution.  But
   * the render traversal has to wait for the data transfer to finish, so each frame may take
   * longer to render, increasing the total time to reach maximum resolution.  
   */
  static void setTex3LoadRate( int loadRate );

  /**
   * Returns the maximum number of tiles (3D textures) allowed to be downloaded into texture (GPU)
   * memory per frame for all the VolumeViz data sets. 
   * See #setTex3LoadRate().
   */
  static int  getTex3LoadRate();

  /**
   * Sets the maximum number of 2D textures allowed to be downloaded into texture (GPU) memory per
   * frame (render traversal) for all the VolumeViz data sets. @BR
   * You can set this parameter separately for each data set using field
   * SoLDMResourceParameter::tex2LoadRate.
   * Default is 100 2D textures per frame.
   *
   * Each tile needed to render a slice (SoOrthoSlice, SoVolumeSkin, etc) must be loaded
   * into GPU memory as a 2D texture.  Increasing the loadRate value reduces the number of
   * frames required to reach maximum resolution.  But the render traversal has to wait for 
   * the data transfer to finish, so each frame may take longer to render, increasing 
   * the total time to reach maximum resolution.  
   */
  static void setTex2LoadRate( int loadRate );

  /**
   * Returns the maximum number of 2D textures allowed to be loaded into texture memory
   * per frame for all the VolumeViz data sets. 
   * See #setTex2LoadRate().
   */
  static int  getTex2LoadRate();

  /**
   * Sets how many simultaneous tile loading threads to use if multi I/O mode is on
   * (it is on when LDM mode is on).
   * 
   * The IVVR_NUM_LOADERS preference (see SoPreference) lets you choose the initial
   * number of loading threads at launch. Changing this preference at runtime will not affect
   * the actual number of loading threads, use this method instead. If this preference
   * is not set, the default is 4.
   * 
   * The IVVR_MAX_NUM_LOADERS preference lets you set a maximum number of tile
   * loading threads. If this method is called with a larger number, the number
   * of loading threads will be set to the maximum instead. If this preference is
   * not set, the maximum is 20.
   */
  static void         setNumIO(unsigned int numIO);

  /**
   * Returns the current number of loading threads.
   * See #setNumIO().
   */
  static unsigned int getNumIO();

  ////////////////////////////////////////////////////////////////////////////////
  // VisualFeedback Parameters section

  /**
   * Visual feedback modes.
   * Used with #setVisualFeedbackParam() method.
   */
  enum VisualFeedbackParam {
  /**
   * Draw the tiles of SoVolumeRender or SoVolumeSkin primitives. They are drawn by default.
   */
    DRAW_TILES,
  /**
   * Use texturing for SoOrthoSlice, SoObliqueSlice, and SoVolumeGeometry primitives. They are used by default.
   */
    DRAW_SLICES_TEX,
  /**
   * Show tiles loaded in main memory.
   * This implies showing the multiresolution topology.
   * The tile outlines of tiles of full resolution are drawn with
   * a brighter yellow.
   * The topology is not drawn by default.
   * @IMAGE drawtopology.jpg
   */
    DRAW_TOPOLOGY,
  /**
   * Draw tile outlines for all primitives. The tile outline of tiles of full resolution
   * are drawn with a brighter gray for the volume, and a brighter green for the slices.
   * Outlines are not drawn by default.
   * @IMAGE drawtileoutline.jpg
   */
    DRAW_TILE_OUTLINE,
  /**
   * Show tiles visited by the Tile Visitor. They are not shown by default.
   *
   * For the default LDM implementation,
   * displays an octree representation showing traversal of the octree. Red dots are tiles
   * that have been valuated. Blue indicates the minimum and maximum resolution thresholds.
   * Yellow shows the octree itself.
   * @IMAGE showtilesvaluated.jpg
   */
    SHOW_TILES_VALUATED,
  /**
   * Show tiles loaded (in red) and unloaded (in blue) in texture memory. They are not
   * shown by default.
   */
    SHOW_LOAD_UNLOAD_TILES,
  /**
   * Use fake data instead of the real data during loading into main memory. The fake data
   * is programmatically generated and is designed to illustrate features of Large Data Management.
   * Fake data is not used by default.
   * @IMAGE fakedata.jpg
   */
    USE_FAKE_DATA
  };

  /**
   * Sets the specified visual feedback parameter to true or false.
   */
  static void setVisualFeedbackParam( VisualFeedbackParam param, SbBool state );

  /**
   * Returns the state of the specified visual feedback parameter.
   */
  static SbBool getVisualFeedbackParam( VisualFeedbackParam param );


  ////////////////////////////////////////////////////////////////////////////////
  // Viewing Parameters section

  /**
   * Sets the view culling flag. 
   * Default is TRUE.  When TRUE, only tiles that are inside the view frustum
   * (i.e. visible) will be loaded and conversely tiles that are outside the
   * view frustum may be discarded.
   */
  static void setViewCulling( SbBool state, SbBool forceRedraw = TRUE );

  /**
   * Returns the view culling flag.
   * See #setViewCulling().
   */
  static SbBool getViewCulling();

  /**
   * Sets the screen resolution culling flag.
   * Default is FALSE. When TRUE, only tiles for which the projection of a voxel is
   * greater than or equal to 1 pixel on screen will be loaded. This avoids unnecessary
   * loading of high resolution data for large volumes.
   * However if you "zoom out" from a volume, it does not force lower resolution tiles
   * to be used (high resolution tiles already in memory will still be used). If necessary
   * use the #SoLDMResourceParameters::fixedResolution field to force use of lower
   * resolution tiles.
   */
  static void setScreenResolutionCulling( SbBool flag );

  /**
   * Returns the screen resolution culling state.
   * See #setScreenResolutionCulling().
   */
  static SbBool getScreenResolutionCulling();

  /**
   * Sets the IgnoreFullyTransparentTiles flag. 
   * Default is FALSE.  When TRUE, tiles for which all voxels between
   * their min and max values are fully transparent (alpha = 0) will not be refined.
   * In other words a low resolution version will always be used.  This can reduce
   * memory requirements for volumes with significant "empty space".
   */
  static void setIgnoreFullyTransparentTiles( SbBool state );

  /**
   * Returns the IgnoreFullyTransparentTiles flag.
   * See #setIgnoreFullyTransparentTiles().
   */
  static SbBool getIgnoreFullyTransparentTiles();

  /**
   * Sets the viewpoint refinement flag. 
   * Default is TRUE.  When TRUE, tile refinement depends on the viewpoint.
   * In other words, tiles closer to the camera have higher priority and higher
   * resolution data will be loaded for closer tiles first.  This can be useful
   * for large volumes that cannot be completely loaded in memory.
   */
  static void setViewpointRefinement( SbBool state );

  /**
   * Returns the viewpoint refinement flag.
   * See #setViewpointRefinement().
   */
  static SbBool getViewpointRefinement();

  /**
   * Sets the slice equal resolution flag. 
   * Default is FALSE. When TRUE, slice primitives, e.g. SoOrthoSlice and SoObliqueSlice,
   * are always rendered using a set of tiles at the same resolution level.  This avoids
   * potentially distracting visual effects caused by mixing tiles of different resolution
   * on the same slice.  But it means that the slice will be rendered at a lower resolution
   * if it is not possible to load every tile needed for the higher resolution.  When FALSE
   * and viewpoint refinement is enabled, higher resolution tiles will be loaded for
   * portions of the slice closer to the camera even if it is not possible to load every
   * high resolution tile.
   *
   * An alternative strategy is to enable the SoSlice::largeSliceSupport field.  When
   * large slice support is enabled, if all the required full resolution tiles have already
   * been loaded, then the slice data is taken from LDM cache memory as usual. But if some 
   * required tiles are not currently in memory, the full resolution slice data will be loaded
   * directly from the volume reader without loading the complete tiles. This reduces disk I/O
   * and reduces the amount of system memory required to display the slice at full resolution, 
   * so larger (or more) slices can be displayed. 
   */
  static void setSliceEqualResolution( SbBool state, SbBool forceRedraw = TRUE );

  /**
   * Returns the slice equal resolution flag.
   * See #setSliceEqualResolution().
   */
  static SbBool getSliceEqualResolution();

  /**
   * Sets the load notification rate (number of tiles).
   * When @B rate@b new tiles have been loaded in main memory, a redraw will be scheduled.
   * This is useful because tiles are loaded asynchronously using multiple threads.
   * The default is 50 tiles.  Which means, for example, if each voxel is one byte and the
   * tileSize is 64x64x64, a redraw will be scheduled whenever 13 MB of data has been loaded.
   */
  static void setLoadNotificationRate(int);

  /**
   * Returns the load notification rate.
   * See #setLoadNotificationRate().
   */
  static int getLoadNotificationRate();

  /**
   * Sets a \if_dotnet delegate \else callback \endif to be called with the tile loading state. 
   * The \if_dotnet delegate \else callback function \endif is called with true when the process
   * starts loading tiles. And with false when loading is complete, in other words when LDM has the
   * best possible set of tiles in memory for the current render nodes, viewpoint, etc.
   *
   * NOTE: This feature is available with VolumeVizLDM only.
   * [OIV-WRAPPER EVENT_NAME{LoadCallback}]
   */
  static void setLoadCB( SoLDMLoadCB *func, void *userData = NULL );

  /**
   * Sets a monitoring \if_dotnet delegate \else callback \endif\.
   * This provides a snapshot of the tile loading performance.
   * The \if_dotnet delegate \else callback \endif will be called each time memAmount MB 
   * of data have been loaded, with the amount of data and the amount of time elapsed to load that data.
   *
   * NOTE: This feature is available with VolumeVizLDM only.
   */
  static void setMonitoringCB( SoLDMMonitoringCB *func, int memAmount, void *userData = NULL );

  /**
   * Returns the default maximum GPU memory in MB.
   */
  static inline int64_t getDefaultMaxTexMem() { return s_defaultMaxTexMemMB; }

  /**
   * Returns the default maximum system (CPU) memory in MB.
   */
  static inline int64_t getDefaultMaxMainMem() { return s_defaultMaxMainMemMB; }

  /**
   * Sets the maximum 2D texture GPU memory in MB allowed to be loaded for all VolumeViz data sets. @BR
   * The sum of 2D texture memory allowed per data set (see SoLDMResourceParameters::max2DTextures field)
   * must be less than or equal to this value.  The total amount of GPU memory allowed is set
   * using the #setMaxTexMemory method.  This method limits the amount of GPU memory that can
   * be used by (for example) SoOrthoSlice and SoVolumeSkin nodes.  This can be useful when combining
   * slices and volume rendering because the 3D textures used by volume rendering require much more GPU
   * memory.
   * Default is 5% of GPU memory available at application startup.
   */
  static void setMax2DTexMemory(uint64_t maxMemory);

  /**
   * Get the maximum number of 2D textures allowed to be loaded.
   * See #setMax2DTexMemory.
   */
  static uint64_t getMax2DTexMemory();

#if SoDEPRECATED_BEGIN(9610)
  /**
   * Sets the low resolution when moving flag.
   * Default is FALSE.  When TRUE, lower resolution tiles are automatically used
   * while the camera is moving (e.g. when user is panning and rotating.
   */
  SoDEPRECATED_METHOD( 9610, "No longer used. Use SoLDMResourceParameters' TileCachePolicies field instead." )
  static void setMoveLowResolution( SbBool state );

  /**
   * Returns the move low resolution flag.
   * See #setMoveLowResolution().
   */
  SoDEPRECATED_METHOD( 9610, "No longer used. Use SoLDMResourceParameters' TileCachePolicies field instead." )
  static SbBool getMoveLowResolution();

#endif /** @DEPRECATED_END */

SoINTERNAL public:

  // These identifiers are binary fixed,
  // do not changed enum value as it will break ART scenario
  enum Cmd{
    SET_VISUAL_FEEDBACK_PARAM = 0,
    SET_MAX_MAIN_MEMORY = 1,
    SET_MAX_TEX_MEMORY = 2,
    SET_MAX_2D_TEXTURES = 3,
    SET_MAX_NUM_TRIANGLES = 4, // no more supported
    SET_TEX3_LOAD_RATE = 5,
    SET_TEX2_LOAD_RATE = 6,
    SET_NUMIO = 7,
    SET_WRITE_ALTERNATE_REP = 8,
    SET_READ_ALTERNATE_REP = 9,
    SET_DELAYED_RENDERING = 10,
    SET_VIEW_CULLING = 11,
    SET_SCREEN_RESOLUTION_CULLING = 12,
    SET_VIEWPOINT_REFINEMENT = 13,
    SET_IGNORE_FULLY_TRANSPARENT_TILES = 14,
    SET_SLICE_EQUAL_RES = 15,
    SET_MOVE_LOW_RES = 16,
    SET_LOAD_NOTIFICATION_RATE = 17
  };

  static void initClass();

  /**
   * Return true if drawing tiles in CPU or GPU memory
   */
  static bool isDrawingOutlines();

#if SoDEPRECATED_BEGIN(9720)

  /** Returns the maximum allowed texture size for the current GL device in bytes */
  SoDEPRECATED_METHOD( 9720, "No longer supported." )
  static size_t getMaxTextureSizeBytes();

#endif /** @DEPRECATED_END */

protected:

// ------------------------------------------------------------
  // VisualFeedback Parameters
  static SbBool s_drawTiles;
  static SbBool s_drawSlicesTex;
  static SbBool s_drawTopology;
  static SbBool s_drawTileOutline;
  static SbBool s_showTilesLoaded;
  static SbBool s_showLoadUnloadTiles;
  static SbBool s_useFakeData;
  // ------------------------------------------------------------
  // Viewing Parameters
  static SbBool s_ignoreFullyTransparentTiles;
  static SbBool s_viewCulling;
  static SbBool s_screenResolutionCulling;
  static SbBool s_viewpointRefinement;
  static SbBool s_sliceEqualResolution;
  static SbBool s_moveLowResolution;

  static int64_t s_defaultMaxTexMemMB;
  static int64_t s_defaultMaxMainMemMB;

};
#endif // _SO_LDM_GLOBAL_RESOURCE_PARAM_


