/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : J.HUMMEL (Dec 2002)
** Modified by : T.DUFOUR (MMM yyyy)
**=======================================================================*/
#ifndef  _SO_LDM_RESOURCE_MANAGER_
#define  _SO_LDM_RESOURCE_MANAGER_

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <LDM/SoLDMLargeDataManagement.h>
#include <LDM/nodes/SoLDMResourceParameters.h>
#include <Inventor/SbVec.h>
#include <Inventor/STL/vector>
#include <LDM/SoLDMTopoOctree.h>
#define MIN_RES_THRESHOLD 10000 //high number meaning not set

class SoDataSet;

// callback function prototypes
/** @memberof SoLDMResourceManager */
typedef void SoLDMTexFrontCB (void *userData);

/**
 * @LDMEXT Resource Manager
 *
 * @ingroup LDM
 *
 * @DESCRIPTION
 *
 * The SoLDMResourceManager maintains information about the resources available on the machine
 * and constrains the data flow to these resource limits. The major constraints are
 * texture memory, main memory, and main memory-to-texture memory bandwidth.
 * Each of them can entirely be specified by the application, at any time.
 *
 * @SEE_ALSO
 * SoLDMLargeDataManagement.
 *
 */

SoEXTENDER_Documented class LDM_API SoLDMResourceManager : public SoLDMLargeDataManagement
{
public:
  /**
   * Constructor.
   */
  SoLDMResourceManager(SoDataSet* ds);

  /**
   * Destructor.
   */
  virtual ~SoLDMResourceManager();


SoEXTENDER public:

  enum LoadPolicy {
    NO_USER_INTERACTION,   // Only load when no user interaction
    ALWAYS,                // Always load
    NEVER                  // Never. No loading happens anymore. Allows suspending loading process.
  };

  /**
   *
   * This function allows you to synchronously unload the data if it needs to be refetched.
   */
  void invalidateData();

  /**
   *
   * Sets whether LDM should use multiple loading threads or not.
   * Default is TRUE.
   */
  static void   setMultiIO(SbBool on);
  /**
   *
   * Returns TRUE if LDM uses multiple loading threads.
   */
  static SbBool getMultiIO();

  static void        setMonitoringAmount(int);
  static int         getMonitoringAmount();

  /**
   * Returns the tile dimension.
   */
  SbVec3i32  getTileSize() const;

  /**
   * Returns the data size in number of bytes.
   */
  int  getDataSize();

  /**
   * Sets the size of a texture element in bits(for instance, 32 if RGBA).
   */
  void setTexelSize( int texelSize );

  /**
   * Returns the size of a texture element in bits.
   */
  int  getTexelSize();

  /**
   * Returns the dimension of the data set.
   */
  const SbVec3i32& getVolumeDim() const;

#if SoDEPRECATED_BEGIN(8000)

  SoDEPRECATED_METHOD(8000,"Use SoLDMRessourceParameter's field tex3LoadRate instead.")
    void setTex3LoadRate( int loadRate );

  SoDEPRECATED_METHOD(8000,"Use SoLDMRessourceParameter's field tex3LoadRate instead.")
    int  getTex3LoadRate();

  SoDEPRECATED_METHOD(8000,"Use SoLDMRessourceParameter's field tex2LoadRate instead.")
    void setTex2LoadRate( int loadRate );

  SoDEPRECATED_METHOD(8000,"Use SoLDMRessourceParameter's field tex2LoadRate instead.")
    int  getTex2LoadRate();

  SoDEPRECATED_METHOD(8000,"Use SoLDMRessourceParameter's field minResolutionThreshold instead.")
    void setMinResolutionThreshold( int threshold = MIN_RES_THRESHOLD );   // don't unload above this threshold.

  SoDEPRECATED_METHOD(8000,"Use SoLDMRessourceParameter's field minResolutionThreshold instead.")
    int  getMinResolutionThreshold();

  SoDEPRECATED_METHOD(8000,"Use SoLDMRessourceParameter's field maxResolutionThreshold instead.")
    void setMaxResolutionThreshold( int threshold );   // don't load below this threshold.

  SoDEPRECATED_METHOD(8000,"Use SoLDMRessourceParameter's field maxResolutionThreshold instead.")
    int  getMaxResolutionThreshold();

#endif /** @DEPRECATED_END */


#if SoDEPRECATED_BEGIN(9000)

  SoDEPRECATED_METHOD(9000,"No longer used. Use getTileSize() instead.")
    int getTileDimension();

#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(9100)

  /**
   * Set the allowed main memory in MB for corresponding dataset
   */
  SoDEPRECATED_METHOD(9100,"No longer used. Use SoLDMRessourceParameter's field maxMainMemory instead.")
  void setMaxMainMemory( int maxMainMemory );

 /** 
  * Get the allowed main memory in MB for corresponding dataset
  */
  SoDEPRECATED_METHOD(9100,"No longer used. Use SoLDMRessourceParameter's field maxMainMemory instead.")
  int  getMaxMainMemory();

  SoDEPRECATED_METHOD(9100,"No longer used. Use SoLDMRessourceParameter's field maxTexMemory instead.")
  void setMaxTexMemory( int maxTexMemoryMB );

  SoDEPRECATED_METHOD(9100,"No longer used. Use SoLDMRessourceParameter's field maxTexMemory instead.")
  int  getMaxTexMemory();

  /**
   * Get the allowed number of tiles in main memory.
   */
  SoDEPRECATED_METHOD(9100,"No longer used.")
  int  getMaxTilesInMainMem();

  SoDEPRECATED_METHOD(9100,"No longer used.")
  int  getMaxTilesInTexMem();
#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(10000)

  /**
  * When using getData functions from #SoDataSet::SoLDMDataAccess, once a tile is unlocked by
  * the application (call to #SoDataSet::SoLDMDataAccess::releaseData), its weight will decrease
  * until the tile is weighted again by LDM. This function allows you to set how long
  * it takes for the tile for its weight to be divided by 2 (initial weight is 1). Default is 2 seconds.
  * The tile weight will be set to 0 when the weight becomes less than 0.05.
  */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMResourceParameters's field tileHalfLife instead." )
  void setTileHalfLife( float timeInSec );

  /**
  * Returns tile half life.
  */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMResourceParameters's field tileHalfLife instead." )
  float getTileHalfLife();

  /**
  * Sets loading behavior. Passing NO_USER_INTERACTION means the asynchronous loading
  * thread will only load when the user does not interact with the scene (i.e., the
  * application idles).
  *
  * The loading occurs as long as there is something to load if ALWAYS is passed.
  * No loading occurs if NEVER is passed.
  * Default is NO_USER_INTERACTION.
  */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMResourceParameters's field loadPolicy instead." )
  void setLoadPolicy( LoadPolicy loadPolicy );
  /**
  * Returns the current #LoadPolicy value.
  */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMResourceParameters's field loadPolicy instead." )
  LoadPolicy getLoadPolicy();

  /*
   * Sets the time (in seconds) for which VViz does not load once a user
   * stopped interaction
   * Consulted only in NO_USER_INTERACTION mode
   * Default is 0.3 seconds.
   */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMResourceParameters::setMovingTimeOut(float) instead." )
  void setMovingTimeOut( float );

  /*
   * gets the time for which VViz should not load once a user
   * stopped interaction
   */
  SoDEPRECATED_METHOD( 10000, "Use float SoLDMResourceParameters::getMovingTimeOut() instead." )
  float getMovingTimeOut();

  /**
   * Sets the total number of tiles allowed to be loaded in texture memory per frame
   * (for all data sets). Default is 1.
   */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMGlobalResourceParameters::setTex2LoadRate(int) instead." )
  static void setTex2VVizLoadRate( int loadRate );

  /**
   * Returns total number of tiles allowed to be loaded in texture memory per frame
   * (for all data sets).
   */
  SoDEPRECATED_METHOD( 10000, "Use int SoLDMGlobalResourceParameters::getTex2LoadRate() instead." )
  static int  getTex2VVizLoadRate();

  /**
   * Sets the total number of tiles allowed to be loaded in texture memory per frame
   * (for all data sets). Default is 1.
   */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMGlobalResourceParameters::setTex3LoadRate(int) instead." )
  static void setTex3VVizLoadRate( int loadRate );

  /**
   * Returns total number of tiles allowed to be loaded in texture memory per frame
   * (for all data sets).
   */
  SoDEPRECATED_METHOD( 10000, "Use int SoLDMGlobalResourceParameters::getTex3LoadRate() instead." )
  static int  getTex3VVizLoadRate();

  /**
   * Sets maximum global amount of main memory in MB. All data sets will share this amount of memory.
   * The sum of each main memory space allowed per data set (#setMaxMainMemory)
   * must be less or equal than #getMaxVVizMainMemory.
   */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMGlobalResourceParameters::setMaxMainMemory(int) instead." )
  static void setMaxVVizMainMemory( int maxMainMemoryMB ); // set/get the allowed main memory in MB

  /**
   * Sets the allowed global texture memory in MB (for all data sets). Default is 48 MB.
   * The sum of each texture memory space allowed per data set (#setMaxTexMemory)
   * must be less or equal than #getMaxVVizTexMemory.
   */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMGlobalResourceParameters::setMaxTexMemory(int) instead." )
  static void setMaxVVizTexMemory( int maxTexMemoryMB );

  /**
   * Returns the maximum global amount of main memory in MB.
   */
  SoDEPRECATED_METHOD( 10000, "Use int SoLDMGlobalResourceParameters::getMaxMainMemory() instead." )
  static int getMaxVVizMainMemory(); // for all the LDM datasets.

  /**
   * Returns the allowed global texture memory in MB (for all data sets).
   */
  SoDEPRECATED_METHOD( 10000, "Use int SoLDMGlobalResourceParameters::getMaxTexMemory() instead." )
  static int getMaxVVizTexMemory();

  /**
   * Sets how many loading threads to use if multiIO mode is on.
   * Default is 4.
   */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMGlobalResourceParameters::setNumIO(unsigned int) instead." )
  static void setNumIO( unsigned int numIO );

  /**
   *
   * Returns current number of loading threads.
   */
  SoDEPRECATED_METHOD( 10000, "Use unsigned int SoLDMGlobalResourceParameters::getNumIO() instead." )
  static unsigned int getNumIO();

  /**
   * When @B rate@b tiles have been loaded in main memory, a redraw will be scheduled.
   * The default is 50 tiles, which means, for example, if tileSize is 64x64x64,
   * the system will be notified that new data has arrived every 13MB.
   */
  SoDEPRECATED_METHOD( 10000, "Use SoLDMGlobalResourceParameters::setLoadNotificationRate(int) instead." )
  static void setLoadNotificationRate( int rate );

  /**
  * Returns notification rate.
  */
  SoDEPRECATED_METHOD( 10000, "Use int SoLDMGlobalResourceParameters::getLoadNotificationRate() instead." )
  static int getLoadNotificationRate();

#endif /** @DEPRECATED_END */

SoINTERNAL public:
  float m_decreaseWeightFactor;
  float getDecreaseWeightFactor() {return m_decreaseWeightFactor;};

  /**
   * Sets dimension of the tile stored in disk.
   */
  void setTileSize( SbVec3i32 tileDim );

  /**
   * Sets the size of the data in number of bytes.
   */
  void setDataSize( int dataSize );

  /**
   * Sets the dimension of the data set.
   */
  void setVolumeDim( const SbVec3i32& dim );

  int         getNumTileToTime();

  /*
   * Enum used to store wich resources has changed. Can be cumulative to 
   * represent several resources modification
   * ie: GPU resources and CPU resource updated -> value =GPU_MEM_UPDATED | CPU_MEM_UPDATED
   */
  enum State
  {
    NO_UPDATE               = 0x00,
    CPU_MEM_UPDATED         = 0x01, // cpu memory has been changed
    GPU_MEM_UPDATED         = 0x02, // gpu memory has been changed
    GPU_2D_UPDATED          = 0x04, // gpu 2d memory has been changed
    LOADRATE_2D_UPDATED     = 0x08, // LoadRate 2D has been changed
    LOADRATE_3D_UPDATED     = 0x10, // LoadRate 3D has been changed
    LOAD_RATE_UPDATED       = 0x20, // Load notification rate has been updated
    UPDATE_ALL              = 0xFF  // Special value to include all previous state
  };

  // Update all is needed according to the given state
  virtual void updateResources(unsigned short resourceState);

  // Internal method
  int  getMaxMainMemoryInKB() const;       // return amount of main memory in KB
  int  getMaxTexMemoryInKB() const;        // return amount of tex memory in KB
  int  getMax2DTexMemoryInKB() const;      // return amount of 2D tex memory in KB

  /*
  Those set of functions returns the sum of resources of all data sets registered
  */
  int  getSumOfMaxTilesInTexMem();
  int  getSumOfTex3LoadRate();
  int  getSumOfMax2DTextures();
  int  getSumOfTex2LoadRate();
  void getSumOfMemAndTileSize(uint64_t &memByte, int& tileSize);
  void getSumOfNumTileInMem( uint64_t &maxTileInMem );

  static void computeInternalCounters(const std::vector<SoLDMResourceManager*>& rsToUpdate);
  static void shareSpareResources(const std::vector<SoLDMResourceManager*>& rsToUpdate, 
    size_t spareAmount, SoLDMResourceManager::State resourceType);

  /* *
  * Set of method to compute and allocate internally amount of resources for each dataset
  * Each method works on a specific resources and determine how much of this resource can
  * be spared between all dataset. The computation is based on global value for each resources,
  * and the amount manually set by application 
  */
  void allocateGPUResources(const std::vector<SoLDMResourceManager*>& rsToUpdate);
  void allocateCPUResources(const std::vector<SoLDMResourceManager*>& rsToUpdate);
  void allocateGPU2DResources(const std::vector<SoLDMResourceManager*>& rsToUpdate);
  void allocateTex2LoadRate(const std::vector<SoLDMResourceManager*>& rsToUpdate);
  void allocateTex3LoadRate(const std::vector<SoLDMResourceManager*>& rsToUpdate);

  void setIdealTextFrontCb( SoLDMTexFrontCB *func, void* userData = NULL);
  void idealTexFrontReached();
  SoLDMTexFrontCB* m_callbackFunc;
  void* m_userData;
  SoDataSet* getDataSet() { return m_ds; };

  /**
   * get maximum of 2D textures memory in MB allowed to be displayed.
   */
  static uint64_t getMaxVViz2DTexMemory();

  /**
   * Sets maximum of 2D textures memory in MB allowed to be displayed.
   */
  static void setMaxVViz2DTexMemory(uint64_t maxMemory);

  /**
   * Set the maximum global of 2D Textures memory in MB displayable.
   */
  void setMax2DTexMemory(int max2DTexMemory);

  /**
   * Returns the maximum global of 2D Textures memory in MB displayable.
   */
  int getMax2DTexMemory();

  /** @see SoLDMResourceParameters::tileBorders */
  void enableTileBorders( bool enableTileBorders );

  /** @see enableTileBorders() */
  bool areTileBordersEnabled() const;

  /** @see SoLDMResourceParameters::minTilesToLoad */
  void setMinTilesToLoad( int minTilesToLoad );

  /** @see setMinTilesToSyncLoad() */
  int getMinTilesToLoad() const
  {
    return m_minTilesToLoad;
  }

  /** @see SoLDMResourceParameters::viewpointRefinementPolicy */
  void setViewpointRefinement( SoLDMResourceParameters::ViewpointRefinementPolicy viewportRefinementPolicy );

  /** @see setViewpointRefinement() */
  bool getViewpointRefinement() const;

  /** @see SoLDMResourceParameters::screenResolutionCullingPolicy */
  void setScreenResolutionCulling(SoLDMResourceParameters::ScreenResolutionCullingPolicy screenResolutionCullingPolicy);

  /** @see setScreenResolutionCulling() */
  bool getScreenResolutionCulling() const;

  /** Size of one tile in CPU mem in bytes */
  uint64_t getDataTileSizeInBytes() const;

  /** Size of one tile in GPU mem in bytes (2D or 3D depending on is3DTile) */
  uint64_t getTexTileSizeInBytes(bool is3DTile = false) const;

private:

  int m_minResThreshold;       // minimum resolution required (0:1/1, 1:1/2, 2:1/4, ...), default is 100.
  int m_maxResThreshold;       // maximum resolution allowed (0:1/1, 1:1/2, 2:1/4, ...), default is 0.

  bool m_tileBordersEnabled;
  int m_minTilesToLoad;
  SoLDMResourceParameters::ViewpointRefinementPolicy m_viewpointRefinementPolicy;
  SoLDMResourceParameters::ScreenResolutionCullingPolicy m_screenResolutionCullingPolicy;

  // TODO: these 2 methods should be removed while the following friend dependencies will be managed
  friend class SoLDMMultiIOTileManager;
  friend class SoLDMMediator;

  //static void updateMainMemParams( int mem, SoLDMResourceManager* exclude );
  static void updateNumTilesInMainMem();

  /** Update max main memory to be at least equal to the sum of available memory for each dataset */
  static void updateMaxMainMem();

  /** update number of 2D Textures for each resources */
  static void updateSumMax2DTextures();

  /**
   * Returns the list of resource managers to update.
   * A SoLDMResourceManager can be updated in case it's SoVolumeData has been rendered once.
   */
  static const std::vector<SoLDMResourceManager*>& getRSToUpdate();

  /////////////////////////////////////////////////////////////////////////////
  // texture memory controls (for SoVolumeRender, SoVolumeGeometry)
  static int s_maxVVizTexMemoryKB; // amount in KB of texture memory allowed for all the LDM datasets, default is 0 KB.

  int m_maxTexMemoryKB;            // amount in KB of texture memory allowed for the corresponding dataset.
  int m_max2DTexMemoryKB;          // amount in KB of 2D texture memory allowed for the corresponding dataset.

  int m_numTilesInTexMem;        // num tiles allowed in texture memory = m_maxTexMemory / tileTexSize

  static void updateNumTilesInTexMem();

  /** Update max tex memory to be at least equal to the sum of available memory for each dataset */
  static void updateMaxTexMem();

  // Check that all dataSets parameters are consistent with global ones.
  // If, for each dataSet, one parameter is not consistent with the global one, it
  // is reset to -1 in these dataSets.
  static void checkParametersConsistency(const std::vector<SoLDMResourceManager*>& rsToUpdate);

  /////////////////////////////////////////////////////////////////////////////
  // texture 3 memory load rate controls (for SoVolumeRender, SoVolumeGeometry)
  static int s_tex3VVizLoadRate;  // num tiles allowed to load in tex mem per frame for all the LDM datasets, default is 1.
  int m_tex3LoadRate;             // num tiles allowed to load in tex mem per frame for the corresponding dataset.


  // texture 2 memory controls (for SoOxxSlice, SoVolumeSkin)
  static uint64_t s_maxVViz2DTexMemory; // 2D tiles memory allowed in texture memory for all LDM datasets, default is 64M ( ~5% of 1G ).

  // texture 2 memory load rate controls (for SoOxxSlice, SoVolumeSkin)
  static int s_tex2VVizLoadRate;  // num tiles allowed to load in tex mem per frame for all the LDM datasets, default is 64.
  int m_tex2LoadRate;             // num tiles allowed to load in tex mem per frame for the corresponding dataset.



  static int s_maxVVizMainMemoryKB; // amount in KB of main memory allowed for all the LDM datasets, default is 256 MB.
  int m_maxMainMemoryKB;            // amount in KB of main memory allowed for the corresponding dataset.

  int m_numTilesInMainMem;        // num tiles allowed in main memory = m_maxMainMemory / tileSize

  static int s_loadNotificationRate;
  static int s_memToLoad;

  //number of IO:
  static unsigned int s_numIO;
  static SbBool       s_isMultiIO;


  SbVec3i32 m_tileDimension;    // tile dimension, default is 64.
  int m_dataSize;         // data size in bytes (1,2 or 4), default is 1.
  int m_texelSize;        // texel size in bits, default is 8.
                          // =8  for 8 bits indexed 3D textures
                          // =32 for RGBA 3D textures
                          // =24 for 8 bits indexed 2D textures
                          // =96 for RGBA 2D textures
  SbVec3i32 m_volDim;     // dataset dimension



  static std::vector<SoLDMResourceManager*> s_resourceManagers; // list of all ResourceManager's

  
   // Internal octree needed to compute resources
  SoLDMTopoOctree m_topoOctree;

  // Adjust the CPU memory allocated per dataset by taking in account
  // dataset that needs less memory
  void adjustMemPerDataset(const std::vector<SoLDMResourceManager*>& rsToUpdate);

  SoDataSet* m_ds;

  // cache value before and after ajustMaxTilesInTexMem
  int m_getSumOfMaxTilesInTexMemAdjusted;
  int m_getSumOfMaxTilesInTexMem;

  /** 
  * Amount of memory needed by the volume to load
  * entirely in memory
  */
  size_t m_maxNeededMemKB;

  // List of resourceManager to take in account for 
  // resource sharing
  static std::vector<SoLDMResourceManager*> s_rsToUpdate;

  // Store the number of mask associated to each data set
  static  std::map<SoLDMMediator*, int> s_maskPerMediator;

  // Store the number of volume data that are not volumeMask, by cumulating their datasize.
  // For example dataset in float will count for 4, while a dataset in byte will count
  // for one. This allow to ponderate properly the amount of resources to 
  // dataset, taking in account their datasize.
  static size_t s_nbVolumeData;

  // store the amount of GPU memory available for dataset not manually configured
  // ie : ( s_MaxVVizTexMemory - ( Sum of memory of all datasets with #maxTexMem != -1 )) / ( number of datasets with #maxTexMem == -1 )
  static size_t s_gpuMemPerDataSetKB;

  // store the amount of CPU memory available for dataset not manually configured
  // ie : ( s_MaxVVizMainMemory - ( Sum of memory of all dataset with #maxMainMem != -1 )) / ( number of datasets with #maxMainMem  == -1 )
  static size_t s_cpuMemPerDataSetKB;

  // store the 2D tiles memory in texture (GPU) memory available for dataset not manually configured MB
  // ie : (s_maxVViz2DTexMemory - ( Sum of memory of all dataset with #max2DTextures != -1 )) / ( number of datasets with #max2DTextures  == -1 )
  static uint64_t s_2DGpuTexMemoryPerDataSetMB;

  // store the amount of 2D Load Rate available for dataset not manually configured
  // ie : (s_tex2VVizLoadRate - ( Sum of 2DLoadRate of all dataset with #tex2LoadRate != -1 )) / ( number of datasets with #tex2LoadRate  == -1 )
  static size_t s_tex2LoadRatePerDataSet;

  // store the amount of 3D Load Rate available for dataset not manually configured
  // ie : (s_tex3VVizLoadRate - ( Sum of 3DLoadRate of all dataset with #tex3LoadRate != -1 )) / ( number of datasets with #tex3LoadRate == -1 )
  static size_t s_tex3LoadRatePerDataSet;

  // store the remainder amount of GPU Memory
  static size_t s_spareGpuMemoryKB;

  // store the remainder amount of CPU Memory
  static size_t s_spareCpuMemoryKB;

  // store the remainder amount of tex2 LoadRate
  static size_t s_spareTex2LoadRate;

  // store the remainder amount of tex3 LoadRate
  static size_t s_spareTex3LoadRate;

  // if true, prints MaxMain(MiB), Max2DTex(MiB), Max3DTex(MiB), Tex2LoadRate, Tex3LoadRate.
  static bool s_debugResourceValues;

  /** Mutex used to protect access to all static values above .
   * As these values are all static, each tilme a method which will update these values
   * is called (like setTileSize which call updateNumTileInMainMem, ...), we should lock this mutex 
   * to avoid several thread call setXXX together */
  static SbThreadMutex s_globalResourcesMutex;
};


#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif // _SO_LDM_RESOURCE_MANAGER_


