/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : J.HUMMEL (Dec 2002)
** Modified by : T.DUFOUR (MMM yyyy)
**=======================================================================*/
#ifndef  _SO_LDM_TILE_ID_
#define  _SO_LDM_TILE_ID_

#include <Inventor/nodes/SoSubNode.h>

#define LDM_TILE_ID_TYPE int64_t

/** 
 * @LDMEXT Tile ID
 *
 * @ingroup LDM
 *
 * @DESCRIPTION
 *
 * Encapsulates the ID of a tile.
 *
 * In the default LDM architecture, the LDM data is based on an octree
 * topology. The index is 0 for the tile of lowest resolution representing the 
 * entire volume (octree root node).
 * The index increments linearly going down through the octree.
 *
 * Indexing works as follows:
 *
 * Tile 1 is the lower back left corner of the cube.
 * The index increments on X, then Y, and finally Z.
 * So the back tiles are:@BR
 *
 * 3  4@BR
 * 1  2
 *
 * And the front tiles are:@BR
 *
 * 7  8@BR
 * 5  6
 *
 * The tiles of full resolution are the leaf tiles.
 *
 *
 */
class LDM_API SoLDMTileID {

 public:

  /**
   * Constructor.
   */
  SoLDMTileID()
    :m_id(0) {}

  /** 
   * Constructor from ID value.
   */
  SoLDMTileID( LDM_TILE_ID_TYPE tileID)
    :m_id(tileID) {}

  /**
   * Equality comparison operator.
   */
  inline bool operator==(const SoLDMTileID&b) const {return (m_id == b.m_id);}
  /**
   * Inequality comparison operator.
   */
  inline bool operator!=(const SoLDMTileID&b) const {return (m_id != b.m_id);}
  /**
   * Relational (less than or equal to) operator.
   */
  inline bool operator<=(const SoLDMTileID&b) const {return m_id <= b.m_id;}

  /**
   * Relational (greater than or equal to) operator.
   */
  inline bool operator>=(const SoLDMTileID&b) const {return m_id >= b.m_id;}

  /**
   * Relational (less than) operator.
   */
  inline bool operator<(const SoLDMTileID&b) const {return m_id < b.m_id;}

  /**
   * Relational (greater than) operator.
   */
  inline bool operator>(const SoLDMTileID&b) const {return m_id > b.m_id;}


  /**
   * Returns the tile ID.
   */
  inline LDM_TILE_ID_TYPE getID() const
  { return m_id; }

  /**
   * ++ Prefix, increment the tile id
   */
  inline SoLDMTileID &operator++();

  /**
   * Postfix ++, increment the tile id
   */
  inline SoLDMTileID operator++(int);

  /**
   * -- Prefix, increment the tile id
   */
  inline SoLDMTileID &operator--();

  /**
   * Postfix --, increment the tile id
   */
  inline SoLDMTileID operator--(int);

private:
  LDM_TILE_ID_TYPE m_id;
};


/*******************************************************************************/
SoLDMTileID &
SoLDMTileID::operator++()
{
  m_id++;
  return *this;
}

/*******************************************************************************/
SoLDMTileID
SoLDMTileID::operator++(int)
{
  SoLDMTileID tileId = *this;
  ++(*this);

  return tileId;
}

/*******************************************************************************/
SoLDMTileID &
SoLDMTileID::operator--()
{
  m_id--;
  return *this;
}

/*******************************************************************************/
SoLDMTileID
SoLDMTileID::operator--(int)
{
  SoLDMTileID tileId = *this;
  --(*this);

  return tileId;
}

/**
 * Writes the tile to the specified output stream.
 */
inline std::ostream& operator << (std::ostream& os, const SoLDMTileID& tile)
{
  return os << "(" << tile.getID() << ")";
}

#endif // _SO_LDM_TILE_ID_


