/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : T. DUFOUR (May 2005)
**=======================================================================*/

#ifndef  _SO_LDM_TOPO_OCTREE_
#define  _SO_LDM_TOPO_OCTREE_

#include <Inventor/SbLinear.h>
#include <Inventor/SbBox.h>
#include <LDM/SoLDMTileID.h>

class SoLDMOctreeNode;


#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

// Note : this level will be never reached (at least this century!)
#define LEVEL_MAX 20

/**
 * @LDMEXT Octree topology queries
 *
 * @ingroup LDM
 *
 * @DESCRIPTION
 *
 * The SoLDMTopoOctree class provides information about the hierarchy of
 * LDM tiles used for a given dataset, which depends only on the dataset
 * dimensions and the tile dimensions.
 *
 * For example, an application can query the total number of LDM tiles in a data set,
 * the number of resolution levels in the LDM tile hierarchy, the id of the tile that
 * contains a specified voxel position, the position and extent of a specified tile id,
 * and more. An application can conveniently query the SoLDMTopoOctree for an actual
 * data set using the SoDataSet::getLDMTopoOctree() method or create an LDMTopoOctree
 * object with any specified dimensions. In the latter case, the LDMTopoOctree
 * must be initialized using the init() method.
 *
 * In VolumeViz LDM every tile has both a tileID and a fileID.  The tileID is
 * the unique identifier of a tile in the (conceptual) complete, symmetrical
 * hierarchy.  The fileID is the index of a tile in a sequential numbering of
 * all the tiles that actually contain data.  The tileID is normally used to
 * identify tiles in an application, but the fileID is used, for example, when
 * calling the readTile method in an SoVolumeReader class.  In a cubical volume
 * (all three dimensions the same), there are equal numbers of tileIDs and
 * fileIDs, but in most volumes there are many fewer fileIDs than tileIDs.
 *
 * The terms 'resolution' and 'level' are both used in this class to identify
 * a specific level in the LDM tile hierarchy.
 * - Resolution level is counted @I up@i from the bottom. @BR
 *   So resolution level 0 is the full resolution tiles. The getTileID() method
 *   takes resolution level as a parameter.
 * - Tile level is counted @I down@i from the top. @BR
 *   So tile level 0 is the lowest resolution tile (tile 0) at the top of the
 *   hierarchy. The level() method returns the tile level, not the resolution level.
 * - Conversion: tileLevel = getLevelMax() - resolutionLevel.
 *
 * @SEE_ALSO
 * SoLDMTileID
 *
 *
 */
class LDM_API SoLDMTopoOctree
{

public:

  /**
   * Constructor.
   */
  SoLDMTopoOctree();

  /**
   * Initialize with the properties of the data set
   *
   * Note: Since OIV 9.0, the border parameter is no longer used.
   */
  void init(const SbVec3i32& dimension, const SbVec3i32& tileDim, const int border = 0);

  /**
   * Initialize with the properties of the data set
   *
   * Note: Since OIV 9.0, the border parameter is no longer used.
   */
  void init(const SbVec3i32& dimension, const int tileDim, const int border=0)
  {
    init(dimension, SbVec3i32(tileDim, tileDim, tileDim), border);
  }

  /**
   * Returns the bounding box, in voxel/cell coordinates, of the specified tile.
   * Regardless the tile's resolution level in the LDM tile hierarchy, the
   * returned bounding box is always in full-resolution voxel coordinates.
   */
  SbBox3i32 getTilePos( const SoLDMTileID& tileID )const;

  /**
   * Given the position of a voxel in full resolution voxel coordinates (i,j,k)
   * and a resolution level, returns the tile ID of the corresponding tile.
   *
   * Resolution level is counted up from the bottom of the tile hierarchy, so
   * resolution level 0 is the full resolution tiles.
   *
   * Regardless of the tile's resolution level in the LDM tile hierarchy, the
   * returned bounding box is always in full-resolution voxel coordinates.
   * In other words, the box is the region of the full-resolution volume that
   * the tile "spans" and the extent of this box is not the same as the tile
   * size unless the specified tile is at resolution level 0 (full resolution).
   * 
   * Example: If the volume is 256 x 256 x 256 with tile size 64
   * - There are 3 levels of tiles.
   * - Level 0 / Res 2 : first tile is tileId 0, pos = 0,0,0 x 255,255,255
   * - Level 1 / Res 1 : first tile is tileId 1, pos = 0,0,0 x 127,127,127
   * - Level 2 / Res 0 : first tile is tileId 9, pos = 0,0,0 x 63,63,63
   */
  SoLDMTileID getTileID( const SbVec3i32& cellPos, const int resolutionLevel )const;

  /**
   * Returns the number of fileIDs in the volume.  Effectively this is the
   * total number of actual tiles, including both full resolution and low resolution,
   * required to store the data set. This value is <= getNumTileIDs().
   */
  int getNumFileIDs()const;

  /**
   * Returns the number of tileIDs in the complete (conceptual) octree.
   * The complete octree is based on the power-of-2 greater than or equal to the
   * actual volume dimensions. This value is >= getNumFileIDs().
   */
  inline size_t getNumTileIDs()const;

  /**
   * Given a tileID, returns the corresponding fileID.
   */
  inline int getFileID( const SoLDMTileID& tileID )const;

  /**
   * Given a fileID, returns the corresponding tileID.
   */
  SoLDMTileID getTileID( const int fileID )const;

  /**
   * Returns false if the octree is empty.
   * Note: Octree must be initialized.
   */
  inline bool isEmpty() const
  { return (m_numFileIDs<=0); }

  /** 
  * Returns the maximum level of the octree tile hierarchy.
  * Example: If there are 3 levels in the octree, returns 2.
  * Note: Octree must be initialized.
  */
  int getLevelMax()const;

  /**
   * Returns the tile level of the given tile Id.
   * Conversion: resolutionLevel = getLevelMax() - tileLevel.
   * Note: Octree must be initialized.
   */
  inline int level( const SoLDMTileID& id ) const;

  /**
   * Returns the tile level of the given tile Id.
   * Conversion: resolutionLevel = getLevelMax() - tileLevel.
   * Note: Octree must be initialized.
   */
  int level( const LDM_TILE_ID_TYPE index ) const;

  /** Return the tilesize used to setup this octree */
  const SbVec3i32& getTileSize() const
  {
    return m_tileDim;
  }

SoINTERNAL public:

  
  /** Relative location to a tile. 
   * can be combined. ie LEFT | BOTTOM represent bottom left of a tile.
   * X axis from left to right,
   * Y axis from bottom to top,
   * Z axis from front to back. */
  enum Location
  {
    LEFT =  0x01,
    RIGHT = 0x02,
    BOTTOM = 0x04,
    TOP = 0x08,
    FRONT = 0x10,
    BACK = 0x20
  };
  /** Return opposite of given location */
  int opposite(int locationMask) const;

  /** All combinations of location:
  * [0, 5] = { LEFT, RIGHT, BOTTOM, TOP, FRONT, BACK }
  * [6, 17] = 
  * { LEFT | BOTTOM
  * , RIGHT | BOTTOM
  * , LEFT | TOP
  * , RIGHT | TOP
  * , FRONT | LEFT
  * , BACK | LEFT
  * , FRONT | RIGHT
  * , BACK | RIGHT
  * , FRONT | BOTTOM
  * , BACK | BOTTOM
  * , FRONT | TOP
  * , BACK | TOP }
  * [18, 25] = 
  * { LEFT | BOTTOM | FRONT
  * , LEFT | BOTTOM | BACK
  * , LEFT | TOP | FRONT
  * , LEFT | TOP | BACK
  * , RIGHT | BOTTOM | FRONT
  * , RIGHT | BOTTOM | BACK
  * , RIGHT | TOP | FRONT
  * , RIGHT | TOP | BACK }; 
  */
  static const int s_allLocations[26];

  /**
   * Destructor.
   */
  ~SoLDMTopoOctree();

  SbVec3i32 getTileOrigin( const LDM_TILE_ID_TYPE tileID )const;

  LDM_TILE_ID_TYPE parent( const LDM_TILE_ID_TYPE index )const;
  inline LDM_TILE_ID_TYPE children( const LDM_TILE_ID_TYPE index )const;

  LDM_TILE_ID_TYPE getNumTilesAtDepth( const int level )const;
  LDM_TILE_ID_TYPE getSumTilesAtDepth( const int level )const;

  int getNumFileIDsAtDepth( const int level )const;
  int getSumFileIDsAtDepth( const int level )const;
  int sizeofTiles( const int numTile, int tileDim )const;

  inline bool hasChildren(const SoLDMTileID& tileId) const;

  /** Return tile dimension for given level*/
  const SbVec3i32& getTileDimAtLevel(int l) const
  {
    return m_tileDimAtLevel[l];
  }

  /** Return tile dimension at the tile's level */
  const SbVec3i32& getTileDimAtLevel(const SoLDMTileID& id) const;

  /**
   * Return true if child is a child of parent
   */
  inline bool isChildOf(const SoLDMTileID& parent,
                        const SoLDMTileID& child) const;

  /**
   * Return true if child has parent in his family (father, grand-father, grand-grand-father...)
   */
  inline bool isInFamilyOf(const SoLDMTileID& parent, const SoLDMTileID& child) const;


  /** Returns the tile position in ijk tile space */
  SbVec3i32 getTileSpacePos( const LDM_TILE_ID_TYPE tileID , const int tileLevel ) const;

  /** Returns the TileID from the ijk position in tile space */
  SoLDMTileID getTileSpaceID( const SbVec3i32& cellPos, const int level) const;

  /**
   * Get neighbor tile of same resolution. location is a mask of #Location.
   * If neighbor does not exist (tileID in border of volume, so neighbor is 
   * outside for example), return -1.
   */
  SoLDMTileID getNeighborTile(const SoLDMTileID& tileID, int locationMask) const;

  SoLDMOctreeNode* operator [](const LDM_TILE_ID_TYPE tileID)const;
  SoLDMOctreeNode* operator ()(const int fileID)const;

  /**
   * Compute view sorted octant
   * @param order is filled with octant number sorted according to viewPoint (back to front)
   *        for example octantOrdered[0]==5 if octant[5] is the farthest  to viewpoint
   * @param viewVector is the view direction
   */
  inline static void getOrderedOctant(int octantOrder[8], const SbVec3f & viewVector);

  /* Allow the octree to use (or not) the optimisation made on thin volume */
  void useThinOptim(bool useThinVolumeOptim);

  /**
  * Return the maximum number of possible children
  * - in case of 3D dataset: returns 8
  * - in case of 2D dataset or thinVolume : returns 4
  */
  inline int getMaxNumChildren() const
  { return m_maxNumChildren; }

  /** internal helper (temporary solution to #35089
   * Initialize with the properties of the data set, but without allocation of data for each fileID.
   * it allows to do query on octree properties.
   */
  void commonInit( const SbVec3i32& dimension, const SbVec3i32& tileDim, const int border=0 );

  const SbVec3i32& getVolumeDimensions() const
  {
    return m_dimension;
  }

private:
  /** Compute tile dims for each levels */
  void fillTileDimAtLevel();

  /** Return tile's origin. Avoid a call to level if already computed */
  SbVec3i32 getTileOrigin( const LDM_TILE_ID_TYPE tileID, int tileLevel ) const;

  /**
   * Given a tileID, returns the corresponding fileID when m_fileIDs table
   * doesn't exist.
   */
  int getFileIDReverseTable( const SoLDMTileID& tileID )const;

  /**
   * Compute the octree level
   *
   * Note: Since OIV 9.0, the border parameter is no longer used.
   */
  static float computeLevelMax(unsigned int maxDim, unsigned int tileDim,
                               unsigned int border);

  static const int MAX_OCTREE_DEPTH;

  // precomputed data
  static const uint64_t s_numTilesAtDepth[LEVEL_MAX+1];
  static const uint64_t s_sumTilesAtDepth[LEVEL_MAX+2];

  // data depending on the volume dimension
  SbVec3i32         m_dimension;
  SbVec3i32         m_tileDim;
  int               m_border;
  int               m_levelMax;
  int               m_maxNumChildren;
  LDM_TILE_ID_TYPE  m_numTileIDs;
  int               m_numFileIDs;
  SoLDMOctreeNode  *m_tileIDs;
  int*              m_fileIDs;
  int               m_numFileIDsAtDepth[LEVEL_MAX+1];
  int               m_sumFileIDsAtDepth[LEVEL_MAX+1];
  bool              m_initIsDone;
  /** Tile's dimension (3D) */
  static const unsigned int TDimension;

  std::vector<SbVec3i32> m_tileDimAtLevel;

  /* Indicate if the octree can use thin volume optimization */
  bool m_useThinVolumeOptim;

};

/******************************************************************************/
size_t
SoLDMTopoOctree::getNumTileIDs() const
{
  return (size_t)(m_numTileIDs);
}

/******************************************************************************/
bool
SoLDMTopoOctree::hasChildren(const SoLDMTileID& tileId) const
{
  return children(tileId.getID()) < (LDM_TILE_ID_TYPE)getNumTileIDs() ;
}

/******************************************************************************/
int
SoLDMTopoOctree::level(const SoLDMTileID& id) const
{
  return level(id.getID());
}

/******************************************************************************/
bool
SoLDMTopoOctree::isChildOf(const SoLDMTileID& parent, const SoLDMTileID& child) const
{
  if ( child.getID() >= m_numTileIDs )
    return false;

  LDM_TILE_ID_TYPE firstChild = children(parent.getID());
  LDM_TILE_ID_TYPE childId = child.getID();
  return childId >= firstChild && childId < firstChild+8;
}

/******************************************************************************/
bool 
SoLDMTopoOctree::isInFamilyOf(const SoLDMTileID& _parent, const SoLDMTileID& child) const
{
  LDM_TILE_ID_TYPE childId = child.getID();
  LDM_TILE_ID_TYPE parentId = _parent.getID();
  if ( childId >= m_numTileIDs || childId < 0 )
    return false;
  else if ( childId == parentId)
    return true;
  else
  {
    LDM_TILE_ID_TYPE parentChildID = parent(childId);
    if ( parentChildID == parentId )
      return true;
    else
      return isInFamilyOf(_parent,parentChildID);
  }
  return false;
}

/******************************************************************************/
int
SoLDMTopoOctree::getFileID( const SoLDMTileID& tileID )const
{
  if ( tileID.getID()<0 || tileID.getID()>=m_numTileIDs)
    return -1;

  // if m_fileIDs exists, use it straight
  if ( m_fileIDs )
    return m_fileIDs[tileID.getID()] - 1;

  return getFileIDReverseTable(tileID);
}

/******************************************************************************/
LDM_TILE_ID_TYPE
SoLDMTopoOctree::children( const LDM_TILE_ID_TYPE index )const
{
  return (index << TDimension) + 1;
}

/******************************************************************************/
void
SoLDMTopoOctree::getOrderedOctant(int octantOrder[8], const SbVec3f & viewVector)
{
  int bitMask = 0;
  for ( int i = 0; i < 3; i++ )
    bitMask |= (viewVector[i] < 0) << i;
  for ( unsigned int i = 0; i < 8; i++ )
    octantOrder[i] = i^bitMask;
}

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif /* _SO_LDM_TOPO_OCTREE_ */


