/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : T.DUFOUR (May 2004)
**=======================================================================*/

#ifndef  _SO_VOLUME_HISTOGRAM_
#define  _SO_VOLUME_HISTOGRAM_

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#pragma warning(disable:4127)
#endif

#include <Inventor/SbDataType.h>
#include <Inventor/SbLinear.h>
#include <Inventor/helpers/SbDataTypeMacros.h>

#include <Inventor/STL/limits>
#include <Inventor/STL/vector>

class SoBufferObject;
class SoArithmetic;

/**
 * Volume Data Histogram.
 *
 * @ingroup LDM
 *
 * @DESCRIPTION
 *   This class allows building a histogram from a dataset or a portion of a dataset.
 *
 *   To implement the histogram computation, follow the steps below:
 *
 *   - Instantiate this class by passing the data type to the constructor.
 *
 *   - Pass the values by calling the method addValues() multiple times.
 *
 *   - Get the results using methods getHistoSize(), getHistogram(), getNumValues(), getValue(),
 *     getMinValue, getMaxValue(), getNumSignificantBits().
 *
 *   - Delete the object.
 *
 *  SoVolumeHistogram has different behavior depending on the datatype:
 *  - For integer data types: @BR
 *    - If sizeof(dataType) == 1, histogram will have 256 bins with a bin width of 1.
 *    - Else, histogram will have 65536 bins with a bin width of (Datatype_max - DataType_min) / 65536.
 *
 *  - For floating types: @BR
 *    The Histogram will have 65536 bins with a bin width of (rangeMax - rangeMin) / 65536, where rangeMin and
 *    rangeMax are specified with #setInputValueRange.
 *
 * @SEE_ALSO
 *    SoDataSet
 *
 *
 */
class LDM_API SoVolumeHistogram
{
public:
  /**
  * Constructor
  */
  SoVolumeHistogram(const SbDataType &dataType);

  /**
   * Destructor
   */
  ~SoVolumeHistogram();

  /**
  * For floating data type, specifies the range the histogram has to be computed in.
  * Must be called before any call to addValues().
  * Default values are -20000. 20000.
  */
  void setInputValueRange( double rangeMin, double rangeMax );

  /**
   * Setup the undefined value to ignore when computing the histogram.
   * [OIVNET-WRAPPER PROPERTY{UndefinedValue},SETTER,SINCE{9.7}]
   */
  void setUndefinedValue( const double undefinedValue)
  { m_undefinedValue=undefinedValue; }

  /**
   * return the current undefined value.
   * [OIVNET-WRAPPER PROPERTY{UndefinedValue},GETTER,SINCE{9.7}]
   */
  double getUndefinedValue() const
  { return m_undefinedValue; };

  /**
  * Returns the size of the histogram
  * [OIVNET-WRAPPER PROPERTY{HistoSize},GETTER,SINCE{9.7}]
  */
  size_t getHistoSize() const;

  /**
  * Returns a 64 bit integer pointer to the histogram data, 
  * the size of the histogram array is getHistoSize()
  * [OIVNET-WRAPPER-RETURN-TYPE ARRAY{HistoSize}]
  * [OIVJAVA-WRAPPER-RETURN-TYPE ARRAY{getHistoSize()}]
  */
  int64_t* getHistogram();

  /**
  * Returns the histogram value corresponding to the entry.
  * Equivalent to getHistogram()[entry]
  */
  int64_t getNumValues(size_t entry);

  /**
  * Returns the value corresponding to the entry for integer data types
  */
  int64_t getValue(size_t entry);

  /**
  * Returns the value corresponding to the entry for floating data types
  */
  double getValueD(size_t entry);

  /**
  * Returns the min value of the dataset for integer data types
  * [OIVNET-WRAPPER PROPERTY{MinValue},GETTER,SINCE{9.7}]
  */
  int64_t getMinValue() const;

  /**
  * Returns the max value of the dataset for integer data types
  * [OIVNET-WRAPPER PROPERTY{MaxValue},GETTER,SINCE{9.7}]
  */
  int64_t getMaxValue() const;

  /**
  * Returns the min value of the dataset for floating data types
  * [OIVNET-WRAPPER PROPERTY{MinValueD},GETTER,SINCE{9.7}]
  */
  double getMinValueD() const;

  /**
  * Returns the max value of the dataset for floating data types
  * [OIVNET-WRAPPER PROPERTY{MaxValueD},GETTER,SINCE{9.7}]
  */
  double getMaxValueD() const;

  /**
   * Returns the max value of the dataset according to the specified data type
   * [OIVNET-WRAPPER PROPERTY{Max},GETTER,SINCE{9.7}]
   */
  double getMax() const;

  /**
   * Returns the max value of the dataset according to the specified data type
   * [OIVNET-WRAPPER PROPERTY{Min},GETTER,SINCE{9.7}]
   */
  double getMin() const;

  /**
  * Returns the number of significant bits for the dataset, only relevant for integer data types
  * [OIVNET-WRAPPER PROPERTY{NumSignificantBits},GETTER,SINCE{9.7}]
  */
  int getNumSignificantBits() const;

  /**
  * Add to the histogram a 1D array of values.
  * NOTES: this API is limited to INT_MAX numValues to handle.
  */
  inline void addValues(SoBufferObject* values, const int numValues);

  /** 
   * Add to the histogram a 3D array of values.
   */
  inline void addValues(SoBufferObject* values, const SbVec3i32& arrayDim);

  /** 
   * Add to the histogram the values in the specified range inside the given array.
   */
  void addValues(SoBufferObject* values, const SbVec3i32& arrayDim, const SbBox3i32& range);

  /**
  * Set the histogram. Useful to get back scalar parameters such as min/max values and number of significant bits
  */
  void set(const std::vector<int64_t> & histo);

  /**
  * Set the histogram. Useful to get back scalar parameters such as min/max values and number of significant bits
  */
  void set(const std::vector<int64_t>& histo, const std::vector<double>& values);

  /**
   * Returns the histogram entry corresponding to the value.
   */
  template<typename T> inline size_t getEntry(T value) const;

  /**
   * Compute min max.
   * @param valuesBuffer Input buffer of values.
   * @param dataType Type of datas stored in valuesBuffer.
   * @param arrayDim Dimensions of valuesBuffer.
   * @param range Box in which min and max values are computed.
   * @param min Computed minimum value.
   * @param max Computed maximum value.
   */
  static void computeMinMax(
    SoBufferObject* valuesBuffer,
    const SbDataType& dataType,
    const SbVec3i32& arrayDim,
    const SbBox3i32& range,
    double& min,
    double& max
    );

  /**
   * Compute min max.
   * @param valuesBuffer Input buffer of values.
   * @param undefinedValue Discarded value in the computation of min and max.
   * @param dataType Type of datas stored in valuesBuffer.
   * @param arrayDim Dimensions of valuesBuffer.
   * @param range Box in which min and max values are computed.
   * @param min Computed minimum value.
   * @param max Computed maximum value.
   */
  static void computeMinMaxWithUndefined(
    SoBufferObject* valuesBuffer,
    const double undefinedValue,
    const SbDataType& dataType,
    const SbVec3i32& arrayDim,
    const SbBox3i32& range,
    double& min,
    double& max
    );

  /**
   * Compute min max.
   * @param valuesBuffer Input buffer of values.
   * @param dataType Type of datas stored in valuesBuffer.
   * @param arrayDim Dimensions of valuesBuffer.
   * @param minMax Vector of computed minimum and maximum values.
   */
  static void computeMinMax(
    void* valuesBuffer,
    const SbDataType& dataType,
    const SbVec3i32& arrayDim,
    SbVec2d& minMax);

  /**
   * Compute min max.
   * @param valuesBuffer Input buffer of values.
   * @param undefinedValue Discarded value in the computation of min and max.
   * @param dataType Type of datas stored in valuesBuffer.
   * @param arrayDim Dimensions of valuesBuffer.
   * @param minMax Vector of computed minimum and maximum values.
   */
  static void computeMinMaxWithUndefined(
    void* valuesBuffer,
    const double undefinedValue,
    const SbDataType& dataType,
    const SbVec3i32& arrayDim,
    SbVec2d& minMax
    );

SoINTERNAL public:
  /**
   * init/finish class. (Must be called before any Histogram creation)
   */
  static void initClass();
  static void exitClass();

private:
  /**
   * Returns the histogram entry corresponding to the value.
   */
  template<typename T> inline size_t getEntryInternal(T value) const;

  /**
   * Cast value according to m_dataType. This avoid ambiguity
   * like getEntry(5) which could be getEntry(int(5)) or
   * getEntry(unsigned int(5))
   */
  template<typename T> inline size_t getEntryCaster(T value) const;

  /** Compute number of bit needed to represent current [minValue, maxValue] */
  unsigned int computeNumSignificantBits() const;

  std::vector<int64_t> m_histo;  // histogram
  int64_t  m_valueMin;           // min integer value of the dataset for integer data types
  int64_t  m_valueMax;           // max integer value of the dataset for integer data types
  double   m_valueMinD;          // min integer value of the dataset for floating data types
  double   m_valueMaxD;          // max integer value of the dataset for floating data types
  unsigned int m_numSignificantBits; // number of significant bits, only relevant for integer data types

  SbDataType m_dataType;     // data type
  double m_rangeMin;         // for floating data type.
  double m_rangeMax;         // for floating data type.
  double m_undefinedValue;   // undefined Value.

  // makeStats template
  template <typename T>
  void makeStat(
    const void* values,
    const SbVec3i32& arrayDim,
    const SbBox3i32& range
    );

  static SoArithmetic* s_arithmeticInterface;
};

/*******************************************************************************/
void
SoVolumeHistogram::addValues(SoBufferObject* values, const int numValues)
{
  addValues(values, SbVec3i32(numValues, 1, 1), SbBox3i32(SbVec3i32(0, 0, 0), SbVec3i32(numValues, 1, 1)));
}

/*******************************************************************************/
void SoVolumeHistogram::addValues(SoBufferObject* values, const SbVec3i32& arrayDim)
{
  addValues(values, arrayDim, SbBox3i32(SbVec3i32(0,0,0),arrayDim));
}

/*******************************************************************************/
template<typename T> size_t
SoVolumeHistogram::getEntryInternal(T value) const
{
  assert(m_dataType.isInteger());
  assert(sizeof(T) <= 4);
  size_t offset = 0;
  size_t factor = 1;

  if ( std::numeric_limits<T>::is_signed )
    offset = (sizeof(T) >= 2) ? 0x8000 : 0x80;

  if ( sizeof(T) == 4 )
    factor = (size_t)0x10000;

  size_t entry = (size_t)(value/int64_t(factor)+int64_t(offset));

  return entry;
}

/*******************************************************************************/
template<typename T> size_t
SoVolumeHistogram::getEntryCaster(T value) const
{
  return getEntryInternal(T(value));
}

/*******************************************************************************/
template<typename T> size_t
SoVolumeHistogram::getEntry(T value) const
{
  size_t ret = 0;

  switch (m_dataType)
  {
  case SbDataType::UNSIGNED_SHORT:
    ret = getEntryInternal<unsigned short>((unsigned short)value);
    break;
  case SbDataType::UNSIGNED_INT32:
    ret = getEntryInternal<uint32_t>((uint32_t)value);
    break;
  case SbDataType::SIGNED_BYTE:
    ret = getEntryInternal<signed char>((signed char)value);
    break;
  case SbDataType::SIGNED_SHORT:
    ret = getEntryInternal<signed short>((signed short)value);
    break;
  case SbDataType::SIGNED_INT32:
    ret = getEntryInternal<int32_t>((int32_t)value);
    break;
  case SbDataType::FLOAT:
    ret = getEntryInternal<float>((float)value);
    break;
  case SbDataType::UNSIGNED_BYTE:
    ret = getEntryInternal<unsigned char>((unsigned char)value);
    break;
  case SbDataType::DOUBLE:
    ret = getEntryInternal<double>((double)value);
    break;
  default:
    assert(0);
    break;
  }

  return ret;
}

/*******************************************************************************/
template<> inline size_t
SoVolumeHistogram::getEntryInternal<float>(float value) const
{
  assert(m_dataType == SbDataType::FLOAT);
  if ( m_rangeMax == m_rangeMin )
    return 0;
  double factorDouble = 0.0f;
  factorDouble = 0xffff / (m_rangeMax - m_rangeMin);
  size_t entry = (size_t)((value-m_rangeMin)*factorDouble);

  return SbMathHelper::Clamp(entry, (size_t)0, (size_t)0xffff);
}

/*******************************************************************************/
template<> inline size_t
SoVolumeHistogram::getEntryInternal<double>(double value) const
{
  assert(m_dataType == SbDataType::DOUBLE);
  if ( m_rangeMax == m_rangeMin )
    return 0;
  double factorDouble = 0.0f;
  factorDouble = 0xffff / (m_rangeMax - m_rangeMin);
  size_t entry = (size_t)((value-m_rangeMin)*factorDouble);

  return SbMathHelper::Clamp(entry, (size_t)0, (size_t)0xffff);
}

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif /* _SO_VOLUME_HISTOGRAM_ */


