/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (Apr 2003)
**=======================================================================*/

#ifndef _SO_BASE_CONVERTER_
#define _SO_BASE_CONVERTER_

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif


#include <LDM/nodes/SoDataSet.h>
#include <Inventor/STL/vector>
#include <Inventor/SbBox.h>
#include <Inventor/SbPList.h>

#include <Inventor/threads/SbThreadMutex.h>

#include <LDM/readers/SoLDMReader.h>
#include <LDM/compressors/SoDataCompressor.h>

class SoLDMTopoOctree;
class ScJob;
class SoVolumeHistogram;
class SiBitFile;
class SoDataCompressInfo;
class SoPerfCounterManager;
class SoConverterParameters;

/**
 * @LDMEXT Abstract base class for converting data sets into LDM format.
 *
 * @ingroup LDMConverters
 *
 * @DESCRIPTION
 *
 * This is the abstract base class for classes that convert existing data sets
 * into the LDM (Large Data Management) format.
 *
 * SoConverter provides powerful features including:
 * - Command line parsing for easy creation of stand-alone converter programs.
 * - Checkpoint and restart for interrupted conversions.
 * - Multiple subsampling algorithms for creating low resolution tiles.
 * - Automatic loading of any supported data format (see SoVolumeData).
 * - Conversion of any data format by providing a custom reader (see SoVolumeReader).
 * - Special handling to minimize disk space for "thin volumes" where one dimension
 *   is smaller than the tile size. (Currently only implemented for Z dimension.)
 * - Optional data compression to reduce disk space.
 * - Optional data type conversion, e.g. float to byte.
 *
 * Information common to all converter classes is included here including
 * command line, incomplete conversion, compression
 * and subsampling.  Normally you will use, for example, the derived class
 * SoVolumeConverter to convert volume data.  You can also subclass from
 * SoVolumeConverter to use a custom volume reader, provide a custom
 * subsampling method, etc.
 *
 * @B Command Line @b
 *
  * When used as a command line application the string should have the following form:
  *
  * inputFilename [-t tiledim] [-o filename] [-O [filename]] [-m memsize] [-h] [-w B|L]
  *               [-f u8|u16|u32|s8|s16|s32|f [-r min max]] [-q] [-D] [-P portnum] [-c type] [-l level]
  *               [-C] [-s [0..N]] [-u value] [-verticalFlip]
  *
  * where:
  *
  *  @TABLE_1B
  * @TR-t tiledim @TD Dimension of tiles. Must be a power of two. Default is 128, meaning 128x128x128.
  *
  * @TR-b bordersize @TD DEPRECATED Since OIV9.0, this parameter will be ignored.
  *
  * @TR-o filename @TD Name of created XML header ldm file. Default is inputFilename with extension .ldm.
  *
  * @TR-O filename   @TD Name of the file containing the data.
  *               If filename is omitted, the default filename is the same
  *               as the output filename specified by the -o option, with extension .dat instead.
  *
  * @TR-m memsize    @TD Maximum size of memory to use in MB. Default is 512 MB.
  *
  * @TR-h            @TD Output the header only.
  *
  * @TR-w wordfmt    @TD Indicates the target machine word format is Big Endian (B) or Little Endian (L).
  *               Default is the current machine word format.
  *
  * @TR-f datafmt    @TD Indicates the output data format with * = 8,16,32. u*:unsigned integer, s*:signed integer, f:float.
  *               If the output data format is float, the values are mapped to the range [0-1].
  *               Default is the input data format.
  * @TR-r min max    @TD Indicates the input data range to map to the new data format specified by the -f option.
  *               If not specified, and input type is float and ouput type is integer, the values are simply cast into the output data type.
  * @TR-q            @TD Quiet mode.
  *
  * @TR-c type @TD Type of compression used. See also "-l level". By default the following compression types are available: 
  *                 - gzip: @BR
  *                   This is a lossless compression and can be used on any type of data.
  *                   See SoGzipDataCompressor.
  *
  *                 - jpeg: @BR
  *                   This is a lossy compression. See SoJpegDataCompressor. @BR
  *                   Limitation: Can only be used on 8-bit scalar data sets.
  *
  * @TR-l level @TD Level of compression used. See also "-c type". The meaning of this parameter depends on the type of compression used:
  *
  *                 - gzip: @BR
  *                   Lossless compression.
  *                   Level must be between 0 and 9: 0 gives no compression, 9 gives best compression but slower speed.
  *                   Default is 3.
  *
  *                 - jpeg: @BR
  *                   Lossy compression.
  *                   Level must be between 0 and 100: 0 gives best compression but most loss of data,
  *                   100 gives minimum compression and minimum data alteration. Default is 85.
  *
  * @TR-C @TD Data integrity check. When compression is enabled using -c, adds a data integrity check value
  *                 to all compressed tiles.
  *                         If the environment variable LDM_COMPRESS_CHECK_CRC32 is set, data values will be checked
  *                         each time they are read from disk.
  * @TR-s 0..n @TD Specifies the algorithm used to build low resolution tiles. Available options are :
  *                 - 0 to use decimation algorithm (one voxel out of two).
  *                 - n to use weighted average algorithm : voxels of tile of resolution N+1 are
  *                   built from the average of the 6 neighbors from resolution N and the current
  *                   voxel value weighted by n.
  *                 If not specified, by default the decimation algorithm is used (n=0).
  * @TR-u value @TD Undefined value. The given value will marked as undefined in the LDM file.
  *                 This is currently used by SoHeightFieldRender (vertices with undefined values
  *                 are not rendered).
  *
  * @TR-rgba @TD Specifies that output will be unsigned int32 RGBA values.
  *
  * @TR-noHistogram @TD Disable histogram computation (conversion will be faster).
  *
  * @TR-B value @TD      Define threshold value to generate a bitset dataset. @BR
  *               Any value in the dataset less than or equal to the threshold will generate a zero bit.
  *               Values greater than threshold will generate a one bit.
  *
  * @TR-verticalFlip @TD Flip the output on Y axis.
  *
  *    @TABLE_END
  *
  * The return value may be one of the following:@BR
  *
  *    @TABLE_1B
  *       @TR CVT_FINISHED_WITH_WARNINGS   @TD Some warnings occured but task complete
  *       @TR CVT_NO_ERROR                 @TD No problem, task complete
  *       @TR CVT_INPUT_PARAMS_PROBLEM     @TD Input parameters problem: input filename missing, bad parameter
  * value, or invalid option
  *       @TR CVT_FILE_EXT_UNKNOWN         @TD Unknown file type (file extension)
  *       @TR CVT_CANT_OPEN_INPUT_FILE     @TD Could not open input file (does not exist)
  *       @TR CVT_CANT_CREATE_DATA_FILE    @TD Could not create data file
  *       @TR CVT_CANT_CREATE_HEADER_FILE  @TD Could not create header file
  *       @TR CVT_NOT_ENOUGH_DISK_SPACE    @TD Could not allocate disk space for the data file
  *       @TR CVT_ABORTED                  @TD Conversion has been aborted by the application, task not complete
  *       @TR CVT_NO_NODE                  @TD Distributed converter cannot be used as the license does not allow any client node
  *       @TR CVT_NO_LDM_LICENSE           @TD No LDM license
  *    @TABLE_END
  *
 *
 * Incomplete Conversion
 *
 * The converter saves the progress of the conversion in case the user wishes to
 * abort or start visualizing the converted data or if the program crashes for any reason.
 * In this case, the LDM files (header and data) are coupled with a completion file
 * (name of the header file with an extension of .fcp).
 * When restarting the converter, if the header file points to a .fcp file
 * (XML tag \<completionFilename\>), the conversion will restart where it previously ended.
 * When visualizing an incompletely converted file, it is possible to know where
 * data actually exists by turning on the topology outlines (see SoVolumeRendering). Red
 * outlines indicate actual data.
 *
 * In order to restart an incomplete conversion, the header file must be in synch with
 * the completion file.
 * For this reason when writing the pair of header/completion files,
 * the converter first backs up the pair of files before overwriting them (.ldm and .fcp with .bck extension).
 * If the header file is lost or its size is less than the size of the backed up header file,
 * then you can rename the backed up completion and header files to restart the conversion.
 *
 * @EXAMPLE
 * A basic command line converter program would look like this:
 * \if_cpp
 * \code
 *   int main( int argc, char ** argv)
 *   {
 *     SoVolumeRendering::init();
 *
 *     // Create parameter object from command line args
 *     SoConverterParameters* pParameters = SoConverterParameters::create(argc,argv);
 *
 *     // Start converter 
 *     SoVolumeConverter converter;
 *     int ret = converter.convert( pParameters );
 *
 *     delete pParameters;
 *     SoVolumeRendering::finish();
 *     return ret;
 *   }
 * \endcode
 * \endif
 * \if_dotnet
 * \code
 *   public class CommandLine
 *   {
 *     public static void Main(string[] args)
 *     {
 *       // Create parameter object from command line args
 *       SoConverterParameters parameters = 
 *           SoConverterParameters.Create(args.Length, args);
 *
 *       // Start converter 
 *       SoVolumeConverter converter = new SoVolumeConverter();
 *       int result = converter.Convert( parameters );
 *     }
 *   }
 * \endcode
 * \endif
 * \if_java
 * \code
 *   public class Main
 *   {
 *     public static void main(String[] args)
 *     {
 *       // Create parameter object from command line args
 *       SoConverterParameters parameters = SoConverterParameters.create( args );
 *
 *       // Start converter 
 *       SoVolumeConverter converter = new SoVolumeConverter();
 *       int result = converter.convert( parameters );
 *     }
 *   } 
 * \endcode
 * \endif
 *
 * @EXAMPLE
 * You can also invoke the converter from an application by creating an instance
 * of SoConverterParameters and setting the desired parameters.  For example to
 * convert an input file (in a format supported by VolumeViz) to LDM format with
 * compression enabled:
 *   \par
 *   \if_cpp
 *   \code
 *     // Create parameter object with input file and desired options
 *     SoConverterParameters* parameters = new SoConverterParameters();
 *     parameters->setInputFileName( "filename.sgy" );
 *     parameters->setTileDim( 128 );
 *
 *     // Start converter
 *     SoVolumeConverter converter;
 *     int result = converter.convert( parameters );
 *   \endcode
 *   See the example programs in $OIVHOME/examples/source/VolumeViz/Converters.
 *   \endif
 *   \if_dotnet
 *   \code
 *     // Create parameter object with input file and desired options
 *     SoConverterParameters parameters = new SoConverterParameters();
 *     parameters.SetInputFileName( "filename.sgy" );
 *     parameters.SetTileDim( 128 );
 *
 *     // Start converter
 *     SoVolumeConverter converter = new SoVolumeConverter();
 *     int result = converter.Convert( parameters );
 *   \endcode
 *   \endif
 *   \if_java
 *   \code
 *     // Create parameter object with input file and desired options
 *     SoConverterParameters parameters = new SoConverterParameters();
 *     parameters.setInputFileName( "filename.sgy" );
 *     parameters.setTileDim( 128 );
 *
 *     // Start converter 
 *     SoVolumeConverter converter = new SoVolumeConverter();
 *     int result = converter.convert( parameters );            
 *   \endcode
 *   \endif
 *
 * @SEE_ALSO
 *    SoConverterParameters,
 *    SoVolumeConverter,
 *    SoVolumeReader,
 *    SoDataCompressor,
 *    SoJpegDataCompressor,
 *    SoGzipDataCompressor
 *
 *
 */
class LDM_API SoConverter {

public:

  /**
   * Destructor.
   */
  virtual ~SoConverter();

  /**
   * Converter errors.
   */
  enum ConverterError{
    /** Finished with warnings */
    CVT_FINISHED_WITH_WARNINGS  = 1,
    /** No error */
    CVT_NO_ERROR                =  0,
    /** Input params problems */
    CVT_INPUT_PARAMS_PROBLEM    = -1,
    /** Unknown file extension */
    CVT_FILE_EXT_UNKNOWN        = -2,
    /** Unable to open file */
    CVT_CANT_OPEN_INPUT_FILE    = -3,
    /** Unable to create data file */
    CVT_CANT_CREATE_DATA_FILE   = -4,
    /** Unable to create header file */
    CVT_CANT_CREATE_HEADER_FILE = -5,
    /** No enough disk space */
    CVT_NOT_ENOUGH_DISK_SPACE   = -6,
    /** Unable to read input file */
    CVT_CANT_READ_INPUT_FILE    = -7,
    /** Conversion aborted */
    CVT_ABORTED                 = -9,
    /** Uninitialized converter */
    CVT_NOT_INITIALIZED         = -10,
    /** No node */
    CVT_NO_NODE                 = -998,
    /** No LDM license */
    CVT_NO_LDM_LICENSE          = -999
  };

  /**
   * Main converter function. @BR
   * Call this method to launch the converter. Conversion parameters may be
   * specified using an instance of SoConverterParameters. Returns
   * 0 if conversion was successful, else returns one of the CVT_ error codes
   * above.
   *
   */
  int convert( SoConverterParameters* parameters = NULL);

  /**
   * Specifies the callback that will be called when the LDM file header is generated.
   * This allows the application to add custom XML tags.
   * The custom tags can be read back using the method SoVRLdmFileReader::getXmlTag().
   */
  void setXmlCallback( void (*xmlCB)( FILE*, void * userData ), void * userData );

#if SoDEPRECATED_BEGIN(9000)
  /**
   * [OIV-WRAPPER-ARG NO_WRAP{((argv != NULL? argv->Length: 0))},IN]
   * [OIVJAVA-WRAPPER-RETURN-TYPE WRAP_AS{SoConverter::ConverterError}]
   */
  SoDEPRECATED_METHOD(9000,"Use convert( SoConverterParameters* parameters = NULL) instead.")
  int convert( int argc, char ** argv);

  SoDEPRECATED_METHOD(9000,"Use convert( SoConverterParameters* parameters = NULL) instead.")
  int convert( const SbStringList& arguments );

#endif /** @DEPRECATED_END */

SoINTERNAL public:

  enum MessageType{
    INIT = 1,
    BUILD_SLAB,
    BUILD_LEVEL,
    CLEAR
  };

  char*     m_slabBuffer;
  SbVec3i32 m_slabBufferDim;
  int64_t   m_subSliceSize;

  //statistics
  SoVolumeHistogram* m_stat;
  double             m_min, m_max;
  std::vector<int64_t>    m_numValues;
  std::vector<double>     m_values;

  int                m_numTilesGenerated;
  float              m_progress;

  struct Stat{
    double  value;
    int64_t numValues;
  };

  void checkProgress();


  //data information
  int                    m_dataSize;

  //routines
  ConverterError init();

  char* allocateSlabBuffer(int& _numTilesOnYAxis, SbVec3i32& _slabBufferDim, int64_t& _subSliceSize);
  int   buildSlab2Level(SbBox3i32& slabBox, char*& slabBufferPtr, void* parentTile, void** octantTile,SbVec3i32& slabBufferDim, int64_t subSliceSize, std::vector<int>& fileIDsBuilt);
  void  buildLevel(std::vector<int>&, int, int fromFileID = -1, int toFileID = -1);
  int   initializeFiles( int argc, char ** argv );
  int   initializeFiles( SoConverterParameters* parameters );

  int   writeSubVolume( const SbBox3i32 & subVolume, const void * data, SbBool doLowerResolution = TRUE, SbBool doOverlappingTiles = TRUE );
  int   writeTile( const SoLDMTileID & tileID, void * data );

  int   writeSubVolume( const SbBox3i32 & subVolume, SoBufferObject* data, SbBool doLowerResolution = TRUE, SbBool doOverlappingTiles = TRUE );
  int   writeTile( const SoLDMTileID & tileID, SoBufferObject* data );

  const SoLDMTopoOctree* getTopo() { return m_LDMTopo; };
  int   finalizeFiles( int retCode = 0 );
  int   buildLevels( int fromLevel, int numLevels );

  void * m_fout;
  SoConverterParameters* getParameters();
  void updateFileCompletion(std::vector<int>& fileIDs);

  std::vector<SoDataSet::DatumElement> m_datum;
  bool                   m_isDataFloat;
  bool                   m_isDataSigned;
  size_t                  m_tileSize;
  int                    m_numSigBits;

protected: //PROTECTED_TO_DOCUMENT
  SoVolumeReader* m_reader;

  /** Constructor. */
  SoConverter();

  //// TO REDEFINE WHEN A CLIENT MODULE INHERITS FROM THE CONVERTER://///
  virtual void getSubSlice( const SbBox2i32& subSlice, int sliceNumber, void * data ) = 0;

  //error returned : 0 = ok, 1 = pb = return
  virtual SbBox3f getSize(int& error) = 0;
  virtual std::vector<SoDataSet::DatumElement> getDatum(int& error) = 0;
  virtual SbVec3i32 getDimension(int& error) = 0;
  virtual int getRange( double & rangeMin, double & rangeMax) = 0;
  virtual SoVolumeHistogram* getHistogram();
  virtual std::vector<SbVec2d>& getTileMinMax(int numTiles);
  virtual void closeInputFileHandles();

  /**
   * Method used to downsample a tile. The input are the eight higher resolution tiles used to create the
   * one lower resolution tile (parent).
   * By default, the converter simply takes one voxel out of two in the higher resolution tile to create the
   * parent tile. This method can be overriden by a custom converter to use a different approach.
   * Inputs are:
   * tileDim is the size of the tile to create in number of voxels.
   * dataType is the type of the data (the value corresponds to the SoDataSet::DataType enum).
   * border only a border of 0 is supported since Open Inventor 9.0.
   * octantTile are the values of the 8 higher resolution tiles.
   * octantExists let the user know if a given octant is free of data (the data is an asymmetric volume).
   * Output is the parentTile buffer.
   */
  virtual void  sampleTile ( const SbVec3i32& tileDim, int border, const void* const octantTile[8], const int octantExists[8], void* parentTile ) = 0;
  virtual void  buildTile  ( SbBox3i32& slabBox, SbVec3i32& slabBufferDim, void* slabBuffer, SbVec3i32& tileOrigin, void* tileBuffer );
          void  buildTile  ( const SbBox3i32& subVolume, const void* subVolumeBuffer, const SbBox3i32& tilePos, void* tileData, const SbBox3i32& intersection );
  virtual void  outputHeader( FILE* ) {};

#if SoDEPRECATED_BEGIN(2023200)

  SoDEPRECATED_METHOD(2023.2, "Use outputTileBefore(int fileTileIndex, size_t dataSize, void *data, bool dataInverted) instead.")
  virtual void  outputTileBefore(int SO_UNUSED_PARAM(fileTileIndex), int SO_UNUSED_PARAM(dataSize),
                                 void* SO_UNUSED_PARAM(data), bool SO_UNUSED_PARAM(dataInverted))
  {}

#endif /** @DEPRECATED_END */

  virtual void  outputTileBefore(int SO_UNUSED_PARAM(fileTileIndex), size_t SO_UNUSED_PARAM(dataSize),
                                 void* SO_UNUSED_PARAM(data), bool SO_UNUSED_PARAM(dataInverted))
  {}

  virtual void  startConversion(){};
  virtual void  endConversion(){};

  virtual bool updateDataFile() { return true; }
  virtual int getSpecificOffset(bool aligned = true ) { return !aligned; }      // dummy return to avoid compilation warning
  virtual SbString getOriginalFilename() const;
  virtual void setOriginalFileName(const SbString& originalFileName);


  //// END LIST OF FUNC TO REDEFINE WHEN A CLIENT MODULE INHERITS FROM THE CONVERTER://///

  /** Abort */
  enum Abort {
    /** */
    CVT_CONTINUE,
    /** */
    CVT_ABORT
  };

  /**
  * Define available algorithm available for downsampling process
  */
  enum SampleType
  {
    /**
     * Decimation : Only one voxel out of two
     */
    DECIMATION,
    /**
     * Average : Voxel value in parent tile is an average of Neighbor voxel in child tile
     */
    AVERAGE
  };

  /**
   * This method is called each time a tile of data is generated.
   * numTilesGenerated represents the number of tiles already generated and numTilesToGenerate is
   * the total number of tiles to generate. So the quotient of these two numbers represents the progress
   * of the task.
   * If this method returns ABORT, the task is stopped. By default it returns CONTINUE.
   */
  virtual Abort progress( int numTilesGenerated, int numTilesToGenerate );

  ConverterError  buildHeader( const SbString&, bool conversionDone = false);

  virtual void outputDataType(FILE* fout);

  SbBox3f          m_volumeSize;

  void inputTile( int fileTileIndex, size_t dataSize, void *data, bool i = false );
  void printTime();
  int buildLevelMax();
  void outputTile( int fileTileIndex, size_t dataSize, void*, bool i = false );

  void printCompressionInfos();

  void (*m_xmlCB)( FILE*, void* );
  void  *m_xmlCBUserData;
  SoLDMTopoOctree *m_LDMTopo;

  //files info
  int64_t   m_filesize;//final data file size
  int64_t   m_compressedFilesize;//final compressed file size
  SbString  m_fileCompletion; //file completion
  SbString  m_fileCompletionName;
  SbString m_fileExt;

  //data info
  int              m_pid;
  int              m_levelMax;
  bool             m_thinDataSet;
  SbVec3i32        m_volumeDimension;


  // progress
  int              m_numTilesToGenerate;
  bool             m_abort;
  int  getNumTileToGenerate(SbBox3i32& slabBox);
  SiBitFile* m_bitFile;

  //Compression
  SoDataCompressor *m_ldmCompressor;
  int64_t m_sizeOfTileIndex;
  char m_compressionType[256];
  SoDataCompressInfo* m_lastWrittenCompressInfo;


  bool       m_fromInitializeFiles;

  //file completion
  void writeTileComplete(int fileID, bool writeHeader = false);

  //abort function
  void abortConversion();

  void readSlab(SbBox3i32& slabBox, char*& slabBufferPtr,int64_t subSliceSize);

  // Define the algorithm used for downsampling
  SampleType m_sampleType;


  // type dependent vars/macros
  static const char *m_dataTypeStr[];
  static const char *m_dataTypeXml[];
  int              m_currentWordFormat;

  // Performance counter manager
  SoPerfCounterManager *m_perfManager;
  SoConverterParameters* m_parameters;
  SbString m_originalFileName;

  virtual void adjustParameters() {};
  virtual void handleCustomTags(FILE*) {};

    /** One minMax for each file ids */
  std::vector<SbVec2d> m_tileMinMax;

  /**
   * Return TRUE if converter can do a fast update of the LDM file
   * (ie: converting a ldm file with borders to a borderless one)
   */
  bool isFastUpdate();

  SoConverter::ConverterError m_statusAdjustParameters;

private:
  bool m_userDefinedParameters;
  int64_t getCrc32(void* data, size_t dataSize, const SoDataCompressor::TileInfo& ti);
  void usage();
  template <typename T> void fillTile(void* ptrDst, int size, void* value );

  template <typename T> void fillTileWithSingleValue(void* ptrDst, int size, T value );

  /** Write min max infos into the ldm file */
  SoConverter::ConverterError outputMinMaxInfos(FILE* fout);

  /** Compute minMax of low res by combining min max of children*/
  void computeLowResMinMax(const SoLDMTileID& tileId);

  bool m_bComplFileAlreadyExists;

  /** Use readtile from SoVRLdmFileBorderReader to generate a border less tile */
  void buildTileFast(int fileID, void* tileData, const SbVec3i32& tileOrigin);

  static SbThreadMutex s_classMutex;

  /** Set by environment variable LDM_CONVERTER_FAST_UPDATE */
  static bool s_doFastUpdate;
};

inline SoConverterParameters*
SoConverter::getParameters()
{
  return m_parameters;
}

inline SbString 
SoConverter::getOriginalFilename() const
{
  return m_originalFileName;
}

inline void
SoConverter::setOriginalFileName(const SbString& originalFilename) 
{
  m_originalFileName = originalFilename;
}

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif //_SO_BASE_CONVERTER_


