/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : P. ESTRADE (Jul 2002)
** Modified by : T.DUFOUR (MMM yyyy)
**=======================================================================*/

/* include files */
#ifndef  _SO_ROI_MANIP_
#define  _SO_ROI_MANIP_

#include <Inventor/draggers/SoDragger.h>
#include <LDM/nodes/SoROI.h>

class SoOneShotSensor;
class SoState;
class SoCallback;

/*----------------------------------------------------------------------------*/

/**
  * @LDMEXT Manipulator to transform an ROI (Region of Interest).
  * 
  * @ingroup LDMManips
  * 
  * @DESCRIPTION
  *   This class defines a manipulator to transform an ROI (Region of Interest).
  *   The manipulator is a composite of SoROI and an SoTabBoxDragger that allows 
  *   the user to interactively move and resize the ROI.
  *
  *   To define a simple ROI, set the limits of the ROI in the SoROI#box field
  *   (and do not set the SoROI#subVolume field). The same result is obtained by
  *   setting the 'box' and 'subVolume' fields to the same value, but this is not necessary.
  *   But note that, like SoROI, the default ROI box is not automatically the
  *   full volume. You must initialize the ROI box to something, for example
  *   the volume dimensions minus one.  See SoROI for more details.
  *
  *   Note: The edge color of the embedded dragger is modified, i.e. is not the
  *   same as the default edge color in SoTabBoxDragger.
  *
  * @EXAMPLE
  *  \if_cpp
  *    \code
  *      // Initialize ROI box to full volume
  *      SoVolumeData* volData = new SoVolumeData();
  *      . . .
  *      SoROIManip*   roiManip = new SoROIManip();
  *      roiManip->box.setValue( SbVec3i32(0,0,0), volData->data.getSize() - SbVec3i32(1,1,1) );
  *      root->addChild( roiManip );
  *    \endcode
  *   \endif
  *   \if_dotnet
  *    \code
  *      // Initialize ROI box to full volume
  *      SoVolumeData volData = new SoVolumeData();
  *      . . .
  *      SoROIManip roiManip = new SoROIManip();
  *      roiManip.box.SetValue( new SbVec3i32(0, 0, 0), volData.data.GetSize() - new SbVec3i32(1, 1, 1));
  *      root.AddChild( roiManip );
  *    \endcode
  *   \endif
  *   \if_java
  *     \code
  *       // Initialize ROI box to full volume
  *       SoVolumeData volData = new SoVolumeData();
  *       . . .
  *       SoROIManip roiManip = new SoROIManip();
  *       roiManip.box.setValue( new SbVec3i32(0, 0, 0), volData.data.getSize().minus( new SbVec3i32(1, 1, 1)));
  *       root.addChild( roiManip );
  *     \endcode
  *   \endif
  * 
  * @FILE_FORMAT_DEFAULT
  *    ROIManip {
  *    @TABLE_FILE_FORMAT
  *		  @TR subVolume	  @TD 0 0 0 0 0 0
  *       @TR box         @TD 0 0 0 1 1 1
  *       @TR flags       @TD 7
  *		  @TR relative	  @TD FALSE
  *       @TR boxOn		  @TD TRUE
  *       @TR constrained @TD FALSE
  *    @TABLE_END
  *    }
  * 
  * @SEE_ALSO
  *    SoROI,
  *    SoDragger
  * 
  * 
  */
class LDM_API SoROIManip : public SoROI {
  
  SO_NODE_HEADER(SoROIManip) ;
  
public:
  /**
   * Constructor.
   */
  SoROIManip() ;
  
  // Fields
  /**
   * Chooses between the 'box' (TRUE) or the 'subVolume' (FALSE) field in the SoROI.
   * Default value is TRUE.
   */
  SoSFBool boxOn;
  
  /**
   * Constrains the dragger to fit in the data volume. Default value is FALSE.
   */
  SoSFBool constrained;

  /**
   * Returns the dragger node being employed by this manip.
   */
  SoDragger *getDragger() ;
  
  /**
  * Replaces the tail of the path with this manipulator. The tail of the path must
  * be an SoROI node (or subclass thereof). If the path has a nodekit,
  * this will try to use setPart() to insert the manipulator. Otherwise, the
  * manipulator requires that the next-to-last node in the path chain be a group.
  * 
  * The field values from the ROI node will be copied to this
  * manipulator, and the ROI node will be replaced.
  * 
  *\if_cpp
  * The manipulator will not call ref() on the node it is replacing.
  *\endif
  * The old node will disappear if it has no references other than from the input path p and
  * its parent, since this manipulator will be replacing it in both of those places.
  * Nor will the manipulator make any changes to field connections of the old node.
  * The calling process is thus responsible for keeping track of its own nodes and
  * field connections.
  */
  SbBool replaceNode(SoPath *p) ;
  
  /**
  * Replaces the tail of the path, which must be this manipulator, with the given
  * SoROI node. If the path has a nodekit, this will try to use
  * setPart() to insert the new node. Otherwise, the manipulator requires
  * that the next-to-last node in the path chain be a group.
  * 
  * The field values from the manipulator will be copied to the ROI
  * node, and the manipulator will be replaced.
  * 
  *\if_cpp
  * The manipulator will not call ref() or unref() on the node which is
  * replacing it, nor will it make any changes to field connections.
  *\else
  * It will not make any changes to field connections.
  *\endif
  * The calling process is thus responsible for keeping track of its own nodes and field
  * connections.
  */
  SbBool replaceManip(SoPath *p, SoROI *newOne) const;
  
SoEXTENDER public:
  // These functions implement all actions for this manip.
  // They first traverse the children, then use the 
  // SoROI version of the actions. They traverse first 
  // so that the SoROI will affect objects which 
  // follow it in the tree, but not the dragger-child.
  /** @copydoc SoNode::doAction */
  virtual void doAction(SoAction *action);
  /** @copydoc SoNode::callback */
  virtual void callback(SoCallbackAction *action);
  /** @copydoc SoNode::GLRender */
  virtual void GLRender(SoGLRenderAction *action);
  /** @copydoc SoNode::getBoundingBox */
  virtual void getBoundingBox(SoGetBoundingBoxAction *action);
  /** @copydoc SoNode::getMatrix */
  virtual void getMatrix(SoGetMatrixAction *action);
  /** @copydoc SoNode::handleEvent */
  virtual void handleEvent(SoHandleEventAction *action);
  /** @copydoc SoNode::pick */
  virtual void pick(SoPickAction *action);
  /** @copydoc SoNode::search */
  virtual void search(SoSearchAction *action);
  /** @copydoc SoNode::write */
  virtual void write( SoWriteAction *action );
  
SoINTERNAL public:
  // call this after SoInteraction::init();
  static void initClass();
  static void exitClass();
 
  virtual SoChildList *getChildren() const;
  
protected:
  // When the dragger moves, this interprets the fields modifications 
  // of the dragger and sets the fields of this SoROI accordingly
  static void valueChangedCB(void *,SoDragger *);
  
  // When one of the field of this node changes, moves the
  // child-dragger to a new location, if necessary.
  SoFieldSensor *m_boxOnSensor;
  SoFieldSensor *m_boxFieldSensor;
  SoFieldSensor *m_subVolumeFieldSensor;
  SoFieldSensor *m_relativeFieldSensor; 
  SoOneShotSensor *m_wasInitSensor; 
  
  static void fieldSensorCB(void *, SoSensor *);
  static void oneShotSensorCB(void *, SoSensor *);
  
  // Establishes the given dragger as the new child-dragger
  void setDragger(SoDragger *newDragger);
  
  // The hidden children.
  SoChildList *m_children;
  virtual SbBool readInstance(SoInput *in, unsigned short flags);
  virtual SbBool readChildren(SoInput *in);
  
  // Destructor
  virtual ~SoROIManip();
  
private:
  SbVec3i32 m_dimension;
  SbBox3f m_size;
  SbBool  m_init;
  SbVec3i32 m_boxmin, m_boxmax;

  SoCallback* m_dataSetMatrixCBNode;
  bool m_useDataSetMatrix;
  SbMatrix m_savedDataSetMatrix;
  
  static void dataToGeometry( const SbVec3i32 &, const SbVec3i32 &, SbVec3f &, SbVec3f &, const SoROIManip * );
  static void geometryToData( const SbVec3f &, const SbVec3f &, SbVec3i32 &, SbVec3i32 &, const SoROIManip *, bool );

  int getNumChildren() const { return (m_children->getLength()); }
  void initBox(SoAction* action);

  bool hasVolumeDataSizeChanged(SoState *state);

  bool pushDataSetModelMatrix(SoAction* action, SoNode* node);
} ;
/*----------------------------------------------------------------------------*/

#endif /* _SO_ROI_MANIP_  */


