/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/


#ifndef LDM_DATA_TRANSFROM_H
#define LDM_DATA_TRANSFROM_H

#include <LDM/SoLDMLargeDataManagement.h>

#include <Inventor/fields/SoFieldContainer.h>
#include <Inventor/fields/SoSFVec3i32.h>
#include <Inventor/SbLinear.h>
#include <Inventor/SbBox.h>

class SoDataSet;
class SoBufferObject;

/**
 * @LDMEXT LDM data transform object.
 *
 * @ingroup LDMNodes
 *
 * @DESCRIPTION
 * This object defines a transform function to be applied to each LDM tile
 * after it is loaded (returned from the volume reader) but before it is
 * stored in system memory (LDM data cache).
 * This allows you to modify the original data (for example, scaling or
 * filtering) before it is displayed.  It can also be used to create
 * multiple data sets in memory from a single data set on disk.
 *
 * Derive a new class from this one and implement the #transformFunction method.
 * Create an instance of the new class and set in the dataTransform field of
 * the SoDataSet node.
 *
 * Notes:
 *   - Applies to all volume data. @BR
 *     A data transform can be applied to @I any@i type of volume data, not
 *     just data loaded from a .ldm format file.  Internally all volume data
 *     sets are managed by LDM as tiles.
 *
 *   - SoVolumeTransform @BR
 *     The SoVolumeTransform node also allows a
 *     transform function to be applied to each LDM tile, but the transform
 *     is applied immediately before the tile is loaded onto the graphics board.
 *     SoVolumeTransform does allow the Data Access API to be used in the
 *     transform function.
 *
 *   - SoDataCompositor @BR
 *     The SoDataCompositor node allows a transform function
 *     to be applied in the special case where multiple volumes should be
 *     combined into a single volume.
 *
 * @SEE_ALSO
 * SoDataSet, SoVolumeData, SoVolumeTransform
 *
 * [OIVJAVA-WRAPPER-CLASS DERIVABLE]
 */
class LDM_API SoLDMDataTransform : public SoFieldContainer
{
  SO_FIELDCONTAINER_ABSTRACT_HEADER(SoLDMDataTransform);
public:

  /**
   * Data transform function. @BR
   * This function allows a tile of data to be transformed after it is loaded, but before
   * it is stored in main memory.  It is not currently possible to access other tiles of
   * data (for example using the data access API) from this function.  Note that the
   * function will be called from LDM data loader threads, so multiple threads may be
   * executing in this function at the same time (on different tiles).
   * Inputs are:
   *  - The associated data set object the function is calling for.
   *    This allows retrieving information such as data type (ds->getDataType()).
   *  - The dimensions of the tile to transform.
   *    This defines the size of the buffer and is the same for every tile in a dataset.
   *    However tiles on the "outside" of the dataset may be partial tiles and contain
   *    less than the full number of actual data values.
   *  - A buffer containing the tile data to transform.
   *    The data should be modified "in place" in this buffer.
   *  - The position and extent of the tile in data space (voxel coordinates).
   *    For lower resolution tiles (level > 0) the extent of the tile will be larger
   *    than the dimensions of the tile (number of values in the tile).
   *  - The tile resolution level.  Level 0 is full resolution data.
   *
   * @NOTES: This method must be thread safe, and must not make any OpenInventor database
   *   modification or notification.
   */
  virtual void transformFunction(
    SoDataSet* ds,                      // current dataset
    const SbVec3i32& bufferDimension,   // tile dim
    SoBufferObject* bufferToTransform,  // buffer to transform
    const SbBox3i32& dataBox,           // position of tile in data space
    int resolutionLevel	                // resolution level of the tile
    )=0;

  /**
   * Returns true if the transformation affects the specified region.
   * Region is expressed in ijk space. Typically, the region represents a tile and this
   * method is used to know if a tile is affected by this transformation.
   */
  virtual bool affect( const SbBox3i32& region ) const;

  /**
   * Returns a list of IJK bounding boxes affected by this transformation.
   * For example, if this transform only affects the diagonal of the dataset, it would
   * be over-estimated to return the bounding box of the whole dataset. In this case, 
   * this method should return a list of smaller bounding boxes located along the
   * dataset diagonal.
   */
  virtual void getAffectedRegion( std::vector<SbBox3i32>& regions ) const;

  SoINTERNAL public:

  /**
   * Specifies the precision of the regions returned by #getAffectedRegion.
   * Typically, regionPrecision
   * should correspond to tileSize so that regions returned by #getAffectedRegion
   * will correspond to tiles that are affected by this transform. A regionPrecision
   * of (1, 1, 1) corresponds to a precision of 1 voxel, this implies that
   * getAffectedRegion may return a list of all voxels affected by this transform.
   * It is not a good idea to set this value to (1, 1, 1).
   * If regionPrecision is (0, 0, 0), getAffectedRegion will return only 1 bbox
   * corresponding to the region affected by this transform, but this bbox can be
   * bigger than needed.
   * @NOTES it is not mandatory that #getAffectedRegion use this value, but it
   * can help it to make a compromise between precision and number of returned regions.
   *
   * Default value is (0, 0, 0).
   */
  SoSFVec3i32 regionPrecision;

protected:

  /**
   * Default constructor.
   */
  SoLDMDataTransform();

  /** Destructor. */
  virtual ~SoLDMDataTransform();

};

#endif /* LDM_DATA_TRANSFROM_H */


