/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/
/*=======================================================================
** Author      : Benjamin Grange (MMM YYYY)
**=======================================================================*/

#ifndef SO_LDM_BORDER_READER_H
#define SO_LDM_BORDER_READER_H

#include <Inventor/SbLinear.h>
#include <Inventor/STL/map>
#include <Inventor/STL/list>

#include <Inventor/nodes/SoNode.h>
#include <Inventor/fields/SoSubField.h>

#include <Inventor/fields/SoSFNode.h>
#include <Inventor/fields/SoSFInt32.h>
#include <Inventor/fields/SoSFUInt32.h>
#include <Inventor/fields/SoSFEnum.h>
#include <Inventor/fields/SoSFFilePathString.h>
#include <Inventor/threads/SbThread.h>
#include <LDM/nodes/SoDataSet.h>
#include <LDM/readers/SoVolumeReader.h>
#include <LDM/readers/SoVRLdmFileReader.h>
#include <LDM/SoLDMTopoOctree.h>

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

class SoLDMTopoOctree;
class SoBufferObject;

/**
 * @LDMEXT LDM file reader with borders.
 *
 * @ingroup LDMReaders
 *
 * @DESCRIPTION
 *
 * Reader for LDM file with borders.
 */
class LDM_API SoVRLdmFileBorderReader : public SoVRLdmFileReader
{
  SO_FIELDCONTAINER_HEADER(SoVRLdmFileBorderReader);

public:

  /** Enum use to set the method for building tiles in lower resolution from full resolution */
  enum BuildTileMethod {
    /** Apply a decimation on the the full resolution tile to build lower resolution tile */
    DECIMATION,
    /** Apply a average method on the the full resolution tile to build lower resolution tile */
    AVERAGE
  };

  /**
    * Default Constructor.
    */
  SoVRLdmFileBorderReader();

  /**
    * @copydoc SoVolumeReader::setFilename()
    */
  virtual int setFilename(const SbString& filename);

  /**
   * @copydoc SoVolumeReader::getDataChar(SbBox3f&,SoDataSet::DataType&,SbVec3i32&)
   */
  virtual ReadError getDataChar( SbBox3f &size, SoDataSet::DataType &type, SbVec3i32 &dim ) override;

  /**
   * Returns the border flag. If 0, tiles do not overlap. If not 0, they do.
   * Since OIV 9.0 always returns 0.
   */
  virtual int getBorderFlag();

  /** @copydoc SoLDMReader::getTileSize(SbVec3i32&) */
  virtual SbBool getTileSize(SbVec3i32 &size);

  /** @copydoc SoLDMReader::readTile(int , const SbBox3i32&) */
  virtual SoBufferObject* readTile(int index, const SbBox3i32& tilePosition);

  /** @copydoc SoLDMReader::isThreadSafe() */
  virtual SbBool isThreadSafe() const;

  /**
   * Set the method used to build tiles in lower resolution.
   * Default value is AVERAGE.
   */
  void setBuildTileMethod(BuildTileMethod method);

  /** Return the current method used to build tiles in lower resolution */
  BuildTileMethod getBuildTileMethod();

SoINTERNAL public:

  /** Fill destBuffer with tile data having fileID index*/
  void readTile(int index, const SbBox3i32& tilePosition, void* destBuffer);

protected:
  /** Destructor */
  ~SoVRLdmFileBorderReader();

private:
  /** Octree computation has changed between OIV8 and OIV9, this one recreate the behavior of OIV8 
    * needed when reading ldm file with borders */
  class SoLDMBorderTopoOctree : public SoLDMTopoOctree
  {
  public:
      void init(const SbVec3i32& dimension, const SbVec3i32& tileDim, const int border);
      SbBox3i32 getTilePos(const SoLDMTileID& tileID) const;

  private:
      std::vector<int> m_deltaZOiv8;
  };
    
  //Tile size rounded to the nearest new power of 2
  SbVec3i32 m_tileSizePow2;

  SoVRLdmFileReader* m_internalLdmReader;

  /** Octree used to communicate with OIV9 (ie: without border)*/
  SoLDMBorderTopoOctree* m_topoOctreeVirtual;

  /** Octree used to read file with borders */
  SoLDMBorderTopoOctree* m_topoOcotreeReal;

  /** Current tile build method */
  BuildTileMethod m_buildMethod;

  /** Fill tiles vector with all tiles intersecting region at resolution */
  bool getIntersectingTiles(const SbBox3i32 &region, int resolution, SoLDM::TileIdVector &tiles);

  /** Copy blockSrcBox block of memory from srcMem into starting destMem from blockDestPos */
  template<typename T>
  void copyBlock(void* destMem, void* srcMem, const SbVec3i32& blockDestPos, const SbBox3i32& blockSrcBox, const SbVec3i32& arrayDim);

  template<typename T>
  void copyScaledBlockUV(void* destMem, void* srcMem, const SbBox3i32& blockDestPos,
                          const SbBox3f& blockSrcUVWBox, const SbVec3i32& arrayDim);

  /** cropped holds 1 if box has been cut by cropbox */
  SbBox3i32 getCroppedBox(const SbBox3i32& cropBox, const SbBox3i32& box, SbBox3i32& cropped);

  /** Remove borders if needed(ie: if tile is not on volume boundary)*/
  void handleBorders(SbBox3i32& tileBox, int border);

  SbBox3f computeBoxToExtract(const SbBox3i32& tileBox, const SbBox3i32& tileBoxCropped) const;

  /** Return trilinearly interpolated value between box defined by ijk0 and ijk1 according to factor*/
  template<typename T>
  inline T getInterpolatedValue(T* src, const SbVec3i32& arrayDim,
                                const SbVec3i32& ijk0, const SbVec3i32& ijk1,
                                const SbVec3f& factor);

  inline static float trilinearInterpolate(const SbVec3f& factor, float v000, float v100, float v010, float v110,
                                            float v001, float v101, float v011, float v111);
    
  /** Get value in src buffer of dimension arrayDim at position ijkSrc */
  template<typename T>
  static T getValue(T* src, const SbVec3i32& ijkSrc, const SbVec3i32& arrayDim);
};

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif /* _SO_DATA_ACCESS_READER_H */

