/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : J. Hummel (Apr 2003)
**=======================================================================*/
#ifndef  _SO_VR_LDM_FILE_READER_
#define  _SO_VR_LDM_FILE_READER_

#include <Inventor/STL/string>
#include <LDM/readers/SoLDMReader.h>
#include <Inventor/threads/SbThread.h>
#include <Inventor/STL/vector>

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

class SbThreadMutex;
class SiBitFile;


/**
 * @LDMEXT LDM file reader.
 *
 * @ingroup LDMReaders
 *
 * @DESCRIPTION
 *
 * LDM is a format for storing hierarchical multi-resolution volume data,
 * defined by VSG. VolumeViz includes a utility program that can
 * convert other formats (supported by VolumeViz) into this format.
 * Preprocessing volume data into this format provides the maximum
 * benefits from the VolumeViz large data management (LDM) features.
 *
 * The default VolumeViz LDM file format is based on an octree topology.
 * The data is stored in tiles of different subsampling resolution.
 *
 * The first tile in the file represents the entire volume at its lowest resolution and
 * corresponds to the root of the octree. The next 8 tiles represent octants
 * of the entire volume at a lower subsample step. There is another set of 8 tiles for
 * each of these tiles, and so on, until reaching tiles
 * of full resolution.
 *
 * The tile index starts at 0 for the root and increments linearly
 * into the octree (1 to 8 for octree children, 9 to 16 for children of tile 1, and so on).
 *
 * The data of a tile is accessed with the #readTile function by passing an index and
 * allocated buffer.
 *
 */
class LDM_API SoVRLdmFileReader : public SoLDMReader  
{
  SO_FIELDCONTAINER_HEADER(SoVRLdmFileReader);

  using SoLDMReader::getDataChar;

public:
  /** Default constructor. */
  SoVRLdmFileReader();

  /**
   * @copydoc SoVolumeReader::getDataChar(SbBox3f&,SoDataSet::DataType&,SbVec3i32&)
   */
  virtual ReadError getDataChar( SbBox3f &size, SoDataSet::DataType &type,  SbVec3i32 &dim ) override;

  /**
   * Returns the minimum and maximum data values.
   */
   virtual SbBool getMinMax(int64_t & min, int64_t & max);

  /**
   * Returns the minimum and maximum data values (for float values).
   * [OIVJAVA-WRAPPER NAME{getDoubleMinMax}]
   */
  virtual SbBool getMinMax(double & min, double & max);

  /**
   *
   * Returns the distribution of data values, i.e., the number of voxels per data value.
   * numVox(0) is the number of voxels with value 0, numVox(1) is the number of voxels
   * with value 1, and so on.
   */
  virtual SbBool getHistogram(std::vector<int64_t>& numVox);

  /**
   *
   * Returns the number of significant bits.
   */
  virtual int getNumSignificantBits() { return m_numSignificantBits; }

  /** @copydoc SoLDMReader::isThreadSafe() */
  virtual SbBool isThreadSafe() const;

#if SoDEPRECATED_BEGIN(8000)

  SoDEPRECATED_METHOD(8000,"Use getSubSlice( const SbBox2i32&, int sliceNumber, SoBufferObject* ) instead.")
  void getSubSlice( const SbBox2i32& subSlice, int sliceNumber, void * data );

#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(9500)

  /**
   * Returns the minimum and maximum data values.
   * [OIVJAVA-WRAPPER NAME{getIntMinMax}]
   */
  SoDEPRECATED_METHOD_NOWARN(9500, "Use getMinMax(int64_t & min, int64_t & max) instead.")
  virtual SbBool getMinMax(int & min, int & max);

#endif /** @DEPRECATED_END */

 SoINTERNAL public:

  //returns histo and values. bool returned whether some values are set or not
  virtual SbBool getHistogramAndValues(std::vector<int64_t>& numVox, std::vector<double>& values);

  // Internal use only. Return true if reader has an histogram (check m_entries size).
  virtual SbBool hasHistogram();

  virtual void saveCustomsTags(FILE* ldmHeader);

  /** 
  * Requests that the input data be converted (if necessary) to the specified data type. @BR
  * This allows, for example, conversion of float data to unsigned byte in order to reduce
  * system memory usage.  If doChange is FALSE no type conversion is applied.
  * Always returns TRUE.
  */
  virtual SbBool setOutputDataType( SbBool doChange, SoDataSet::DataType type );

 protected:
    bool  m_isDataSigned;
    bool  m_isDataFloat;

    SoDataSet::DataType m_type;

    int m_numSignificantBits;

    //min, max
    int64_t m_dataMin , m_dataMax;
    double  m_dataMinD, m_dataMaxD;

    //histo
    std::vector<int>     m_entries;
    std::vector<int64_t> m_values;
    std::vector<double>  m_valuesD;
    std::vector<int64_t> m_numVoxels;

    virtual int handleHeader(TiXmlElement *);

private:
    void internalInit();
    bool isTagCustom(const SbString& tagValue );

    SoDataSet* m_ds;
    bool m_changeFormat;
    SoDataSet::DataType m_typeOut;
    std::set<SbString> ldmKnownTags;
};

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif // _SO_VR_LDM_FILE_READER_


