/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : P. ESTRADE (Mar 2000)
**=======================================================================*/
#ifndef  _SO_VOLUME_READER_
#define  _SO_VOLUME_READER_

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

#include <Inventor/nodes/SoNode.h>
#include <Inventor/SoPreferences.h>
#include <LDM/nodes/SoDataSet.h>

#include <Inventor/STL/vector>
#include <Inventor/STL/map>

class SbThreadMutex;
class SoVolumeWriter;
/**
 * @LDMEXT Abstract base class for volume data set readers.
 *
 * @ingroup LDMReaders
 *
 * @DESCRIPTION
 *   This virtual class provides the interface through which VolumeViz
 *   accesses volume data that is not already in memory.  Predefined
 *   reader classes are provided for many common formats like DICOM and SEGY.
 *   See the list of supported file formats and class names below.
 *
 *   Application developers may implement custom reader classes. Custom
 *   reader classes allow VolumeViz to access data stored in other file
 *   formats. This is particularly useful for converting data to LDM format.
 *   A custom reader class could also allow VolumeViz to access data through
 *   a proprietary data manager or data stored in application memory.
 *
 * @B Minimum implementation:@b
 *
 *   The application must implement a class derived from SoVolumeReader, and must
 *   implement the method getDataChar(). In addition, the optional
 *   methods getNumSignificantBits() and getMinMax() can be implemented. If
 *   getNumSignificantBits() is not implemented (or returns 0), then the 
 *   number of significant bits depends on the data type. If getMinMax() is
 *   not implemented (or returns false), then VolumeViz will find the min and
 *   max voxel values. Note that this requires loading all the data and could
 *   take a lot of time for a large volume.
 *
 *   VolumeViz always calls getDataChar() and getNumSignificantBits() before
 *   requesting any data from the reader.
 *
 *   If the isDataConverted() method is not implemented (or returns false),
 *   then VolumeViz will only call getSubSlice() to request data and will
 *   build the LDM tile hierarchy in memory, using the slice data. In this
 *   case, note that: 
 *   - VolumeViz always caches data in memory using an LDM multi-resolution,
 *     tiled hierarchy.  Therefore getSubSlice() may be called multiple times
 *     with the same slice number as tiles are constructed in memory.
 *
 *   - The tile dimensions used to cache data in memory are determined by the
 *     @I tileDimension@i field of the SoLDMResourceParameters object
 *     associated with the SoVolumeData node. This field can be set by the
 *     application, but VolumeViz will not automatically set it (as it does
 *     in the case of an LDM reader).  Of course a custom reader could implement
 *     the getTileSize() method for the application to call, even though
 *     VolumeViz doesn't use it.
 *
 *   - VolumeViz will not call the getHistogram() method. So if the
 *     application calls getHistogram() on the SoVolumeData node, VolumeViz
 *     will build the histogram. This requires loading all the data and
 *     could take a lot of time for a large volume. However if VolumeViz builds the
 *     histogram, it also finds the min and max values, so there is no extra
 *     time required for that query.
 *
 * @B LDM (tiled) volume reader:@b
 *
 *   In addition to the usual requirements for an application-defined reader,
 *   e.g., implementing the getDataChar() method, an application-defined
 *   reader for LDM data is required to:
 *
 *   - Implement the #isDataConverted() method and return TRUE. @BR
 *     The method must return TRUE when the reader is passed to
 *     SoVolumeData to ensure that LDM manager classes are
 *     correctly initialized.
 *
 *   - Implement the readTile() method. @BR
 *       - When implementing the readTile method, be aware that
 *         how the SoCpuBuffer object is created affects how its memory will be managed.  
 *         In general you should create the object using the constructor with no parameters, 
 *         then allocate memory inside the object using the setSize() method.  In this case the 
 *         buffer object owns the memory and the memory will be freed when LDM no longer needs this tile.
 *         This implies for example that to avoid data copying when reading tile data from a file, 
 *         you should create and allocate the buffer object, map the buffer object, then read the data into 
 *         the buffer object. If you create the buffer object "wrapping" an existing block of memory, 
 *         that memory will not be freed when LDM releases the buffer object.
 *         The application owns that memory and is responsible for freeing it.
 *       - Note that readTile can return one of the specialized sub-classes of SoCpuBufferObject
 *         that use less system memory:
 *           - SoCpuBufferUniform: This buffer contains a "uniform" tile in which all the voxels
 *             have the same value.  It is stored efficiently as a single value until needed.
 *           - SoCpuBufferCompressed: This buffer contains a compressed tile.  It is stored
 *             efficiently in its compressed form until needed.
 *           - SoCpuBufferBasicProperty: This buffer contains a standard tile, but can also store
 *             the tile's min and max data values for use in various optimizations.
 *           - SoCpuBufferBitset: This contains a bitset tile in which each voxel is a boolean
 *             value. It is stored efficiently in a packed form until needed.
 *       - Also note that readTile can return an undefined tile by returning either NULL or an empty buffer.
 *         See readTile() for detail.
 *
 *   - Implement the getTileSize() method
 *
 * An application-defined LDM reader may optionally:
 *
 *   - Implement the #getMinMax() method  @BR
 *     Return TRUE and the requested values if the min and max data values for the
 *     volume are known to the reader, e.g., are in the file header.
 *     The reader should implement this method if possible, because applications typically
 *     query these values to setup their SoDataRange node.  If this method is not implemented,
 *     and the application calls SoVolumeData::getMinMax() is called, then VolumeViz must
 *     @I load every tile @i to compute the volume min and max.  This can cause a long delay.
 *
 *   - Implement the #getTileMinMax() method @BR
 *     Return the min and max values for the specified data tile. This information benefits
 *     optimizations such as SoLDMGlobalResourceParameters::setIgnoreFullyTransparentTiles.
 *     If this method is not implemented, VolumeViz will automatically compute the min and
 *     max values for each tile when it is loaded.  Automatic computation works fine for
 *     volume data, but we @I strongly @i recommend implementing this method for height field data.
 *
 *   - Implement the getHistogram() method @BR
 *     Return TRUE and the requested values if the min and max
 *     data values are known to the reader, e.g., are in the file header.
 *
 * @B General information:@b
 *
 *   Starting with Open Inventor 7.0, general rectilinear grids are supported.
 *   This feature allows non-uniform voxel spacing along each axis of
 *   the volume.
 *   The AmiraMesh reader (.am file), the in-memory reader, and the LDM reader
 *   support rectilinear coordinates.  Call the method setRectilinearCoordinates
 *   to specify rectilinear coordinates (if they are not already stored in
 *   the data file and set by the reader).
 *
 *   Starting with Open Inventor 7.0, SoVolumeReader is derived from SoFieldContainer.
 *   This allows client classes like SoVolumeData to be automatically notified
 *   when the reader's state changes, and update their own state appropriately.
 *   Any reader method that changes the volume properties (dimension, size, data type,
 *   etc) should trigger notification by calling the reader's touch() method.  If this
 *   notification is not done, client node fields, for example SoDataSet::extent, won't be updated
 *   correctly.  For example, a reader with a method setData(SbVec3i32 size, void* data)
 *   that loads a new data set should call touch() at its end.  This reader could also
 *   be implemented using an SoSFArray3D field instead of the setData method.  Modifying
 *   this field will automatically trigger notification.
 *
 *	 Applications should subclass from SoVolumeReader when creating any type of custom volume reader, 
 *   including an LDM (tiled) volume reader. The classes SoLDMReader and SoVRLdmFileReader are internal 
 *   classes which are specific to the VSG defined LDM file format.
 *
 *   This class cannot be instantiated directly.
 *
 * @H3 Supported file formats: @h3
 *
 *   @TABLE_1B
 *   @TR @B File extension @b  @TD @B Reader class @b     @TD @B Description @b
 *   @TR .am                   @TD SoVRAmFileReader       @TD Avizo Mesh file format
 *   @TR .dc3, .dic, .dicom    @TD SoVRDicomFileReader    @TD DICOM file format
 *   @TR .fld                  @TD SoVRAvsFileReader      @TD AVS field file format
 *   @TR .lda or .ldm          @TD SoVRLdmFileReader      @TD LDM file format
 *   @TR .mrc                  @TD SoVRMrcFileReader      @TD MRC file format
 *   @TR .sgy or .segy         @TD SoVRSegyFileReader     @TD SEG Y rev 1 file format
 *   @TR .vol                  @TD SoVRVolFileReader      @TD Vol file format
 *   @TR .vox                  @TD SoVRVoxFileReader      @TD Vox file format
 *   @TR .lst                  @TD SoVRRasterStackReader  @TD Lst file format (stack of images)
 *   @TABLE_END
 *
 *   File format notes:
 *   - Avizo mesh @BR
 *     Avizo mesh is a general purpose file format that can contain many different kinds of data. The
 *     VolumeViz file reader can load Avizo mesh files containing a 3-dimensional "Lattice" data object
 *     with uniform coordinates and any data type.  See SoVRAmFileReader for limitations.
 *
 *   - AVS field @BR
 *     AVS field is a general purpose file format that can contain many different kinds of data. The
 *     VolumeViz file reader can load AVS field files containing 3-dimensional, uniform data of type
 *     "byte". See SoVRAvsFileReader.
 *
 *   - DICOM @BR
 *     A widely used format for storing medical image data (CT, MRI, etc), defined by the National
 *     Electrical Manufacturers Association (NEMA) (medical.nema.org). See SoVRDicomFileReader
 *
 *   - LDM @BR
 *     LDM is a format defined by VSG for storing hierarchical multi-resolution volume data.
 *     VolumeViz includes a utility program that can convert any other format supported by VolumeViz
 *     into this format (see SoVolumeConverter). Preprocessing volume data into this format provides
 *     the maximum benefits from the VolumeViz large data management (LDM) features. See SoVRLdmFileReader.
 *
 *   - MRC @BR
 *     MRC is a file format that has become an industry standard for cryo-electron microscopy
 *     (cryoEM) and electron tomography (ET) (http://www.ccpem.ac.uk/mrc_format/mrc2014.php).
 *     See SoVRMrcFileReader.
 *
 *   - SEGY @BR
 *     A widely used format for storing seismic trace data, defined by the Society of Exploration
 *     Geophysicists publication "Digital Tape Standards" (www.seg.org). The VolumeViz reader
 *     supports all sizes of integer and float data, and can correctly determine the number of
 *     samples per trace in many cases. However the reader also has many options to adapt to
 *     differences in SEGY file headers.  See SoVRSegyFileReader.
 *
 *   - VOL @BR
 *     A simple volume interchange format (see "Introduction to Volume Rendering", Lichtenbelt,
 *     Crane, Naqvi, 1998). The VolumeViz reader can load files containing 8- or 16-bit voxels.
 *     See SoVRVolFileReader.
 *
 *   - VOX @BR
 *     A volume interchange format defined by TeraRecon Inc. (www.terarecon.com). The VolumeViz
 *     reader can load "Vox1999a" files containing 8- or 16-bit voxels (first volume only).
 *     See SOVRVoxFileReader.
 *
 *   - LST (stack of images) @BR
 *     A simple format for loading a stack of images (one image per file).
 *     Specify the names of the image files in a .lst file.  VolumeViz can load image data in
 *     most common image formats including BMP, DDS, GIF, JPEG, JPEG2000, PNG and TIFF. See
 *     SoVRRasterStackReader for details and limitations.
 *
 *   Note: '3D TIFF' files (multiple images in one file) are not currently supported.
 *
 * @SEE_ALSO
 *    SoDataSet,
 *    SoVolumeData,
 *    SoConverter
 *
 * [OIVNET-WRAPPER-CLASS ALL_DERIVABLE]
 */
class LDM_API SoVolumeReader : public SoFieldContainer
{
  SO_FIELDCONTAINER_ABSTRACT_HEADER(SoVolumeReader);

 public:

  /**
   * Read error.
   */
   enum ReadError {
    /**
     * No error.
     */
    RD_NO_ERROR,
    /**
     * The data file is not found.
     */
    RD_FILE_NOT_FOUND_ERROR,
    /**
     * Invalid data.
     */
    RD_INVALID_DATA_ERROR,
    /**
     * Unsupported data type (but the data is valid).
     */
    RD_UNSUPPORTED_DATA_TYPE_ERROR,
    /**
     * Some error were detected in the file making it impossible
     * to read.
     */
    RD_FILE_FORMAT_NOT_VALID_ERROR,
    /**
     * Unknown error.
     */
    RD_UNKNOWN_ERROR
   } ;

  /**
   * Which axis to handle
   * @see getRectilinearCoordinates
   */
  enum Axis {
    X,
    Y,
    Z
  };

  /**
   * Constructor.
   * [OIVJAVA-WRAPPER NO_WRAP]
   */
   SoVolumeReader();

  /**
   * Gets the characteristics (file header) of the data volume. See
   * SoDataSet::setVolumeData().@BR
   * You can use the convenience method setFilename() to specify the file location,
   * in which case you will not have to open the file yourself. Then you can use
   * the convenience method getBuffer() to read the header in order to get the requested
   * information.
   *
   * NOTE: We strongly recommend that readers implement this version of #getDataChar,
   * introduced in VolumeViz 5.1, because it uses SbVec3i32 for the volume dimension.
   *
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onGetDataChar()},PACK{DataInfo}]
   * [OIVJAVA-WRAPPER-RETURN-TYPE NAME{readError}]
   */
  virtual ReadError getDataChar( SbBox3f &size, SoDataSet::DataType &type, SbVec3i32 &dim )=0;

  /**
   * Sets the output data type. Returns FALSE if the reader does not support this feature.
   * If the reader does support this feature and @B doChange @b is set to TRUE, data is
   * converted before being returned (by getSubSlice() for instance).
   * If @B doChange @b is set to FALSE, data is not converted and is returned as is.
   */
  virtual SbBool setOutputDataType( SbBool doChange, SoDataSet::DataType outputType );

  /**
   * Requests that the input be converted from the specified range to the range depending on
   * the output data type.
   * This allows, for instance, if the output data type is unsigned byte, conversion of float data
   * from range [min,max] to range [0,255].
   * If doChange is FALSE no range conversion is applied.
   * Always returns TRUE.
   */
  virtual SbBool setInputDataRange( SbBool doChange, double min, double max );

  /**
   * This method is optional. It returns the number of significant bits of the volume data.
   *
   * If it is not implemented, the default return value is 0, meaning the number of bits
   * depends on the data type. See the last parameter of SoVolumeData::setVolumeData().
   * This method is called immediately after getDataChar().
   * For float data type, number of bits is forced to 32.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{if(customVolumeReader) return 0}]
   */
  virtual int getNumSignificantBits() { return 0; };

  /**
   * Must copy the rectangular part defined by @B subSlice @b of the XY slice @B sliceNumber @b
   * to the memory referenced by @B data@b.
   * Slices will not always be read sequentially.
   *
   * @param subSlice 2D region of the slice to return.
   * @param sliceNumber Slice number on the volume Z axis (first slice is 0).
   * @param data Copy the data into this buffer.
   *
   * You can use the convenience method getBuffer() to read data from file. Note:
   * setFilename() must have been called previously.
   *
   * [OIVNET-WRAPPER NATIVE_HELPER_BEGIN{SbVec2i32 subSliceMin, subSliceMax; subSlice.getBounds(subSliceMin, subSliceMax); const size_t byteSize = (subSliceMax[0] - subSliceMin[0] + 1) * (subSliceMax[1] - subSliceMin[1] + 1) * getDataSize()}]
   * [OIVNET-WRAPPER-ARG IN,IN,ARRAY{byteSize}]
   */
  virtual void getSubSlice(const SbBox2i32& subSlice, int sliceNumber, void* data);

  /** 
   * \if_java
   * This method is called by Open Inventor during an LDM conversion or during volume loading.
   * It must be overridden to define a custom reader. The implementation must copy
   * the rectangular part defined by @B subSlice @b of the XY slice @B sliceNumber @b
   * to the given buffer object.
   *
   * @NOTES
   * When loading or converting a large volume data, the size of the given bufferObject may
   * exceed 2^31 elements. In that case, it is not possible to get the Java array that backs
   * the bufferObject because an array cannot contain more than 2^31 elements in Java.
   * That may lead to generate a Java.lang.IllegalArgumentException with a negative capacity message.
   *
   * That occurs in the following situation: In order to build the root tile of the model,
   * LDM call getSubSlice with a buffer of size S = Si * Sj * data type size. Si is the lowest
   * power of 2 bigger than the I dimension of the volume. Sj is the lowest power of 2 bigger
   * than the J dimension of the volume. For instance if the volume contains 17000*35000*1500
   * short values, a buffer of size 32768*65536*2 bytes is requested. This example will lead
   * to an exception.
   *
   * \else
   * Same as getSubSlice( const SbBox2i32& subSlice, int sliceNumber, void * data )
   * but using an SoBufferObject as the target of the data.
   * If not reimplemented then the version with "unsigned char*" parameter is called
   * \endif
   *
   * @param subSlice 2D region of the slice to return.
   * @param sliceNumber Slice number on the volume Z axis (first slice is 0).
   * @param dataBuffer The target buffer to be filled.
   *
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onGetSubSlice()}]
   */
  virtual void getSubSlice( const SbBox2i32& subSlice, int sliceNumber, SoBufferObject * dataBuffer);

  /**
   * Utility function provided by SoVolumeReader for subclass readers to call.
   *
   * Returns the size of the brick the reader must use, based on @B subsamplingLevel@b
   * and @B realSize@b of the brick.
   * If the subsampling level is greater than 0 on an axis, the corresponding size
   * is computed as follows:
   *   -# @B realSize@b is divided by 2**@B subsamplingLevel@b
   *   -# @B brickSize@b is the next greater power of 2
   *
   * For example, if @B subsamplingLevel@b[0]=1 and @B realSize@b[0]=21, then the returned @B readerSize@b[0]=16.
   *
   * If @B subsamplingLevel@b is 0, the corresponding size is the @B realSize@b.
   */
   SbVec3i32 getNumVoxels( const SbVec3i32& realSize, const SbVec3i32& subsamplingLevel );

   /**
    * Utility function provided by SoVolumeReader for subclass readers to call.
    *
    * If the reader uses the NO_COPY policy, this method returns the size
    * to allocate for the brick.
    * For each axis of the brick, this size is:
    *
    * (first power of 2 greater than @B realSize@b) / ( 2**@B subsamplingLevel@b)
    */
   SbVec3i32 getSizeToAllocate( const SbVec3i32& realSize, const SbVec3i32& subsamplingLevel );

  /**
  * Specifies the path of the file. Returns 0 for success. Any other return
  * value indicates failure.
  */
  virtual int setFilename( const SbString& filename );

 /**
  * Returns the path of the file.
  */
  SbString getFilename() const;

  //------------ for LDM -----------------//
  /**
   * Returns TRUE if the data is already organized in tiles for the LDM module. @BR
   * In other words, all drivers that directly support the getTile() method should
   * return TRUE from isDataConverted.
   * If TRUE is returned, VolumeViz will use the readTile method and will NOT call
   * getSubVolume() or getSubSlice().
   * [OIVJAVA-WRAPPER HELPER_BEGIN{if(customVolumeReader) return false}]
   * [OIVNET-WRAPPER-CUSTOM-CODE]
   */
  virtual SbBool isDataConverted() const;

  virtual void reloadTileMinMax() {};

  /**
   * Returns tile dimensions in voxels when using data stored in tiles. @BR
   * Return FALSE if data is not stored in tiles.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{if(customVolumeReader) return null}]
   */
  virtual SbBool getTileSize(SbVec3i32& size);

  /**
   * Given an index, reads a tile if the data is organized in tiles (for LDM). @BR
   * In the default LDM architecture, the LDM data is based on an octree
   * topology (see SoLDMFileReader). The index passed is 0 for the tile
   * of lowest resolution representing the entire volume (octree root node).
   * The index increments linearly going down through the octree.
   *
   * A tile can be undefined or empty which means it is not rendered and it isn't taken into account
   * for the rendering of its adjacent cells. An empty buffer or NULL can be returned by this method
   * to specify an undefined tile.
   *
   * @param index specifies a fileID, the id of an existing tile (fileID=tileID in a cubical volume).
   * @param tilePosition specifies the position of the data in the associated volume data of the tile
   * corresponding to the given index. In the default SoVRLdmFileReader, the tilePosition isn't actually
   * used but it is passed as a convenience for customized readers (can be used for mapping to a
   * different index scheme).
   *
   * @return the buffer containing the data of the tile, or NULL (or an empty buffer) in order to specify an undefined tile.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{if(customVolumeReader) return new com.openinventor.inventor.devices.SoCpuBufferObject()}]
   */
  virtual SoBufferObject* readTile(int index, const SbBox3i32& tilePosition);

   /** 
   * Read directly from the data source, a trace inside a tile. @BR
   * @param index the fileId of an existing tile.
   * @param buffer the buffer in which the data is returned.
   * @param tilePosition Specifies the position of the tile in the associated volume, in voxel coordinates. @BR
   * In the default SoVRLdmFileReader, the tilePosition isn't actually used by the reader
   * but it is passed as a convenience for customized readers (can be used for mapping to a
   * different index scheme).
   * @param tracePosition represents the (i,j) coordinates of the trace.
   *
   * [OIVNET-WRAPPER NATIVE_HELPER_BEGIN{::SbVec3i32 tileSize; getTileSize(tileSize); const size_t byteSize = ((size_t) tileSize[2]) * getDataSize()}]
   * [OIVNET-WRAPPER-ARG IN,WRAP_AS{void *}&ARRAY{byteSize},IN,IN]
   */
  virtual SbBool readXTraceInTile(int index, unsigned char*& buffer, const SbBox3i32& tilePosition, const SbVec2i32& tracePosition);

   /** 
   * Read directly from the data source, an orthoslice on the Y axis (Zaxis == 1) inside a tile.
   * @param index The fileId of the tile.
   * @param buffer The buffer in which the data is returned.
   * @param tilePosition Specifies the position of the tile in the associated volume, in voxel coordinates. @BR
   * In the default SoVRLdmFileReader, the tilePosition isn't actually used by the reader
   * but it is passed as a convenience for customized readers (can be used for mapping to a
   * different index scheme).
   * @param slicePosition The slice position in the tile.
   *
   * [OIVNET-WRAPPER NATIVE_HELPER_BEGIN{::SbVec3i32 tileSize; getTileSize(tileSize); const size_t byteSize = ((size_t) tileSize[2]) * tileSize[2] * getDataSize()}]
   * [OIVNET-WRAPPER-ARG IN,WRAP_AS{void *}&ARRAY{byteSize},IN,IN]
   */
  virtual SbBool readYSliceInTile(int index, unsigned char*& buffer, const SbBox3i32& tilePosition, const uint32_t& slicePosition );

   /** 
   * Read directly from the data source, an orthoslice on the Z axis (Zaxis == 2) inside a tile.
   * @param index The fileId of the tile.
   * @param buffer The buffer in which the data is returned.
   * @param tilePosition Specifies the position of the tile in the associated volume, in voxel coordinates. @BR
   * In the default SoVRLdmFileReader, the tilePosition isn't actually used by the reader
   * but it is passed as a convenience for customized readers (can be used for mapping to a
   * different index scheme).
   * @param slicePosition The slice position in the tile.
   *
   * [OIVNET-WRAPPER NATIVE_HELPER_BEGIN{::SbVec3i32 tileSize; getTileSize(tileSize); const size_t byteSize = ((size_t) tileSize[0]) * tileSize[0] * getDataSize()}]
   * [OIVNET-WRAPPER-ARG IN,WRAP_AS{void *}&ARRAY{byteSize},IN,IN]
   */
  virtual SbBool readZSliceInTile(int index, unsigned char*& buffer, const SbBox3i32& tilePosition, const uint32_t& slicePosition );

   /** 
   * Read directly from the data source, an orthoslice on the X axis (Zaxis == 0) inside a tile.
   * @param index The fileId of the tile.
   * @param buffer The buffer in which the data is returned.
   * @param tilePosition Specifies the position of the tile in the associated volume, in voxel coordinates. @BR
   * In the default SoVRLdmFileReader, the tilePosition isn't actually used by the reader
   * but it is passed as a convenience for customized readers (can be used for mapping to a
   * different index scheme).
   * @param slicePosition The slice position in the tile.
   *
   * [OIVNET-WRAPPER NATIVE_HELPER_BEGIN{::SbVec3i32 tileSize; getTileSize(tileSize); const size_t byteSize = ((size_t) tileSize[1]) * tileSize[2] * getDataSize()}]
   * [OIVNET-WRAPPER-ARG IN,WRAP_AS{void *}&ARRAY{byteSize},IN,IN]
   */
  virtual SbBool readXSliceInTile(int index, unsigned char*& buffer, const SbBox3i32& tilePosition, const uint32_t& slicePosition );

  /**
   * Should return TRUE if the reader is thread safe. @BR
   * This function is called by VolumeViz when using the multiIO mode (LDM only).
   * LDM multi-threaded data loading can only be used if this method returns TRUE.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{if(customVolumeReader) return false}]
   * [OIVNET-WRAPPER PROPERTY{IsThreadSafe},GETTER]
   */
  virtual SbBool isThreadSafe() const;

  /**
   * Should return TRUE if at least one file has been ignored during reading
  */
  virtual SbBool isIgnoredFile() const;

  /**
   * Returns original file name from which the data has been
   * converted to LDM format if stored in file.
   */
  virtual SbString getOriginalFilename() const
  { return SbString(""); }

  /**
   * Returns min and max for integer data type, if available.
   * If not available, return false.
   * The reader should implement this method if possible, because applications often
   * query these values to setup their SoDataRange node.  If this method is not implemented,
   * and the application calls SoVolumeData::getMinMax() is called, then VolumeViz must
   * load every tile to compute the volume min and max.  This can cause a long delay.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{if(customVolumeReader) return null}]
   */
  virtual SbBool getMinMax(int64_t & min, int64_t & max);

  /**
   * Returns min max for float data type, if available.
   * If not available, return false.
   * The reader should implement this method if possible, because applications often
   * query these values to setup their SoDataRange node.  If this method is not implemented,
   * and the application calls SoVolumeData::getMinMax() is called, then VolumeViz must
   * load every tile to compute the volume min and max.  This can cause a long delay.
   * [OIVJAVA-WRAPPER NAME{getDoubleMinMax},HELPER_BEGIN{if(customVolumeReader) return null}]
   */
  virtual SbBool getMinMax(double & min, double & max);

  /**
   * Returns histogram if available. If not, return false.
   *
   * [OIVNET-WRAPPER NATIVE_HELPER_BEGIN{const bool useLegacyMethod = m_managedObjectHandle->OverridesLegacyGetHistogram(); System::Collections::Queue ^_numVox = useLegacyMethod ? (gcnew System::Collections::Queue()) : nullptr; const bool legacyRV = useLegacyMethod ? m_managedObjectHandle->GetHistogram(_numVox) : false; if (useLegacyMethod) numVox.clear(); if (useLegacyMethod) numVox.reserve(_numVox->Count); System::Collections::IEnumerator ^iterator = useLegacyMethod ? _numVox->GetEnumerator() : nullptr; if (useLegacyMethod) while (iterator->MoveNext()) numVox.push_back(System::Convert::ToInt64(iterator->Current)); if (useLegacyMethod) _numVox->Clear(); if (useLegacyMethod) return legacyRV}]
   */
  virtual SbBool getHistogram(std::vector<int64_t>& numVox);

  /**
   * Sets whether or not the reader should automatically try to detect if the coordinate system
   * used is direct or not.
   * The function will return FALSE if the feature is not supported
   * by the current reader.
   * Default is false.
   */
  virtual SbBool setDirectCoordSysAutoDetection(SbBool autoValue);
  /**
   * Return automatic detection value.
   */
  virtual SbBool getDirectCoordSysAutoDetection();

  /**
   * Specify if the coordinate system used is direct or not.
   * The function will return FALSE if the feature is not supported
   * by the current reader.
   * Default is true.
   */
  virtual SbBool setDirectCoorSys(SbBool directCoord);

  /**
   * Return whether the coordinate system used is direct or not.
   */
  virtual SbBool getDirectCoordSys();

  /**
  * Returns the minimum and maximum data values for the given tile.
  * This information benefits optimizations such as SoLDMGlobalResourceParameters::setIgnoreFullyTransparentTiles
  * and custom SoVolumeReader able to return SoCpuBufferUniform.
  *
  * VolumeViz will only call this method if the data is organized in tiles like the LDM file format.
  * In other words, if #isDataConverted() returned true.  The LDM Converter program automatically
  * computes this information and stores it in the LDM header file. Custom volume readers that
  * implement tiled data, i.e. return true when #isDataConverted is called, should consider implement this
  * method when the min max values are available from their backend API.
  *
  * @B NOTES: @b
  * - Automatic computation of tile min/max values works fine for actual volume data.
  *   But we @I strongly @i recommend implementing this method for height field data (see
  *   SoHeightFieldGeometry etc).  Because of the way the height field algorithm works, if tile
  *   min/max info is not available, VolumeViz must load @I all @i height field tiles before the
  *   first render.  This can cause a long delay before the first rendering appears.
  * - When all the voxels in the requested tile (fileId) have the same value, we recommend
  *   that the volume reader implement two optimizations.  First, the readTile() method should
  *   return an SoCpuBufferUniform object (instead of SoCpuBufferObject).
  *   This reduces the amount of CPU and GPU memory needed to store the tile, allowing more tiles
  *   to be loaded in the same amount of memory.  Second, the getTileMinMax() method should return
  *   an empty range (min = max) for this tile.  This allows VolumeViz to optimize sending tiles to the GPU.
  *   However, note that the getTileMinMax() method is called before the readTile() method.
  *   So ideally the volume reader should be able to return the tile min/max without
  *   actually loading the tile data, for example by using metadata stored with the tiles.
  *
  * \if_cpp
  * Default implementation returns SbVec2d(numeric_limits<double>::max(),-numeric_limits<double>::max())
  * \endif
  * \if_java
  * Default implementation returns new SbVec2d(Double.MAX_VALUE, -Double.MAX_VALUE)
  * \endif
  * \if_dotnet
  * Default implementation returns new SbVec2d(System.Double.MaxValue, -System.Double.MaxValue))
  * \endif
  * @param index The fileId of the tile.
  * [OIVJAVA-WRAPPER HELPER_BEGIN{if(customVolumeReader) return new com.openinventor.inventor.SbVec2d(Double.MAX_VALUE, -Double.MAX_VALUE);}]
  */
  virtual SbVec2d getTileMinMax( int index ) const;

  /**
   * Available reader type.
   */
  enum ReaderType {
    /** Unknown */
    NO_READER,
    /** Amira .am */
    AM,
    /** AVS */
    AVS,
    /** Dicom */
    DICOM,
    /** Generic */
    GENERIC,
    /** Open Inventor LDM */
    LDM,
    /** In memory */
    MEMORY,
    /** Raster Stack */
    RASTERSTACK,
    /** Segy */
    SEGY,
    /** Vol */
    VOL,
    /** Vox */
    VOX,
    /** Tiff */
    TIFF,
    /** Mrc */
    MRC
  };

  /**
   * Returns the reader type.
   */
  virtual SoVolumeReader::ReaderType getReaderType (){return NO_READER;};

   /** 
   * Returns a volume writer that corresponds to this reader @BR
   * (same format, parameters, etc).
   * If no writer can be created, NULL is returned.
   *
   * Notes:
   * - The simplest way to implement this behavior is to return a new writer every time.
   *
   * - The returned writer is not synchronized with this reader nor with other writers.
   *
   * \if_cpp
   * - The returned writer must be manually refed/unrefed.
   *
   * For example:
   * \code
   *   MyVolumeReader* reader = new MyVolumeReader();
   *   reader->setFilename( "someFile.dat" );
   *   MyVolumeWriter* writer1 = (MyVolumeWriter*)reader->getConfiguredWriter();
   *   // This writer will write to "someFile.dat"
   *
   *   reader->setFilename( "someOtherFile.dat" );
   *   // The writer will still write to "someFile.dat"
   *
   *   writer1->setFilename( "stillAnotherFile.dat" );
   *   SoVolumeWriter* writer2 = reader->getConfiguredWriter();
   *   // writer2 will write to "someOtherFile.dat", because the
   *   // reader is configured to read from "someOtherFile.dat"
   * \endcode
   * \endif
   * \if_dotnet
   * For example:
   * \code
   *   MyVolumeReader reader = new MyVolumeReader();
   *   reader.SetFilename( "someFile.dat"  );
   *   MyVolumeWriter writer1 = (MyVolumeWriter)reader.GetConfiguredWriter();
   *   // This writer will write to "someFile.dat"
   *
   *   reader.SetFilename( "someOtherFile.dat" );
   *   // The writer will still write to "someFile.dat"
   *
   *   writer1.SetFilename( "stillAnotherFile.dat" );
   *   SoVolumeWriter writer2 = reader.GetConfiguredWriter();
   *   // writer2 will write to "someOtherFile.dat", because
   *   // the reader is configured to read from "someOtherFile.dat"
   * \endcode
   * \endif
   * \if_java
   * For example:
   * \code
   *   MyVolumeReader reader = new MyVolumeReader();
   *   reader.setFilename( "someFile.dat"  );
   *   MyVolumeWriter writer1 = (MyVolumeWriter)reader.getConfiguredWriter();
   *   // This writer will write to "someFile.dat"
   *
   *   reader.setFilename( "someOtherFile.dat" );
   *   // The writer will still write to "someFile.dat"
   *
   *   writer1.setFilename( "stillAnotherFile.dat" );
   *   SoVolumeWriter writer2 = reader.getConfiguredWriter();
   *   // writer2 will write to "someOtherFile.dat", because
   *   // the reader is configured to read from "someOtherFile.dat"
   * \endcode
   * \endif
   *
   */
  virtual SoVolumeWriter* getConfiguredWriter();

  /**
   * Coordinate type used by this data set.
   */
  enum CoordinateType {
  /**
   * Uniform grid spacing along each axis.
   */
    COORDINATES_UNIFORM = 0,
  /**
   * Grid spacing defined by x, y, z values.
   */
    COORDINATES_RECTILINEAR
  };

  /**
   * Returns coordinate type used by the data set.
   */
  CoordinateType getCoordinateType();

  /**
   * Returns the coordinates for the specified axis.
   *
   * [OIVNET-WRAPPER HELPER_BEGIN{OIV::Inventor::SbBox3f volExtent; OIV::LDM::Nodes::SoDataSet::DataTypes dataType; OIV::Inventor::SbVec3i32 volDims; GetDataChar(volExtent, dataType, volDims)}]
   * [OIVNET-WRAPPER-RETURN-TYPE ARRAY{volDims[(int) axis]}]
   */
  const float * getRectilinearCoordinates(Axis axis) const;

  /**
   * Sets rectilinear coordinates for the data set.
   */
  void setRectilinearCoordinates(const float *x, const float *y, const float *z);

  /**
   * Returns TRUE if the data set contains RGBA color values.
   * [OIVJAVA-WRAPPER HELPER_BEGIN{if(customVolumeReader) return false}]
   * [OIVNET-WRAPPER PROPERTY{IsRGBA},GETTER]
   */
  virtual SbBool isRGBA() const 
  { return m_isRGBA; }

  /**
   * Specifies if data must be considered as RGBA.
   * [OIVNET-WRAPPER PROPERTY{IsRGBA},SETTER]
   */
  inline void setRGBA(const SbBool flag)
  { m_isRGBA = flag; }

  /** 
   * Close all resources that are locked by the reader. 
   * Allows other classes to use the same resources.
   * For example, if the reader reads from a file, this method closes the file
   * so that an SoVolumeWriter associated with the same file can re-open it in write mode.
   * The reader remembers the resources that were closed, so they can re-opened
   * by calling #restoreAllHandles().
   */
  virtual void closeAllHandles() {}; 

  /** 
  * Restore resources that were closed by #closeAllHandles().
  */
  virtual void restoreAllHandles() {}; 

  /**
   * Returns a preconfigured SoVolumeReader instance that can be used to load
   * the given file (based on the filename extension).
   *
   * For example, would return an instance of SoVRDicomFileReader for a file named "data.dic".
   *
   * If no SoVolumeReader is associated to this extension through registerVolumeReaderExtension()
   * then NULL is returned.
   *
   * @M_SINCE 9.4
   */
  static SoVolumeReader* getAppropriateReader(const SbString& filename);

#if SoDEPRECATED_BEGIN(8500)

  /**
   * Given an index, reads a tile if the data is organized in tiles (for LDM). @BR
   * In the default LDM architecture, the LDM data is based on an octree
   * topology (see SoVRLdmFileReader). The index passed is 0 for the tile
   * of lowest resolution representing the entire volume (octree root node).
   * The index increments linearly going down through the octree.
   *
   * Indexing works as follows:
   *
   * Tile 1 is the lower back left corner of the cube.
   * The index increments on X, then Y, and finally Z.
   * So the back tiles are:@BR
   *
   * 3  4@BR
   * 1  2
   *
   * And the front tiles are:@BR
   *
   * 7  8@BR
   * 5  6
   *
   * The tiles of full resolution are the leaf tiles.
   *
   * In the default SoVRLdmFileReader, the tilePosition isn't actually used by the reader
   * but it is passed as a convenience for customized readers (can be used for mapping to a
   * different index scheme).
   *
   * [OIVNET-WRAPPER NATIVE_HELPER_BEGIN{::SbVec3i32 tileSize; getTileSize(tileSize); const size_t byteSize = ((size_t) tileSize[0]) * tileSize[1] * tileSize[2] * getDataSize()}]
   * [OIVNET-WRAPPER-ARG IN,WRAP_AS{void *}&ARRAY{byteSize},IN]
   */
  SoDEPRECATED_METHOD_NOWARN(8500,"Use SoBufferObject* readTile(int index, const SbBox3i32& tilePosition) instead.")
  virtual SbBool readTile(int SO_UNUSED_PARAM(index), unsigned char*& SO_UNUSED_PARAM(buffer), const SbBox3i32& SO_UNUSED_PARAM(tilePosition)) { return FALSE; }

  /**
   * Same as readTile(int index, unsigned char*& buffer, const SbBox3i32& tilePosition)
   * but using an SoBufferObject (allocated by LDM) as the target of the data.
   * If not reimplemented then the version with "unsigned char*" parameter is called
   */
  SoDEPRECATED_METHOD_NOWARN(8500, "Use SoBufferObject* readTile(int index, const SbBox3i32& tilePosition) instead.")
  virtual SbBool readTile(int index, SoBufferObject *buffer, const SbBox3i32& tilePosition);

#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(9000)

  /**
   * Returns border (overlap) value if stored in file.
   */
  SoDEPRECATED_METHOD_NOWARN(9000,"No longer used. Only kept to read old file format that contains borderFlag.")
  virtual int getBorderFlag()
  {return -1;}

#endif /** @DEPRECATED_END */

#if SoDEPRECATED_BEGIN(9500)

  /**
   * Returns min max if stored in file.
   * [OIVJAVA-WRAPPER NAME{getIntMinMax},HELPER_BEGIN{if(customVolumeReader) return null}]
   */
  SoDEPRECATED_METHOD_NOWARN(9500, "Use getMinMax(int64_t & min, int64_t & max) instead.")
  virtual SbBool getMinMax(int& min, int& max);

#endif /** @DEPRECATED_END */

SoINTERNAL public:
  //for volume having a series of data per datum
  //used by ReservoirViz. The converter asks the eclipse reader for the datum.
  //not used in vviz which uses getDataChar(...type...). (must keep for compatibility reasons).
  virtual ReadError getDataChar( SbBox3f &size, std::vector<SoDataSet::DatumElement>& datum, SbVec3i32 &dim );

  void lockFileMutex();
  void unlockFileMutex();

  /**
   * Converts from type @B typeIn @b to type @B typeOut @b.
   * By default, if typeOut is of smaller size than typeIn, only the most significant bytes are
   * copied into the dest buffer (data is right shifted).
   * If typeOut is of bigger size than typeIn, then the data is shifted to the left
   * because VolumeViz truncates the least significant bytes if the data range is larger than the color range.
   * (Tt maps an unsigned short on an 8-bit color map by looking at the 8 most significant bits.)
   * If shiftData is FALSE, nothing is shifted (the data isn't changed).
   */
  void convert( SoDataSet::DataType typeIn , SoBufferObject* bufferIn,
                SoDataSet::DataType typeOut, SoBufferObject* bufferOut, size_t size, SbBool shiftData = true);

  void convertDataRange( bool doRange, double min, double max );

  virtual int getNumBytesPerDatum();

  // number of bytes in a voxel
  // uses DataType if getNumBytesPerDatum() returns 0
  int getDataSize();

  virtual bool isLDMReader() { return false; }
  virtual bool isVolumeMaskReader() { return false; }

  virtual bool hasDeadCell() { return false; }

  virtual bool isDead(int i,int j,int k);

  virtual uint8_t* getCellGridState() { return NULL; }

  virtual int getNbCell() { return 0; }

  virtual uint64_t getTileSizeFromFid(int fileId);// Virtual method, only used
                                                  // by resviz for now

  /**
   * Returns true if per tile min-max information is available.
   */
  bool hasPerTileMinMaxComputed();

#if SoDEPRECATED_BEGIN(8000)

  SoDEPRECATED_METHOD(8000," Use convert( SoDataSet::DataType, SoBufferObject* ,SoDataSet::DataType, SoBufferObject*, int , SbBool) instead.")
  void convert( SoDataSet::DataType typeIn , void* bufferIn,
                SoDataSet::DataType typeOut, void* bufferOut, size_t size, SbBool shiftData = true);

#endif /** @DEPRECATED_END */

  /**
   * Returns the underlying reader (if any).
   * Readers that wrap another reader should override this method to return their internal reader.
   */
  virtual SoVolumeReader* getInternalReader() const
  {
    return const_cast<SoVolumeReader*>(this);
  }

 protected: //PROTECTED_TO_DOCUMENT
  /**
   * Register an SoVolumeReader class to handle a file extension.
   *
   * This method can be called for any class derived from SoVolumeReader in order
   * to allow the #getAppropriateReader() method to return an instance of
   * the specified reader for files with the specified extension. VolumeViz automatically
   * associates the built-in volume reader classes with their corresponding file
   * extensions. For example, the extension "dic" is associated with SoVRDicomFileReader.
   * This method can be used to associate a custom volume reader with an application
   * data file extension. When this association exists, the application can simply
   * set the SoVolumeData node's filename field and VolumeViz will create an instance
   * of the associated volume reader to load the data.
   *
   * This method can be called multiple times with different file extensions.
   * Extension strings are not case sensitive and should not include the '.' character.
   *
   * This method should normally be called from the volume reader's initClass() method
   * and the #unregisterVolumeReaderExtensions() method should be called from the exitClass() method.
   *
   * If the given filename extension is already associated with another SoVolumeReader
   * class then the call is ignored and false is returned.
   *
   * @param fileExtension file extension to associate with the given readerType (without .)
   * @param readerType reader class type to associate with the given fileExtension
   * @return true on success.
   *
   * @M_SINCE 9.4
   */
  static bool registerVolumeReaderExtension(const SbString& fileExtension, const SoType& readerType);

  /**
   * Unregister all filename extensions associated with the specified SoVolumeReader class.
   * This method must be called by any SoVolumeReader derived class that previously called #registerVolumeReaderExtension().
   *
   * This method should normally be called once in the volume reader's exitClass() method.
   * It returns false if no filename extension was registered for the specified class.
   *
   * @M_SINCE 9.4
   */
  static bool unregisterVolumeReaderExtensions(const SoType& readerType);

  /**
   * Destructor.
   */
  virtual ~SoVolumeReader();

  /**
   * Returns a pointer to a buffer of @B size @b bytes corresponding to a part of the file
   * at @B offset @b bytes from the beginning of the file.
   * Then you can directly read the memory pointed to by the returned pointer. This method
   * uses the memory paging system mechanism and thus is quite efficient.
   *
   * [OIV-WRAPPER-RETURN-TYPE ARRAY{size}]
   */
  void *getBuffer(int64_t offset, unsigned int size);

  // Convenient methods

  /**
   * Utility method that returns an integer from @B sizeBytes @b bytes starting at address @B ptr@b.
   * Also takes into account the machine architecture (little/big endian).
   *
   * [OIVNET-WRAPPER-ARG ARRAY,NO_WRAP{((ptr != nullptr) ? ptr->Length : 0)}]
   */
  static int bytesToInt(const unsigned char* ptr, int sizeBytes);

  /**
   * Utility method to convert an integer into the correct architecture (little/big endian).
   *
   * [OIVNET-WRAPPER-ARG ARRAY,NO_WRAP{((intPtr != nullptr) ? (intPtr->Length * 4) : 0)}]
   */
  static void swapBytes( int *intPtr, int sizeBytes );

  /**
   * Utility method to swap bytes of each element of an array of @B numElements @b elements.
   * The size of each element is @B numBytesPerElement@b.
   *
   * [OIVNET-WRAPPER-ARG WRAP_AS{unsigned char *}&ARRAY,NO_WRAP{((buffer != nullptr) ? (buffer->Length / numBytesPerElement) : 0)},IN]
   */
  static void swapBytesN( void *buffer, int numElements, int numBytesPerElement );

  /**
   * Utility method to check if value is a valid IEEE 754 floating point number
   */
  static SbBool isValidFloat( const float val );

  /**
   * Returns the size of the file when file I/O is handled by this class.
   */
  int64_t fileSize();

#if SoDEPRECATED_BEGIN(8500)

  SoDEPRECATED_MEMBER_NOWARN(8500," Use isDataConverted() instead.")
  SbBool m_dataConverted;

#endif /** @DEPRECATED_END */

  //------------------------------------------------------------------------------
#ifndef HIDDEN_FROM_DOC
  SbString m_filename;

#endif // HIDDEN_FROM_DOC


  /** VolumeViz doesn't handles empty extent (for ex. flat extent). But this case may happen some time.
   * For ex. Avizo generally specify a flat extent for single slice volume.
   * In that case, we try to compute a thickness  based on other dimension voxel size.
   * This method will return an extent that correspond to given input extent, but with a non null thickness
   * which is equal to smallest non null voxel size. */
  static SbBox3f adjustFlatExtent(const SbString& fileName, const SbBox3f& extent, const SbVec3i32& dim);

protected:

  //if true, try to detect automatically whether the coord sys is diret or not.
  //default is true
  SbBool m_directCoordSysAutoDetect;
  //has the data been stored using a direct coordinate system?
  //default is true
  SbBool m_directCoordSys;

  std::vector<SoDataSet::DatumElement> m_datum;
  int m_bytesPerVoxel;

  double m_rangeMin, m_rangeMax;
  bool   m_doRange;

  //True if header has already been read
  SbBool m_headerRead;

  CoordinateType m_coordinateType;
  std::vector<float> m_rectilinearCoordinates[3];

  SbBool       m_ignoredFile;

SoINTERNAL protected:
  // Used to guard the buffer obtained by calling getBuffer()
  SbThreadMutex* m_fmmMutex;

private:
  SbBool       m_isRGBA;
  SbBool       m_useFMM;
  void        *m_filehandle;
  int          m_filedesc;
  int64_t      m_filesize;
  void        *m_fmmdata;
  int64_t      m_fmmoffset;
  unsigned int m_fmmsize;
  bool m_hasPerTileMinMaxComputed;

  void releasefmm();
  void bestmap(int64_t);

  SbThreadMutex* m_fileMutex;

  // keep a map of readerType <-> extension association
  typedef std::map<SbString,SoType> SoVolumeReaderExtensionAssociationMap;
  static SoVolumeReaderExtensionAssociationMap s_volumeReaderExtensionAssociation;
  
  // Convert range from input type to given TypeOut type.
  template <typename TypeInt, typename TypeOut>
  void convert_range(size_t size, void* bufferIn,  void* bufferOut, int64_t outMin, int64_t outMax);

  // Convert float input type to any supported type
  template <typename TypeOut>
  void convert_out(size_t size, void* bufferIn,  void* bufferOut, int64_t outMin, int64_t outMax);

  // Convert any input type to float output
  template <typename TypeIn>
  void convert_in(size_t size, void* bufferIn,  void* bufferOut, int64_t outMin, int64_t outMax);

};

inline SbBool
SoVolumeReader::readXTraceInTile(int , unsigned char*& , const SbBox3i32& , const SbVec2i32& )
{
  return FALSE;
}

inline SbBool
SoVolumeReader::readXSliceInTile(int , unsigned char*& , const SbBox3i32& , const uint32_t& )
{
  return FALSE;
}

inline SbBool
SoVolumeReader::readYSliceInTile(int , unsigned char*& , const SbBox3i32& , const uint32_t& )
{
  return FALSE;
}

inline SbBool
SoVolumeReader::getMinMax(int& , int& )
{
  return FALSE;
}

inline SbBool
SoVolumeReader::getMinMax(int64_t &, int64_t &)
{
  return FALSE;
}

inline SbBool
SoVolumeReader::isIgnoredFile() const
{
  return m_ignoredFile;
}

inline SbBool
SoVolumeReader::getMinMax(double & , double & )
{
  return FALSE;
}

inline SbBool
SoVolumeReader::getHistogram(std::vector<int64_t>&)
{
  return FALSE;
}

inline SbBool
SoVolumeReader::setDirectCoordSysAutoDetection(SbBool)
{
  return FALSE;
}

inline SbBool
SoVolumeReader::getDirectCoordSysAutoDetection()
{
  return m_directCoordSysAutoDetect;
}

inline SbBool
SoVolumeReader::setDirectCoorSys(SbBool)
{
  return FALSE;
}

inline SbBool
SoVolumeReader::getDirectCoordSys()
{
  return m_directCoordSys;
}

inline bool
SoVolumeReader::isDead(int, int, int)
{
  return false;
}

inline uint64_t
SoVolumeReader::getTileSizeFromFid(int)
{
  return 0;
}

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#endif // _SO_VOLUME_READER_


