/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#pragma once

#include <Inventor/devices/SoCpuBufferObject.h>
#include <LDM/tiles/SoBufferPropertiesInterface.h>
#include <LDM/tiles/SoLDMBufferObjectInterface.h>

class SoGLBufferObject;
class SoCpuBufferUniform;
class SoCpuBufferBitSetImpl;

/**
* @LDMEXT LDM buffer object optimized for bitset tile data.
*
* @ingroup LDMTiles
*
* @DESCRIPTION
* This type of buffer efficiently represents a tile containing
* bitset data.  The buffer can be used like any other buffer,
* but uses less memory when its contents are not actually being
* accessed.
*
* @SEE_ALSO
*    SoBufferPropertiesInterface, SoCpuBufferObject
*/
class LDM_API SoCpuBufferBitSet: public SoCpuBufferObject, public SoBufferPropertiesInterface, public SoLDMBufferObjectInterface
{
  SO_TYPED_CLASS_HEADER();

  // ----------------------- Public usage --------------------------------------
public:

  /**
   * Default constructor.
   */
  SoCpuBufferBitSet();

  /**
   * Constructor from a SoCpuBufferObject containing a bitset.
   */
  SoCpuBufferBitSet(SoCpuBufferObject* cpuBuffer);

  /**
   * Create a new buffer with the same properties as the current one. @BR
   * The new instance will have the same context
   * or device properties, but no memory is allocated.
   */
  virtual SoBufferObject* createInstance() const;

  /**
   * Map the current buffer object into the specified buffer object. @BR
   * It is useful in order to use a buffer in multiple contexts.
   *
   * @param targetBufferObject The buffer object which will be the mapped version of
   *                           this buffer.
   * @param accessMode The access mode used for the mapping.
   * @param startPosition offset in source buffer to map from (default is start of buffer).
   * @param mappingSize size from the startPosition, if SO_BUFFER_SIZE_ALL then the whole buffer is mapped.
   * 
   */
  virtual void map( SoBufferObject* targetBufferObject, SoBufferObject::AccessMode accessMode, size_t startPosition = 0, size_t mappingSize = SO_BUFFER_SIZE_ALL );

  /**
   * Map the current buffer object into the specified CPU buffer object. @BR
   *
   * See the general map method for more information.
   */
  virtual void map( SoCpuBufferObject* targetBufferObject, SoBufferObject::AccessMode accessMode, size_t startPosition = 0, size_t mappingSize = SO_BUFFER_SIZE_ALL);
  
  /**
   * Map the current buffer object into the specified GL buffer object. @BR
   *
   * See the general map method for more information.
   */
  virtual void map( SoGLBufferObject* targetBufferObject, SoBufferObject::AccessMode accessMode, size_t startPosition = 0, size_t mappingSize = SO_BUFFER_SIZE_ALL);

  /**
   * Returns a pointer to the memory array used for the storage of the buffer.
   *
   * @return Pointer to the memory array used for the storage.
   * [OIV-WRAPPER-RETURN-TYPE ARRAY{GetSize()}]
   */
  virtual void* map( SoBufferObject::AccessMode accessMode, size_t offset = 0, size_t count = SO_BUFFER_SIZE_ALL );

  /**
  * Unmap the specified CPU buffer object @BR
  * It is defined just to speed up the call when the type of the object is known. @BR
  * Remove the specified bufferObject from the list of buffers which map
  * the current buffer.
  *
  * See the general unmap function for more information.
  */
  virtual void unmap( SoCpuBufferObject* bufferObject );

  /**
  * Unmap the specified GL buffer object @BR
  * It is defined just to speed up the call when the type of the object is known. @BR
  * If the access mode supports writing the specified buffer is sync'd with the buffer.
  *
  * See the general unmap function for more information.
  * @param bufferObject Buffer to be unmapped.
  */
  virtual void unmap( SoGLBufferObject* bufferObject );

  /**
   * Remove the specified bufferObject from the list of buffers which map
   * the current buffer.
   *
   * If the access mode supports writing the specified buffer is sync'd with the buffer.
   *
   * @param bufferObject Buffer to be unmapped.
   */
  virtual void unmap( SoBufferObject* bufferObject );

  /**
   * Unmap a mapped buffer.
   *
   * No error is reported if the buffer was not mapped.
   */
  virtual void unmap();

  /** @copydoc SoBufferObject::memcpy */
  virtual void memcpy( SoBufferObject* sourceBufferObject, size_t destOffset = 0, size_t sourceOffset = 0, size_t copySize = SO_BUFFER_SIZE_ALL );

  /** 
   * This function sets the contents of (or a portion of) this buffer object to the specified value.
   * Sets 0 if value = 0, 1 if value != 0.
   *
   * The valueSize parameter provides a way to do a memset with float, short, byte, etc values.
   * The first value changed in this buffer is given by the offset argument. The number of values to change.
   *
   * @param value is a pointer to the value to set in the buffer.
   * @param valueSize The size in bytes of the value. Default is 1 byte.
   * @param offset Index of first value to be changed. Default is 0.
   * @param count The number of values to change.  Default is number of values in buffer.
   */
  virtual void memset( void* value, size_t valueSize = 1, size_t offset = 0, size_t count = SO_BUFFER_SIZE_ALL );

  /**
   * Get the min and max values in the buffer, if they are known. @BR
   * Returns false if these values have not been set or computed.
   * Returns true if values are available.
   *
   * @param min user variable to store min value
   * @param max user variable to store max value
   */
  virtual bool getMinMax( double& min, double& max);

  /**
  * Set min and max values for this buffer.
  *
  * @param min user variable containing min value
  * @param max user variable containing max value
  */
  virtual void setMinMax( const double& min, const double& max);
  
  /**
  * Returns the size of the buffer in bytes. @BR
  * For this type of buffer the real size is the same as the
  * size of the SoBufferObject.
  */
  virtual size_t getRealSize() const;

SoINTERNAL public:

  /** Optimized version for SoCpuBufferBitSet. Mainly used by editing.
   * Do not put in virtual, this will cause perf issues. */
  void memcpy( SoCpuBufferBitSet* sourceBufferObject, size_t destOffset = 0, size_t sourceOffset = 0, size_t copySize = SO_BUFFER_SIZE_ALL );

  /** Optimized version for SoCpuBufferUniform. Mainly used by editing.
   * Do not put in virtual, this will cause perf issues. */
  void memcpy( SoCpuBufferUniform* sourceBufferObject, size_t destOffset = 0, size_t sourceOffset = 0, size_t copySize = SO_BUFFER_SIZE_ALL );

  /** Convenient internal API to copy content of a buffer to bitSet. 
   * Set bit to 0 for any value = 0, bit to 1 for any value != 0. */
  void memcpyInternal( SoBufferObject* sourceBufferObject, size_t dataSize, size_t destOffset = 0, size_t sourceOffset = 0, size_t copySize = SO_BUFFER_SIZE_ALL );

  /** Access the raw bit buffer. 
   * Don't forget to call unmapRawBitSet once finished. */
  unsigned char* mapRawBitSet(SoBufferObject::AccessMode accessMode);

  /** Unmap the raw bit buffer. */
  void unmapRawBitSet();

  /** Helper to retrieve the value of specified bit in buffer. */
  static inline unsigned char getBit(unsigned char* dataPtr, const int pos)
  { return (((dataPtr[pos>>0x3]) >> (7- (pos & 0x7))) & 1); }

  // Set the cache id allocated by the DataSet
  virtual void setCacheSignature(const uint64_t& cacheSignature);

  /** Set bit located at pos to value v in specified buffer. */
  static inline void setBit(unsigned char* dataPtr, const size_t pos, bool v)
  {
    unsigned char& val = dataPtr[pos/8];
    const unsigned char mask = 0x80 >> pos%8;
    const unsigned char bitChar = (unsigned char)v << (7 - pos%8);
    val = (val & ~mask) | bitChar;
  }

SoINTERNAL protected:
  friend class SoCpuBufferBitSetImpl;
  /** [OIV-WRAPPER NO_WRAP] */
  SoCpuBufferBitSet(SoCpuBufferBitSetImpl* impl);

protected:
  /**
   * Destructor.
   */
  virtual ~SoCpuBufferBitSet();
};

