/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#pragma once

#include <Inventor/devices/SoCpuBufferObject.h>
#include <LDM/tiles/SoBufferPropertiesInterface.h>
#include <LDM/tiles/SoLDMBufferObjectInterface.h>
#include <LDM/SoLDMTileID.h>

class SoDataSet;
class SoGLBufferObject;
class SoCpuBufferFromVolumeReaderImpl;

/**
*
* @LDMEXT LDM buffer object optimized for in memory reader tile data.
*
* @ingroup LDMTiles
*
* @DESCRIPTION
*
* @SEE_ALSO
*    SoBufferPropertiesInterface, SoCpuBufferObject
*/
class LDM_API SoCpuBufferFromVolumeReader: public SoCpuBufferObject, public SoBufferPropertiesInterface, public SoLDMBufferObjectInterface
{
  SO_TYPED_CLASS_HEADER();

  // ----------------------- Public usage --------------------------------------
public:

  /**
   * Constructor with dataset and tile Id as parameter
   *
   * @param dataset pointer to dataset associated with buffer
   * @param tileId tileId representing the tile the buffer will store
   */
  SoCpuBufferFromVolumeReader( SoDataSet* dataset, SoLDMTileID tileId );

  /**
   * Map the current buffer object into the specified buffer object. @BR
   * It is useful in order to use a buffer in multiple contexts.
   *
   * @param targetBufferObject The buffer object which will be the mapped version of
   *                           this buffer.
   * @param accessMode The access mode used for the mapping.
   * @param startPosition offset in source buffer to map from (default is start of buffer).
   * @param mappingSize size from the startPosition, if SO_BUFFER_SIZE_ALL then the whole buffer is mapped.
   * 
   */
  virtual void map( SoBufferObject* targetBufferObject, SoBufferObject::AccessMode accessMode, size_t startPosition, size_t mappingSize = SO_BUFFER_SIZE_ALL);

  /**
   * Map the current buffer object into the specified CPU buffer object. @BR
   *
   * See the general map method for more information.
   */
  virtual void map( SoCpuBufferObject* targetBufferObject, SoBufferObject::AccessMode accessMode, size_t startPosition = 0, size_t mappingSize = SO_BUFFER_SIZE_ALL );

  /**
   * Map the current buffer object into the specified GL buffer object. @BR
   *
   * See the general map method for more information.
   */
  virtual void map( SoGLBufferObject* targetBufferObject, SoBufferObject::AccessMode accessMode, size_t startPosition = 0, size_t mappingSize = SO_BUFFER_SIZE_ALL );

  /**
  * Map the buffer to a system memory address and allows the mapping of a sub part
  * of the buffer object into CPU memory.
  *
  * Notes:
  * - The unmap() method must be called before using the buffer.
  *
  * @return \if_cpp Pointer \else Reference \endif to data in the mapped region of the buffer memory.
  * [OIV-WRAPPER-RETURN-TYPE ARRAY{GetSize()}]
  */
  virtual void* map(SoBufferObject::AccessMode accessMode, size_t offset = 0, size_t count = SO_BUFFER_SIZE_ALL);
 
  /**
  * Unmap the specified CPU buffer object @BR
  * It is defined just to speed up the call when the type of the object is known. @BR
  * Remove the specified bufferObject from the list of buffers which map
  * the current buffer.
  *
  * See the general unmap function for more information.
  */
  virtual void unmap( SoCpuBufferObject* bufferObject );

  /**
  * Unmap the specified GL buffer object @BR
  * It is defined just to speed up the call when the type of the object is known. @BR
  * If the access mode supports writing the specified buffer is sync'd with the buffer.
  *
  * See the general unmap function for more information.
  * @param bufferObject Buffer to be unmapped.
  */
  virtual void unmap( SoGLBufferObject* bufferObject );

  /**
   * Unmap the specified buffer object @BR
   * It is defined just to speed up the call when the type of the object is known. @BR
   *
   * See the general unmap function for more information.
   */
  void unmap( SoBufferObject* bufferObject );

  /**
   * Unmap a mapped buffer.
   *
   * No error is reported if the buffer was not mapped.
   */
  virtual void unmap();

  /**
   * Get the min and max values in the buffer, if they are known. @BR
   * Returns false if these values have not been set or computed.
   * Returns true if values are available.
  *
  * @param min user variable to store min value
  * @param max user variable to store max value
  */
  bool getMinMax( double& min, double& max);

  /**
  * Returns the real size of the buffer in bytes.
  */
  size_t getRealSize() const;

protected:
  /**
   * Destructor.
   */
  virtual ~SoCpuBufferFromVolumeReader();

SoINTERNAL protected:
  // Create internal uncompress version of the buffer
  virtual SoCpuBufferObject* createCachedBuffer();

  virtual void prefetchBuffer();

  friend class SoCpuBufferFromVolumeReaderImpl;
  /** [OIV-WRAPPER NO_WRAP] */
  SoCpuBufferFromVolumeReader( SoCpuBufferFromVolumeReaderImpl* impl );

private:
  /**
   * Default constructor.
   */
  SoCpuBufferFromVolumeReader() {}
};
