/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Jun 2011)
**=======================================================================*/

#if !defined  _SO_VR_LDM_FILE_WRITER_H_
#define  _SO_VR_LDM_FILE_WRITER_H_

#include <LDM/writers/SoVolumeWriter.h>
#include <LDM/converters/SoLDMWriter.h>
#include <Inventor/fields/SoSFUInt32.h>
#include <Inventor/fields/SoSFVec3i32.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFDouble.h>
#include <Inventor/fields/SoSFVec2d.h>
#include <Inventor/fields/SoMFFloat.h>
#include <LDM/readers/SoVolumeReader.h>

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif // ! _MSC_VER


/**
 * @ingroup LDMWriters
 *
 * @LDMEXT Write data to disk in LDM format.
 *
 * @DESCRIPTION
 *   Creates an LDM file (in the VSG .ldm format)
 *   and allows the application to store data blocks in any order.  The most common 
 *   usage is to store blocks of full resolution data.  This class can automatically generate 
 *   the lower resolution (subsampled) tiles after the full resolution data has 
 *   been stored (when the finish() method is called and #doMultiResolution is set
 *   to TRUE).  However the 
 *   writeTile() method also allows the application to directly store lower 
 *   resolution tiles in case a proprietary subsampling algorithm is being used.  
 *   The result will normally be a .ldm file (LDM header) and a .dat file (data).
 *   
 *   It is not necessary, in all cases, to create the lower resolution tiles or even 
 *   to create the complete set of full resolution tiles.  This class supports 
 *   partial writing feature.  If some tiles are missing when the 
 *   #finish method is called, then in addition to the header and data files there 
 *   will also be a .fcp file with the same name as the header and data files.  The 
 *   LDM header file will contain a reference to this file in the CompletionFilename 
 *   tag.  The .fcp file stores information about which tiles actually exist.  Tiles 
 *   that do not exist are considered to be filled with a constant default value 
 *   (see #defaultValue).  This feature allows us, for 
 *   example, to compute and visualize a subset of the volume before committing to 
 *   computation on the whole volume.  However note that the writer currently 
 *   pre-allocates disk space for the ".dat" file assuming that all tiles will exist.
 *   So skipping creation of the subsampled tiles or writing only a subset of the 
 *   tiles can reduce computation time and disk I/O time, but it does not reduce the 
 *   disk space requirement for the volume.
 *   
 *   The first step is to create an SoVRLdmFileWriter object and initialize it with the 
 *   desired characteristics for the output volume.  The #initialize methods
 *   initialize the writer using its current parameters. It must not be modified
 *   once initialized.
 *
 *   The next step is to write data into the LDM file using the #writeTile methods.
 *
 *   The final step is to call the #finish method.  This will optionally
 *   generate the lower resolution tiles (see #doMultiResolution), cleanup and 
 *   close the LDM file.
 *
 * @EXAMPLE
 * \if_cpp
 *  \code
 *     SoVRLdmFileWriter* writer = new SoVRLdmFileWriter();
 *     // set parameters
 *     writer->headerFilename = "myFile.ldm";
 *     writer->dimension.setValue( 128, 128, 128 );
 *     writer->extent.setValue( -1,-1,-1, 1,1,1  );
 *
 *     // Initialize
 *     writer->initialize();
 *
 *     // Write data
 *     SoLDMTileID tileID = 0;
 *     SoCpuBufferObject* tileBuffer = new SoCpuBufferObject();
 *     writer->writeTile( tileID, tileBuffer );
 *
 *     // Finalize
 *     writer->finish();
 *  \endcode
 * \endif
 * \if_dotnet
 *  \code
 *     SoVRLdmFileWriter writer = new SoVRLdmFileWriter();
 *     // set parameters
 *     writer.headerFilename.Value = "myFile.ldm";
 *     writer.dimension.SetValue( 128, 128, 128 );
 *     writer.extent.SetValue( -1,-1,-1, 1,1,1  );
 *
 *     // Initialize
 *     writer.Initialize();
 *
 *     // Write data
 *     SoLDMTileID tileID = 0;
 *     SoCpuBufferObject tileBuffer = new SoCpuBufferObject();
 *     writer.WriteTile( tileID, tileBuffer );
 *
 *     // Finalize
 *     writer.Finish();
 *  \endcode
 * \endif
 * \if_java
 *  \code
 *     SoVRLdmFileWriter writer = new SoVRLdmFileWriter();
 *     // set parameters
 *     writer.headerFilename.setValue( "myFile.ldm" );
 *     writer.dimension.setValue( 128, 128, 128 );
 *     writer.extent.setValue( -1,-1,-1, 1,1,1  );
 *
 *     // Initialize
 *     writer.initialize();
 *
 *     // Write data
 *     SoLDMTileID tileID = new SoLDMTileID( 0 );
 *     SoCpuBufferObject tileBuffer = new SoCpuBufferObject();
 *     writer.writeTile( tileID, tileBuffer );
 *
 *     // Finalize
 *     writer.finish();
 *  \endcode
 * \endif
 *
 * @SEE_ALSO
 *    SoVolumeWriter
*/
class LDM_API SoVRLdmFileWriter : public SoVolumeWriter
{
  SO_FIELDCONTAINER_HEADER( SoVRLdmFileWriter );

public:

  /** Header Filename. */
  SoSFString headerFilename;

  /** Data Filename. By default, take headerFilename and use ".ldm" extension. */
  SoSFString dataFilename;

  /** Filename of original file. Empty by default. Needed if you want to keep an 
   * "history" of file. */
  SoSFString originalFilename;

  /** Tile dimension. Default is (128, 128, 128). */
  SoSFVec3i32 tileDimension;

  /** Auto compute sub-resolution tiles when finish writing. Default is true. */
  SoSFBool doMultiResolution;

  /** Specifies the algorithm used to build low resolution tiles. Available options are :
  * - 0 to use decimation algorithm (one voxel out of two).
  * - n to use weighted average algorithm : voxels of tile of resolution N+1 are
  *  built from the average of the 6 neighbors from resolution N and the current
  *  voxel value weighted by n.
  *  by default the decimation algorithm is used (n=0).
  */
  SoSFInt32 lowResAlgorithm;

  /** Save the data as a bitSet (8 values per byte). You should specified the 
   * #bitSetThreashold if you use this feature. Default is FALSE. */
  SoSFBool saveAsBitSet;

  /** 
   * Set the bitset threshold value. 
   * For each value (V) of a dataset, if V > threshold the bitset is set to 1 
   * for this bit, or to 0 otherwise. Default is 0. 
   * Used only if #saveAsBitSet is TRUE.
   */
  SoSFDouble bitSetThreashold;

  /** true if RGBA mode (output will be unsigned int32 RGBA values). 
   * Default is FALSE. */
  SoSFBool isRGBA;

  /** 
   * Specify if the histogram must be computed or not. Default is TRUE.
   *
   * Disabling histogram generation will greatly speedup the writing process,
   * but some VolumeViz rendering features and optimizations won't be available.
   * 
   * Enabling the histogram enables computation of:
   * - dataset min and max values
   * - per Tile min max values
   *
   * Note also that if RGBA mode is set then no histogram is computed.
   */
  SoSFBool enableHistogram;

  /** 
   * Specify if tile min/max should be computed even if #enableHistogram is false.
   * Note that when enableHistogram to true, tile min/max is always computed.
   * Default is false.
   */
  SoSFBool enableTileMinMax;

  /**
  * Set the undefined value. @BR
  * The given value will be stored in the LDM file. Default is NaN (Not a Number).
  * This is currently used by SoHeightFieldRender (vertices with undefined values are not rendered).
  */
  SoSFDouble undefinedValue;

  /**
   * Set the default value.
   * The given value will be stored in the LDM file. Default is 0.
   * If tiles are missing, LDM will use this value to display missing tiles.
   */
  SoSFDouble defaultValue;

  /** 
   * Compression name.
   * The following compression types are currently supported: "gzip" and "jpeg". 
   * Jpeg is a lossy compression format and should not be used if data 
   * degradation is unacceptable. You can set the #compressionLevel field 
   * to adjust compression quality.
   * If empty, no compression will be used. Default is empty.
   */
  SoSFString compressionName;

  /**
  * Set compression level. 
  * The meaning of this parameter depends on the type of compression used 
  * (see #compressionName): 
  * -For gzip, 
  * it must be between 0 and 9: 0 gives no compression, 9 gives best compression 
  * but slower speed. Default is 3. 
  * -For jpeg, it must be between 0 and 100: 0 gives best compression but most 
  * loss of data, 100 gives minimum compression and minimum data alteration. 
  * Default is 85. 
  *
  * -1 use default value of specified compressor (3 for gzip, 85 for jpeg).
  * Default is -1.
  */
  SoSFInt32 compressionLevel;

  /** Generate only header file. Default is FALSE. */
  SoSFBool headerOnly;

  /** verbose. Default is TRUE. */
  SoSFBool verbose;

  /** Set CRC32 check on compression. 
   * When compression is enabled using (see #compressionName), adds a data 
   * integrity check value to all compressed tiles. If the environment variable 
   * LDM_COMPRESS_CHECK_CRC32 is set, data values will be checked each time they 
   * are read from disk. 
   * Default is FALSE. 
   */
  SoSFBool crcCheck;

  /** Range of input datas ((min, max) of data). If not specified, range has to 
   * be computed, which will increase writing time. Default is an invalid range (1, -1)*/
  SoSFVec2d inputDataRange;

  /** @useenum{CoordinateType}. Default is COORDINATES_UNIFORM. */
  SoSFEnum coordinateType;

  /** Rectilinear coordinates along X. Used only if coordinateType is COORDINATES_RECTILINEAR. */
  SoMFFloat coordinatesX;
  
  /** Rectilinear coordinates along Y. Used only if coordinateType is COORDINATES_RECTILINEAR. */
  SoMFFloat coordinatesY;

  /** Rectilinear coordinates along Z. Used only if coordinateType is COORDINATES_RECTILINEAR. */
  SoMFFloat coordinatesZ;

  /** Coordinate type used by this data set. */
  enum CoordinateType {
  /** Uniform grid spacing along each axis. */
    COORDINATES_UNIFORM = SoVolumeReader::COORDINATES_UNIFORM,
  /** Grid spacing defined by x, y, z values. */
    COORDINATES_RECTILINEAR  = SoVolumeReader::COORDINATES_RECTILINEAR
  };


  /** Constructor */
  SoVRLdmFileWriter();

  /** Initialize writer from current parameters. Created and allocated data file
   * on disk. This operation can be quite long with big dataset. */
  virtual SbBool initialize();

  /**
   * Close all opened files (ldm, data and completion file) so that someone else
   * can access them for reading or writing.
   */
  virtual SbBool closeAllHandles();

  /** Re-open ldm, data and completion file that was closed by the #closeAllHandles() method. */
  virtual SbBool restoreAllHandles();

  /** Return TRUE, as ldm format is organized in tiles. */
  virtual SbBool isDataConverted() const;

  /** @copydoc SoVolumeWriter::writeTile */
  virtual SbBool writeTile( const SoLDMTileID& tileId, SoBufferObject* buffer );

  /** Finalize creation of files. If #doMultiResolution is set to true,
   * all missing sub-resolutions will be computed. This operation can be very long
   * with big dataset. */
  virtual SbBool finish();

SoINTERNAL public:

  /** 
   * Force recomputing lowRes even if it was already computed.
   * used if you write several time the same tile (while doing editing for ex.)
   * Else, low res tiles will not be updated because they will be mark as already
   * written in the completion file.
   */
  SoSFBool forceAlwaysUpdateLowRes;

  /** Just to forbid modification of field when writer was initialized */
  virtual void notify(SoNotList *);

protected:

  /** Destructor */
  virtual ~SoVRLdmFileWriter();

private:

  /** Does the writer is initialized or not. return to false when finish is called. */
  bool m_isInitialized;

  /** ldm writer. For the moment, this class is just an API that redirect everything
   * to SoLDMWriter (and thus, soConverter). */
  SoLDMWriter m_ldmWriter;

  /** Command line given to m_ldmWriter.initialize(). Act like an argv parameter. */
  std::vector<char*> m_writerParameters;

};

#ifdef _MSC_VER
#pragma warning(pop)
#endif

#endif // _SO_VR_LDM_FILE_WRITER_H_


