/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Jun 2011)
**=======================================================================*/

#if !defined  _SO_VOLUME_WRITER_H_
#define  _SO_VOLUME_WRITER_H_

#include <Inventor/fields/SoFieldContainer.h>
#include <Inventor/SbString.h>
#include <Inventor/SbBox.h>
#include <Inventor/SbDataType.h>
#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/fields/SoSFString.h>
#include <Inventor/fields/SoSFBox3f.h>
#include <Inventor/fields/SoSFVec3i32.h>
#include <Inventor/fields/SoSFEnum.h>
#include <LDM/SoLDMTileID.h>

class SoVolumeReader;

/**
 * @ingroup LDMWriters
 *
 * @LDMEXT Abstract base class for volume data writer.
 *
 * @DESCRIPTION
 * Base class used to write volume data. Data are generally organised by tiles
 * (like in LDM) or by slices (like in DICOM). 
 *
 * The #isDataConverted() method must be reimplemented in derived classes and 
 * return TRUE if the data is organized in tiles.  In this case, the #writeTile()
 * method MUST be implemented and will be used to write data.
 *
 * If #isDataConverted() returns FALSE, the #writeSubSlice() method MUST be 
 * implemented and will be used to write data.
 *
 * @EXAMPLE
 * \if_cpp
 *  \code
 *     SoVRLdmFileWriter* writer = new SoVRLdmFileWriter();
 *     // set parameters
 *     writer->headerFilename = "myFile.ldm";
 *     writer->dimension.setValue( 128, 128, 128 );
 *     writer->extent.setValue( -1,-1,-1, 1,1,1  );
 *
 *     // Initialize
 *     writer->initialize();
 *
 *     // Write data
 *     if ( writer->isDataConverted() ) {
 *       SoLDMTileID tileID = 0;
 *       SoCpuBufferObject* tileBuffer = new SoCpuBufferObject();
 *       writer->writeTile( tileID, tileBuffer );
 *     }
 *     else {
 *       int sliceIndex = 0;
 *       SbBox2i32 subSlice( 0,0, 128,128 );
 *       SoCpuBufferObject* sliceBuffer = new SoCpuBufferObject();
 *       writer->writeSubSlice( sliceIndex, subSlice, sliceBuffer );
 *     }
 *
 *     // Finalize
 *     writer->finish();
 *  \endcode
 * \endif
 * \if_dotnet
 *  \code
 *     SoVRLdmFileWriter writer = new SoVRLdmFileWriter();
 *     // set parameters
 *     writer.headerFilename.Value = "myFile.ldm";
 *     writer.dimension.SetValue( 128, 128, 128 );
 *     writer.extent.SetValue( -1,-1,-1, 1,1,1  );
 *
 *     // Initialize
 *     writer.Initialize();
 *
 *     // Write data
 *     if ( writer.IsDataConverted() ) {
 *       SoLDMTileID tileID = 0;
 *       SoCpuBufferObject tileBuffer = new SoCpuBufferObject();
 *       writer.WriteTile( tileID, tileBuffer );
 *     }
 *     else {
 *       int sliceIndex = 0;
 *       SbBox2i32 subSlice = new SbBox2i32( 0,0, 128,128 );
 *       SoCpuBufferObject sliceBuffer = new SoCpuBufferObject();
 *       writer.WriteSubSlice( sliceIndex, subSlice, sliceBuffer );
 *     }
 *
 *     // Finalize
 *     writer.Finish();
 *  \endcode
 * \endif
 * \if_java
 *  \code
 *     SoVRLdmFileWriter writer = new SoVRLdmFileWriter();
 *     // set parameters
 *     writer.headerFilename.setValue( "myFile.ldm" );
 *     writer.dimension.setValue( 128, 128, 128 );
 *     writer.extent.setValue( -1,-1,-1, 1,1,1  );
 *
 *     // Initialize
 *     writer.initialize();
 *
 *     // Write data
 *     if ( writer.isDataConverted() ) {
 *       SoLDMTileID tileID = new SoLDMTileID( 0 );
 *       SoCpuBufferObject tileBuffer = new SoCpuBufferObject();
 *       writer.writeTile( tileID, tileBuffer );
 *     }
 *     else {
 *       int sliceIndex = 0;
 *       SbBox2i32 subSlice = new SbBox2i32( 0,0, 128,128 );
 *       SoCpuBufferObject sliceBuffer = new SoCpuBufferObject();
 *       writer.writeSubSlice( sliceIndex, subSlice, sliceBuffer );
 *     }
 *
 *     // Finalize
 *     writer.finish();
 *  \endcode
 * \endif
 *
 * @SEE_ALSO
 *    SoVRLdmFileWriter, SoVRMemoryWriter
 *
 * [OIVNET-WRAPPER-CLASS ALL_DERIVABLE]
 */
class LDM_API SoVolumeWriter : public SoFieldContainer
{
  SO_FIELDCONTAINER_ABSTRACT_HEADER(SoVolumeWriter);

public:

  /** Extent of dataset. Default is (-1, -1, -1, 1, 1, 1)*/
  SoSFBox3f extent;

  /** Dimension of dataset, in voxels. Default is empty. */
  SoSFVec3i32 dimension;

  /** 
   * Type of input data (that will be given in the writeXXX method)
   * @useenum{SoDataSet::DataType}. Default is UNSIGNED_BYTE.
   */
  SoSFEnum dataType;

  /**
   * Endianess of data.
   * @useenum{WordFormat}. Default is current machine endianess.
   */
  SoSFEnum wordFormat;

  /** Endianess. Big endian or little endian. */
  enum WordFormat {
    // Must have same values that SiFileManage::getMachineWordFormat.
    // Note: BIG_ENDIAN and LITTLE_ENDIAN are already defined in types.h on linux
    // but have not the right values...
    WF_BIG_ENDIAN    = 0,
    WF_LITTLE_ENDIAN = 1
  };

  /** 
   * Initialize writer. Allocates memory and other resources depending on the kind of writer.  
   * By default do nothing. 
   */
  virtual SbBool initialize();

  /**
   * Close all ressources that are locked by the writer so that someone else can 
   * read or write to them.
   * For example, if the writer write to a file, this method must close the file
   * so that someone else can re-open it in read mode.
   */
  virtual SbBool closeAllHandles(); 

  /**
  * Restore ressources that was closed by #closeAllHandles.
  */
  virtual SbBool restoreAllHandles(); 

  /**
   * Returns TRUE if the data is already organized in tiles for the LDM module. @BR
   * In other words, all writers that directly support the writeTile() method should
   * return TRUE from isDataConverted.
   *
   * If TRUE is returned, VolumeViz will use the writeTile method and will not call
   * writeSubVolume() or writeSubSlice().
   *
   * Must be reimplemented in children classes.
   */
  virtual SbBool isDataConverted() const = 0;

  /** 
   * Given a tileID, writes a tile if the data is organized in tiles (see SoLDMTileID). @BR
   * This function must be called only if isDataConverted() return TRUE.
   *
   * Must be reimplemented in children classes.
   *
   * Please refer to #SoLDMTileID and #SoLDMTopoOctree for relation between 
   * tileId and tile position.
   */
  virtual SbBool writeTile( const SoLDMTileID& tileId, SoBufferObject* buffer );

  /** 
   * Writes the data contained in the specified subslice. 
   * This function must be called only if isDataConverted() return FALSE.
   *
   * Must be reimplemented in children classes.
   */
  virtual SbBool writeSubSlice( int sliceIndex, const SbBox2i32& subSlice, SoBufferObject* buffer );

  /** 
   * Finish writing the file, free ressources, ... depending on  kind of writer.
   */
  virtual SbBool finish();

protected:

  /** Constructor */
  SoVolumeWriter();

  /** Destructor */
  virtual ~SoVolumeWriter();
};

#endif // _SO_VOLUME_WRITER_H_


