// Open Inventor Medical Helper functions

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _INVENTOR_MEDICAL_HELPER_H_
#define _INVENTOR_MEDICAL_HELPER_H_

#include <Medical/InventorMedical.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/actions/SoSearchPathAction.h>

// SoDialogViz.h must be included before this file in order for
// DialogViz-related functions to be available.
// This is to avoid linking DialogViz with the Medical library
// so that the user code can decide which DialogViz library to use
// with their application.
#if defined(__DIALOGVIZLIB)
#include <DialogViz/dialog/SoTopLevelDialog.h>
#include <DialogViz/dialog/SoDialogCustom.h>
#endif

#include <Inventor/Axis.h>

#include <Inventor/STL/vector>

class SoCamera;
class SoDataRange;
class SoImageBackground;
class SoImageDataAdapter;
class SoMemoryDataAdapter;
class SoOrthoSlice;
class SoTransferFunction;
class SoVolumeData;
class SoVRDicomFileReader;
class SoMatrixTransform;

/**
* @VSGEXT @PREVIEWTAG @OIVMETAG Utility class for medical applications.
*
* @ingroup MedicalHelpers
*
* @DESCRIPTION
*
* Utility functions for medical visualization using Open Inventor.
*
* Open Inventor includes a general purpose 3D graphics core library plus optimized
* extensions for specific data types and markets, including medical visualization.
* We recommend using these utility functions (and the Medical utility nodes) to
* get the best performance and productivity from Open Inventor for medical applications.
*
* @B SUMMARY@b
*
* - #dicomAdjustVolume(): @BR
*   Call this method to adjust VolumeViz parameters for optimal performance with
*   medical data and to position a DICOM volume correctly in 3D space.
*
* - #dicomAdjustDataRange(): @BR
*   Call this method to set the VolumeViz data range based on the Window Center
*   and Window Width tags in a DICOM file.  If those tags are not present,
*   automatically sets the data range to the actual range of values in the data.
*
* - #dicomSetWindowCenterWidth() and #dicomGetWindowCenterWidth(): @BR
*   In Open Inventor the range of voxel values that map onto the color map are
*   specified as the 'min' and 'max' values of the range (see SoDataRange).
*   These methods provide a convenient way to set and query the data range using
*   the medical convention of 'window center' and 'window width'.
*
* - #dicomFindFilesbySeries(): @BR
*   Call this method to build a list of DICOM image files in a directory that
*   belong to the same series as the specified file. This list can be passed
*   to the DICOM volume reader using the #setFilenameList() method.
*
* - #orientView(): @BR
*   This method is useful for orienting the camera to view a single image (a
*   slice in VolumeViz jargon) along one of the standard axes (Axial, Coronal
*   or Sagittal).  But remember that the volume is always a 3D object to Open
*   Inventor and you must also set the 'axis' field on the SoOrthoSlice node.
*   See the dicomImageViewer example.
*
* @EXAMPLE
* \code
* // Data reader: Load all images in a series
* SoVRDicomFileReader* dicomReader = new SoVRDicomFileReader();
*   std::vector<SbString>& fileList;
*   MedicalHelper::dicomFindFilesbySeries( firstFilename, fileList );
*   MedicalHelper::setFilenameList( *dicomRender, fileList );
*
* // Data node
* SoVolumeData* volume = new SoVolumeData();
*   volData->setReader( *dicomReader );
*   MedicalHelper::dicomAdjustVolume( volData ); // Adjust position, extent, ...
*   root->addChild(volData);
*
* // Data range (from data file by default)
* SoDataRange* dataRange = new SoDataRange();
*   MedicalHelper::dicomAdjustDataRange( dataRange, volData ); // Use window center/width if possible...
*   root->addChild( dataRange );
* \endcode
*
* @EXAMPLE
* \code
* // Start a filter pipeline from a VolumeViz data set
* SoMemoryDataAdapter* adapter = MedicalHelper::getImageDataAdapter( volume );
*
* // Add a filter
* SoDeblurProcessing2d* filter = new SoDeblurProcessing2d();
*   filter->sharpeningFactor = 2;
*   filter->inImage = adapter;
* \endcode
*
* @SEE_ALSO
*  InventorMedical,
*  DicomInfo,
*  Gnomon,
*  Magnifier,
*  PlaneBoxIntersection,
*  PlaneGeometryIntersection,
*  Ruler,
*  SliceOrientationMarkers,
*  SliceScaleBar,
*  TextBox,
*  VolumeMarchingCubes
*
* @PREVIEWFEATURES
*/

class INVENTORMEDICAL_API MedicalHelper {
public:

  /** Medical axis names.
   *  These names are mapped to the corresponding XYZ enum value in Open Inventor. */
  enum Axis {
    /** Axial (Z axis) */
    AXIAL      = 2,
    /** Transverse (synonym for Axial) */
    TRANSVERSE = 2,
    /** Coronal (Y axis) */
    CORONAL    = 1,
    /** Sagittal (X axis) */
    SAGITTAL   = 0
  };

  /** Orient view.
   *  The camera is rotated to give the conventional orientation for the specified axis,
   *  then if a volume data node was specified:
   *
   *  - If the camera is an SoOrthographicCamera (the usual case for viewing a slice),
   *    viewAll() is called and, in addition, the view volume height is adjusted to make
   *    the slice image fill the window (subject to the 'slack' parameter). This is
   *    convenient because viewAll() computes a very loose fit around the slice image.
   *
   *  - If the camera is an SoPerspectiveCamera (commonly used for viewing a 3D volume),
   *    ViewAll() is called, but no additional adjustment is done to the camera.
   *
   *  For a 3D volume rendering view, typically specify the Coronal axis.
   *
   *  The orientation calculation is based on the DICOM LPS (Left, Posterior, Superior)
   *  reference coordinate system for a bipedal patient, where:
   *  - +X direction is right to left.
   *  - +Y direction is front to back (anterior to posterior).
   *  - +Z direction is feet to head (inferior to superior).
   *
   *  This works correctly in many, but not all, cases.  In particular it does not
   *  handle non-default values for the DICOM Image Orientation attribute.
   *
   *  Parameters:
   *  - axis: @BR
   *    - Axial : @BR
   *      Looking from feet to head, i.e looking toward +Z, with -Y up and +X to the right.
   *    - Coronal : @BR
   *      Looking at front of patient, i.e Looking toward +Y, with +Z up and +X to the right.
   *    - Sagittal : @BR
   *      Looking at left side of patient, i.e. Looking toward -X, with +Z up and +Y to the right.
   *
   *  - camera: The camera node that will be modified (fails if null).
   *
   *  - volume: [optional] The data volume that is being viewed (used to compute view volume, not modified).
   *            May be null, but a default volume extent of -1 to 1 will be used and viewAll() is not called.
   *
   *  - slack: Values greater than 1 will leave additional space between image and window edge
   *           when using an SoOrthographicCamera.
   *
   * Returns true if the operation succeeded.
   */
  static SbBool orientView( MedicalHelper::Axis axis, SoCamera* camera,
                            const SoVolumeData* volume = NULL, float slack = 1.01 );

  /** Optimize volume data node for DICOM volumes.
   *  Returns true if successful.
   *  The volume properties are modified (but not the data values).
   *  Do not call this method before setting the volume data node's
   *  fileName field or calling setReader.
   *  - First, adjusts the LDM tile size to optimize data loading. @BR
   *    Note this adjustment is valid for any data format that is not already
   *    tiled, i.e. most formats except .lda and .ldm.
   *
   *  - Second, adjusts the volume extent in 3D to be the actual physical extent
   *    of the data set measured from outside edge to outside edge of the voxels.
   *    This is a DICOM specific adjustment.
   *
   *  - Third, if 'useImagePosition' is true, adjusts the volume extent so the
   *    center of the first voxel is positioned at the coordinates specified in
   *    the data set's "Image position" (0020,0032) attribute.  By default
   *    the DICOM volume reader sets the volume extent so the center of the volume
   *    is positioned at 0,0,0 in 3D space. This is a DICOM specific adjustment.
   *
   * Note that this method doesn't support non axis aligned acquisition.
   * In such a case, prefer using the dicomAdjustVolume( SoVolumeData*, SoMatrixTransform* )
   * below.
   */
  static SbBool dicomAdjustVolume( SoVolumeData* volume, SbBool useImagePosition = TRUE );

  /**
   * Similar to #dicomAdjustVolume( SoVolumeData*, SbBool ) above but returns an SoMatrixTransform
   * that can be used to properly locate VolumeData in patient space in case of non axis aligned
   * acquisition:
   * \code
   * ...
   * SoMatrixTransform* matrixTransform = new SoMatrixTransfrom();
   * dicomAdjustVolume(volumeData, matrixTransfrom)
   * root->addChild(matrixTransform);
   * root->addChild(volumeData);
   * ...
   * \endcode
   * The computed matrix embeds the image position and image orientation as described here:
   * https://dicom.innolitics.com/ciods/ct-image/image-plane/00200037
   * Scaling part is embedded in VolumeData extent.
   *
   * If SoVolumeData is not a DICOM volume, matrixTransform will be set to identity.
   *
   * Returns true if successful.
   */
  static SbBool dicomAdjustVolume( SoVolumeData* volume, SoMatrixTransform* imgToPatient );

  /** Adjust data range based on values in the DICOM file,
   *  i.e. the window center (0028,1050) and window width (0028,1051) values.
   *  Returns true if successful. Assigns default values if necessary.
   *  If there are multiple slices, values are taken from the first slice.
   *  Volume is needed to get DICOM attributes, but is not modified.
   */
  static SbBool dicomAdjustDataRange( SoDataRange* rangeNode, const SoVolumeData* volume );

  /** Automatically adjusts the color map (transferFunction) if the specified
   *  volume is a DICOM data set (reader is SoVRDicomFileReader) and the
   *  the Photometric Interpretation (0028,0004) value is MONOCHROME1.
   *  Specifically, if the above conditions are true, this method reverses the
   *  values in the current color map.
   *
   *  Typically used as a convenience call when rendering slices.  For intensity
   *  based rendering of slices, set the transfer function to the predefined color
   *  map INTENSITY.  This is an intensity ramp from black to white that is
   *  appropriate for Photometric Interpretation MONOCHROME2 (the most common
   *  case).  When reversed it is appropriate for P.I. MONOCHROME1.
   *
   *  Returns true if change was needed and was successful.
   *  Volume is needed to get DICOM attributes, but is not modified.
   *  Transfer function is modified if necessary.
   */
  static SbBool dicomCheckMonochrome1( SoTransferFunction* cmapNode, const SoVolumeData* volume,
                                       SbBool forceReverse = FALSE );

  /** Get the "Image Position (Patient)" attribute (0020,0032) from a DICOM volume.
   *  DICOM calls this the upper left hand corner of the image, but more precisely
   *  it's the @I center@i of the first voxel, in millimeters (mm).  For VolumeViz the
   *  upper left hand corner of the image is literally the @I corner@i of the voxel,
   *  one-half voxel different from Image Position.
   *  Returns true if the query is successful.  If not successful, imagePos is 0,0,0.
   *  If there are multiple slices, value is taken from the first slice.
   *
   *  Note the SoVRDicomData getPosition() method does @I not@i return the value of
   *  the Image Position attribute. It returns a value computed from Image Position.
   */
  static SbBool dicomGetImagePosition( const SoVolumeData* volume, SbVec3f& imagePos );

  /** Get the window center (level) and width values from a DICOM volume.
   *  If the query fails, returns false and sets 'winCW' to 0,0.
   *
   *  Uses the Window Center (0028,1050) and Window Width (0028,1051) tags from the
   *  first image in the stack. If these tags do not exist in the volume, then window
   *  center and width are computed from the actual data min and max values. (Note
   *  that querying the actual min and max values may take some time because every
   *  voxel must be loaded.)
   *
   *  This method may be called with a non-DICOM volume. In that case the actual
   *  min and max data values are used.
   */
  static SbBool dicomGetWindowCenterWidth( const SoVolumeData* volume, SbVec2f& winCW );

  /** Get the window center (level) and width values from an SoDataRange node.
   *  Returns 0,0 on failure. Convenient for displaying data range in medical terms.
   */
  static SbVec2f dicomGetWindowCenterWidth( const SoDataRange* dataRange );

  /** Set an SoDataRange node from the window center (level) and width values.
   *  Returns true if successful. Convenient for converting from medical user
   *  interface to data range.
   */
  static SbBool dicomSetWindowCenterWidth( SoDataRange* dataRange, const SbVec2f& winCW );

  /** Get files in DICOM series.
   *  Given the path to a DICOM file, returns a list containing all the files in
   *  the same directory that are part of the same DICOM series, based on the
   *  series UID (0x0020,0x000E). File name extensions are not considered.
   *
   *  Returns 0 if the specified file was not found or is not a valid DICOM file.
   *
   *  This allows the application user to select a DICOM series in a file selection
   *  dialog box by just selecting one file instead of having to select all the
   *  files explicitly.
   *
   *  To load the series in Open Inventor, create an SoVRDicomFileReader and call
   *  the setFilenameList() method with the file list.  Then set this reader in
   *  the SoVolumeData node using the setReader() method.
   *  Note that SbStringList is a list of pointers to SbString objects,
   *  not a list of SbString objects.
   *
   *  The file names in the returned list are not in any particular order. It
   *  doesn't matter for loading the volume because the DICOM volume reader
   *  automatically sorts the images by their 'Slice Location' attribute (0x0020,0x1041).
   *
   *  See also: SbFileHelper::listFiles().  This method returns a list of files
   *  based on a specified pattern (regular expression) for the filenames.
   */
  static int dicomFindFilesbySeries( const SbString& firstFile, std::vector<SbString>& files );

  /** Get files in DICOM series.
   *  We recommend using the alternate version of this method.
   *  Note that SbStringList is a list of pointers to SbString objects,
   *  not a list of SbString objects.
   */
  static int dicomFindFilesbySeries( const SbString& firstFile, SbStringList& files );

  /**
   * Make the scene appear larger or smaller.
   *
   * Provided as a convenience to simplify application code. Effect is the same
   * as creating an SoCameraInteractor and calling the dolly() method, but note
   * that the value expected by that method is the @I inverse@i which is less
   * intuitive.  Using this method, values greater than 1 make the scene appear
   * larger and values less than 1 make the scene appear smaller.  For example
   * a of 2 will make the scene approximately 2 times larger.  The specific
   * effect on the camera is a 'dolly' or a 'zoom' depending on the type of camera.
   *
   * This is often useful after calling ViewAll(). That method sets the
   * camera parameters based on a bounding sphere around the scene which
   * often leaves the scene not "filling" the viewport.
   *
   * Details:
   * - SoPerspectiveCamera: changes the camera 'position' field.
   *   For example, values greater than 1 move the camera closer to the
   *   focal point (divide the camera's distance from the focal point by the
   *   given value), which makes the scene appear larger. This is a 'dolly'.
   * - SoOrthographicCamera: changes the camera 'height' field.
   *   For example, values greater than 1 decrease the view volume height
   *   (scale the height by the inverse of the given value), which makes the
   *   scene appear larger.  This is a 'zoom' operation.
   * - No effect if value is <= 0 or camera is null.
   */
  static void dollyZoom( float value, SoCamera* camera );

  /** Set filename list using a list of strings (file names).
   *
   *  This is a convenient alternative to calling SoVRDicomFileReader's method
   *  setFilenameList() directly. That method requires an SbStringList object
   *  which is a simple list of _pointers_, not a list of objects.
   */
  static SbBool setFilenameList( SoVRDicomFileReader& reader, const std::vector<SbString>& list );

  /** Returns the voxel size in 3D space.
   *  This is just a convenience.  The voxel size is simply the volume extent
   *  in 3D space divided by the number of voxels.
   */
  SbVec3f getVoxelSize( const SoVolumeData* volume ) const;

  /** Returns the volume's physical size in 3D space.
   *  This is just a convenience.  The physical size in 3D is simply the span
   *  of the volume extent.
   */
  SbVec3f getPhysicalSize( const SoVolumeData* volume ) const;

  /** Returns the DICOM volume origin in 3D space.
   *  This is just a convenience.
   *  In DICOM terms the image/volume origin is the @I center@i of the first
   *  voxel, but in VolumeViz the minimum point of the 3D extent is the outer
   *  edge of the first voxel.  The difference is 1/2 voxel.
   *
   *  Note: dicomGetImagePosition returns the actual value of the Image
   *  Position Patient attribute from the DICOM file (if possible).
   */
  SbVec3f getDicomOrigin( const SoVolumeData* volume ) const;
  
  /** Build a scene graph to display slice orientation markers.
    *
    * This is just a convenience method and helps keep the demo/example programs
    * consistent.  Applications can use SliceOrientationMarkers directly.
    *
    * Creates a SliceOrientationMarkers node with a default color and background
    * and connects its 'axis' field to the 'axis' field of the specified slice node.
    */
  static SoSeparator* buildSliceOrientationMarkers( const SoOrthoSlice* orthoSlice );

  /** Build a scene graph to display dynamic scale bars for slice viewing.
    *
    * This is just a convenience method and helps keep the demo/example programs
    * consistent.  Applications can use SliceScaleBar directly.
    *
    * Note that a typical length, 10 cm, is assumed.
    *
    * Creates horizontal and vertical SliceScaleBar nodes with a default color and
    * line width and sets their 'trackedCamera' field to the specified camera node.
    */
  static SoSeparator* buildSliceScaleBars( const SoCamera* camera );

  /** Slice viewer annotations.
    *
    * This is just a convenience method and helps keep the demo/example programs
    * consistent.  Applications can use SliceScaleBar (etc) directly.
    *
    * Calls the #buildSliceOrientationMarkers() and #buildSliceScaleBars() methods
    * and also creates some DicomInfo nodes using tag info from the optional
    * image filename.
    */
  static SoSeparator* buildSliceAnnotation( const SoCamera* camera, 
                                            const SoOrthoSlice* sliceNode,
                                            const SbString* dicomFilename );

  /** Returns standard window size for Open Inventor medical examples.
   *  This method is not important for customer applications.
   */
  static const SbVec2s& exampleWindowSize();

  /** Returns a logo image node for Open Inventor medical examples.
   *  This method is not important for customer applications, but the technique
   *  can be useful for adding logo images to an application window.
   *  Returns an SoImageBackground node, but as an SoNode.
   *  The returned node has a ref count of zero.
   */
  static SoNode* exampleLogoNode();

  /** Returns a collection of DicomInfo annotation nodes for Open Inventor medical examples.
   *  This method is not important for customer appications, but the technique
   *  can be useful for adding annotation text to an application window. @BR
   *  - filename is the path to a DICOM file from which info will be extracted.
   */
  static SoNode* exampleDicomAnnotation( const SbString& filename );

  /**
   * Returns the medical axis (AXIAL, CORONAL, SAGITTAL) corresponding to the
   * specified view axis (X, Y, Z) based on the DICOM Image Orientation element
   * (0x0020,0x0037).
   *
   * Use this method when you know the camera will be aligned with one of the
   * 3D world coordinate axes and you need to know which medical axis the view
   * axis corresponds to.  You need the medical axis in order to display the
   * correct slice orientation letters.  For example, if the Image Orientation
   * is identity and the camera is aligned with the Z axis, then this is an
   * Axial view.
   *
   * The conversion may not be valid if the DICOM Image Orientation is not
   * approximately axis aligned. The conversion uses the largest component of
   * each direction cosine to detemine the associated axis.
   *
   * volData parameter: This parameter is used to get the DICOM Image Orientation
   * element. If the element is not available or the volume is not a DICOM
   * data set (reader is not an SoVRDicomFileReader), then the method returns
   * the value of the specified axis.
   *
   * In many cases the medical axis and view axis are the same. However this
   * depends on the DICOM Image Orientation element (0x0020,0x0037).
   * If Image Orientation is missing or has the default value "1/0/0/0/1/0"
   * (identity matrix), then the view and medical axes have the default DICOM
   * relationship:
   * - View X axis is the medical SAGITTAL (RL) axis
   * - View Y axis is the medical CORONAL (AP) axis
   * - View Z axis is the medical AXIAL (FH) axis
   *
   * But, for example, if Image Orientation is "0/1/0/0/0/-1", then the view
   * Z axis is actually the medical SAGITTAL (RL) axis.
   *
   * Applications should use the view axis (e.g. Z) to set the camera direction
   * and slice axis (see SoOrthoSlice), but use the medical axis (e.g. AXIAL)
   * to select the slice orientation labels (see SliceOrientationMarkers).
   *
   * Notes:
   * - VolumeViz (SoVRDicomFileReader) always loads the images along the 3D world
   *   coordinate Z axis. Image viewer applications should normally default to
   *   a Z axis view of a Z axis OrthoSlice.
   *
   * - DICOM Image Orientation specifies the direction cosines of the first row
   *   and the first column of image pixels with respect to the patient.
   *   Essentially two vectors defining the (possibly rotated) directions of the
   *   volume X and Y axes in 3D space. Open Inventor assumes these vectors are
   *   orthogonal.  That is normally true but not required by DICOM.
   *
   * - This is not a complete solution to the medical vs 3D space view conversion.
   *   This method correctly returns which @I axis@i, but does not tell you which
   *   direction the camera should be pointed (+ or -) along that axis.
   */
  static Axis MedicalAxisFromViewAxis(openinventor::inventor::Axis::Type viewAxis, const SoVolumeData* volData);

  /**
   * Returns the medical axis (AXIAL, CORONAL, SAGITTAL) corresponding to the
   * specified view axis (X, Y, Z) based on the specified orientation matrix.
   *
   * See MedicalAxisFromViewAxis(Axis, const SoVolumeData*) for details.
   *
   * Normally this matrix is computed from the DICOM Image Orientation element.
   * This version of the method allows the application to extract the orientation
   * matrix and save it. The orientation matrix can be created from the DICOM
   * Image Orientation vectors or conveniently queried using SoVRDicomData::getOrientation().
   */
  static Axis MedicalAxisFromViewAxis(openinventor::inventor::Axis::Type viewAxis, const SbMatrix& orientationMatrix);

  /**
   * Returns the view axis (X, Y, Z) corresponding to the specified medical axis
   * (AXIAL, CORONAL, SAGITTAL) based on the DICOM Image Orientation element (0x0020,0x0037).
   *
   * Use this method when you know the medical axis that should be displayed
   * and you need to know the view axis (X, Y, Z) that corresponds to that.
   * For example, if the Image Orientation is identity and an Axial view
   * should be displayed, then the camera and the SoOrthoSlice should be set to Z.
   */
  static openinventor::inventor::Axis::Type ViewAxisFromMedicalAxis(Axis medicalAxis, const SoVolumeData* volData);

  /**
   * Returns the view axis (X, Y, Z) corresponding to the specified medical axis
   * (AXIAL, CORONAL, SAGITTAL) based on the DICOM Image Orientation element (0x0020,0x0037).
   *
   * See ViewAxisFromMedicalAxis(Axis, const SoVolumeData*) for details.
   *
   * Normally this matrix is computed from the DICOM Image Orientation element.
   * This version of the method allows the application to extract the orientation
   * matrix and save it. The orientation matrix can be created from the DICOM
   * Image Orientation vectors or conveniently queried using SoVRDicomData::getOrientation().
   */
  static openinventor::inventor::Axis::Type ViewAxisFromMedicalAxis(Axis medicalAxis, const SbMatrix& orientationMatrix);

  //------------------------------------------------------------------------------
  /** Convenience function to find first specified node in specified scene graph. */
  template <typename NodeType>
  static NodeType*
    find(SoNode* root, const SbString& nodeName = SbString(), const bool okIfNotFound = false)
  {
    if (root == NULL)
      return NULL;

    SoSearchAction* spa = NULL;
    if (!nodeName.isEmpty())
    {
      spa = new SoSearchPathAction;
      ((SoSearchPathAction*)spa)->setSearchString(nodeName.getString());
    }
    else
    {
      spa = new SoSearchAction;
    }
    spa->setSearchingAll(TRUE);
    spa->setType(NodeType::getClassTypeId());

    // Make sure the action doesn't destroy this node...
    // Do not unref until _after_ the action is destroyed!
    // Apply() will ref and unref, but root will also be ref'd by the resulting SoPath.
    root->ref();
    spa->apply(root);

    if (spa->getPath() == NULL)
    {
      delete spa;
      if (!okIfNotFound)
        std::cerr << "Cannot find \"" << nodeName << "\" of type \"" << NodeType::getClassTypeId().getName() << "\" in scene graph." << std::endl;
      return NULL;
    }

    NodeType* node = dynamic_cast<NodeType*>(spa->getPath()->getTail());
    if (node == NULL)
    {
      delete spa;
      if (!okIfNotFound)
        std::cerr << "Cannot find \"" << nodeName << "\" of type \"" << NodeType::getClassTypeId().getName() << "\" in scene graph." << std::endl;
      return NULL;
    }

    delete spa;             // This will also destroy the SoPath, which will unref 'root'.
    root->unrefNoDelete();  // Finally, remove our safety ref from 'root'
    return node;
  }

  //------------------------------------------------------------------------------
  /** Convenience function to find all node of specified type in specified graph. */
  template <typename NodeType>
  static std::vector<NodeType*>findNodes(SoNode* scene)
  {
    SoSearchAction* sa = new SoSearchAction;
    sa->setType(NodeType::getClassTypeId());
    sa->setInterest(SoSearchAction::ALL);

    scene->ref(); // Make sure action does not destroy this node...
    sa->apply(scene);
    scene->unrefNoDelete();

    std::vector<NodeType*> ret;

    const SoPathList& path = sa->getPaths();
    if (path.getLength() == 0)
    {
      std::cerr << NodeType::getClassTypeId().getName().getString() << " not found" << std::endl;
      return ret;
    }

    for (int i = 0; i < path.getLength(); i++)
      ret.push_back(dynamic_cast<NodeType*>(path[i]->getTail()));

    delete sa;

    return ret;
  }

#if defined(__DIALOGVIZLIB)
  //------------------------------------------------------------------------------
  /**
   * Convenience function to find first specified node in specified GUI graph.
   *
   * SoDialogViz.h must be included before this file in order for this function to be available.
   */
  template <typename NodeType>
  static NodeType*
    find(SoTopLevelDialog* root, const SbString& auditorId = SbString(), bool exitIfNotFound = true)
  {
    if (root == NULL)
      return NULL;

    NodeType* node = dynamic_cast<NodeType*>(root->searchForAuditorId(auditorId));
    if (node == NULL)
    {
      std::cerr << "Cannot find " << auditorId << " of type " << NodeType::getClassTypeId().getName() << " in interface." << std::endl;
      if (exitIfNotFound)
        exit(1);
    }

    return node;
  }
#endif /* __DIALOGVIZLIB */

  //------------------------------------------------------------------------------
  /** Convenience function to retrieve bounding box of specified node.
   *  If node is a group that contains an SoVolumeData, uses SoVolumeData extent.
   */
  static SbBox3f getBoundingBox(SoNode* node);

  //------------------------------------------------------------------------------
  /** Convenience function to draw a cube representing the specified bounding box.
   *  Color is inherited. */
  static SoSeparator* createCube(const SbBox3f& bbox);

  //------------------------------------------------------------------------------
  /** Convenience function to draw a wireframe box representing the specified bounding box.
   *  Default color is red (1,0,0). */
  static SoSeparator* createBoundingBox(const SbBox3f& bbox, SbColor* color = NULL);

  //------------------------------------------------------------------------------
  /** Convenience function to read a .iv file and return pointer to the root
   *  of the created scenegraph. */
  static SoSeparator* readFile(const char *filename);

#if defined(__DIALOGVIZLIB)
  //------------------------------------------------------------------------------
  /** Convenience function to read a .iv file defining a DialogViz user interface.
   *  Not available for the Open Inventor Headless build.
   *
   *  SoDialogViz.h must be included before this file in order for this function to be available.
   */
  template<typename WidgetType>
  static WidgetType
  buildInterface(WidgetType window, const char* filename, const char* viewerName, SoTopLevelDialog** topLevelDialog)
  {
    SoInput myInput;
    if (!myInput.openFile(filename))
      return NULL;

    SoGroup *myGroup = SoDB::readAll(&myInput);
    myInput.closeFile();
    if (!myGroup)
      return NULL;

    *topLevelDialog = (SoTopLevelDialog *)myGroup->getChild(0);

    SoDialogCustom *customNode = (SoDialogCustom *)(*topLevelDialog)->searchForAuditorId(SbString(viewerName));

    (*topLevelDialog)->buildDialog(window, customNode != NULL);
    (*topLevelDialog)->show();

    return customNode ? customNode->getWidget() : window;
  }
#endif /* __DIALOGVIZLIB */
};

#endif
