// Simple Marching Cubes for VolumeViz

///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _VOLUME_MARCHING_CUBES_
#define _VOLUME_MARCHING_CUBES_

#include <Medical/InventorMedical.h>

#include <Inventor/nodes/SoTriangleSet.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <LDM/SoLDMTileID.h>
#include <LDM/SoLDMDataAccess.h>

#include <vector>

class  SoLDMTopoOctree;


/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Class for computing an isosurface from a VolumeViz data set.
 *
 * @ingroup MedicalHelpers
 *
 * @DESCRIPTION
 * This class contains a simple implementation of the classic Marching Cubes
 * algorithm that computes the geometry (triangles) of an isosurface from a
 * VolumeViz data set. It returns a shape node (e.g. SoTriangleSet) with an
 * attached SoVertexProperty node containing the vertices.
 *
 * The computed geometry can be exported using the usual Open Inventor tools,
 * for example SoSTLWriteAction will write an STL format file.
 *
 * This class is not intended to be used for rendering an isosurface and
 * therefore is not implemented as an Open Inventor node. The computed
 * geometry @I may@i be rendered, if desired, by adding the returned node to
 * the scene graph. However, the SoVolumeIsosurface node is a much better tool
 * for rendering isosurfaces. Using the GPU, SoVolumeIsosurface can render
 * multiple isosurfaces simultaneously with interactive performance.  We
 * recommend using SoVolumeIsosurface for rendering and using this class only
 * when the actual geometry is needed for export.
 *
 * There are faster and better implementations of isosurface extraction,
 * including the one in Open Inventor's MeshVizXLM extension.
 * You can use that implementation by following the "VolumeMesh" example
 * provided with the Open Inventor SDK. The intent here is for the code to be
 * easy to understand so that applications can extend and modify.
 *
 * Notes:
 *
 *   - Performance. @BR
 *     VolumeViz automatically manages volume data internally as "tiles".
 *     Accessing the data in a single tile for computation is very efficient,
 *     but accessing across tile boundaries takes more time.  This class has
 *     optimized code for volumes that fit in a single tile.  For best
 *     performance, set the volume's tile size equal to the power-of-2 that
 *     is greater than or equal to the largest volume dimension, but do not
 *     set the tile size larger than 512.
 *
 *   - Geometry extent. @BR
 *     We consider the center of each voxel to be a vertex of a regular mesh
 *     so we can apply the MarchingCubes algorithm to the mesh.  As a result
 *     the 3D extent of the mesh starts at volumeMin + 1/2 the voxel size
 *     and ends at volumeMax - 1/2 the voxel size.
 *
 *   - Normal vectors. @BR
 *     The algorithm does not currently compute normal vectors for the
 *     triangle vertices.  For rendering, Open Inventor will automatically
 *     compute normal vectors, but by default the rendered shape will not
 *     be "smooth".  We recommend that applications add an SoShapeHints node
 *     to the scene graph above the marching cubes geometry node and set the
 *     'creaseAngle' field to PI (3.14159).
 *
 * Limitations:
 *
 *   - Separate triangles. @BR
 *     The geometry is returned as a collection of separate triangles.
 *     It would be possible to create an indexed geometry where multiple
 *     triangles share vertices, but the implementation would be much more
 *     complex. Note that if the geometry will be exported as STL then it
 *     will be exported as separate triangles anyway.
 *
 *   - Triangle quality. @BR
 *     The classic Marching Cubes algorithm produces a very large number of
 *     triangles from a typical volume data set and some triangles may have
 *     poor aspect ratio.
 *
 *   - Region of Interest. @BR
 *     Currently the algorithm does not consider the region of interest (SoROI).
 *     The isosurface is extracted from the whole volume.
 *
 * @SEE_ALSO
 *   PlaneGeometryIntersection
 * 
 * @PREVIEWFEATURES
 */

class INVENTORMEDICAL_API VolumeMarchingCubes
{
public:
  /** Constructor */
  VolumeMarchingCubes();

  /** Destructor */
  ~VolumeMarchingCubes();

  /** Returns the geometry for an isosurface with the specified value.
   *  The vertices are stored in an SoVertexProperty node which can be
   *  accessed in the @I vertexProperty@i field of the returned shape node.
   *
   *  Returns null if any error occurs (for example accessing volume data).
   *  Returns null if volume contains RGBA data.
   */
  SoTriangleSet* getIsosurface( const SoVolumeData& volume, float isovalue );

  /** Returns number of triangles in most recently generated isosurface.
   *  Returns zero if no isosurface has been extracted.
   */
  size_t getNumTriangles() const;

  /** Returns isovalue of most recently generated isosurface.
   *  Returns zero if no isosurface has been extracted.
   */
  float getIsovalue() const;

protected:
  /** Apply marching cubes to the entire volume.
   *  Iterates over "cells" in the volume, calling marchCube().
   *  Less efficient than marchTile(), but correctly handles
   *  volumes that contain multiple tiles.
   */
  void marchVolume();

  /** Apply marching cubes to the specified tile.
   *  Iterates over "cells" in the tile, calling marchCube().
   *  Can only handle cells that are completely within the tile, but is more
   *  efficient than marchVolume() when the volume is a single tile.
   */
  void marchTile( SbVec3i32 tileDim, SbVec3f tileMin, SbVec3f tileMax, void* tileData );

  /** Apply marching cubes to one cube.
   *  This method implements the Marching Cubes algorithm. The cube is completely
   *  specified by its XYZ coordinates and scalar values, so the implementation
   *  is not specific to VolumeViz.
   *
   *  Return: number of triangles created (maximum 5, minimum 0).
   *  corner: xyz position of cube lower left back corner. @BR
   *  size  : xyz size of cube (voxel size). @BR
   *  values: value at each corner of cube (voxel values). @BR
   *  isoval: isovalue @BR
   *  verts : xyz vertices of triangles (3 * number of triangles).
   *
   * Values array order is:
   *   0: lower-left-back , 1: lower-right-back , 2: upper-right-back , 3: upper-left-back,
   *   4: lower-left-front, 5: lower-right-front, 6: upper-right-front, 7: upper-left-front.
   */
  int marchCube( SbVec3f& corner, SbVec3f& size, float values[8], float isoval, SbVec3f vertices[15] );

  /** Get the value of the voxel at location i,j,k.
   *  If there are multiple tiles in the volume, this method is used to get
   *  the values at the corners of each "cell" in the mesh.  This method
   *  correctly handles volumes containing multiple tiles, but could be
   *  optimized (see corresponding method in MeshVizXLM VolumeMesh example.
   */
  float getVoxelValue( int i, int j, int k);

  // Volume properties
  SbVec3i32 m_voldim;     // IJK dimensions
  SbBox3f   m_volext;     // XYZ extent
  SbVec3i32 m_tiledim;    // IJK dimensions
  SbVec3f   m_tileSize;   // XYZ size
  SbVec3f   m_voxelSize;  // XYZ size
  unsigned int        m_voxelBytes;
  SoDataSet::DataType m_voxelType;

  // Stuff to handle multiple tiles
  SoVolumeData*    m_volData;
  SoLDMDataAccess::DataInfo* m_dataInfo;  // Data for current locked tile (if any).
  SbVec3i32 m_tileMin;
  SbVec3i32 m_tileMax;

  // Only used during extraction.  Should be hidden in an impl class.
  std::vector<SbVec3f> m_vertices;

  // Other
  float m_isoValue;
  int   m_resolution;
  size_t m_numTriangles;
};

#endif
