///////////////////////////////////////////////////////////////////////////////
//
// This class is part of the Open Inventor Medical utility library.
//
// The medical utility classes are provided as a prebuilt library named
// "fei_inventor_medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _MEDICAL_MAGNIFIER_H
#define _MEDICAL_MAGNIFIER_H

#include <Medical/InventorMedical.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/draggers/SoTranslate2Dragger.h>
#include <Inventor/sensors/SoFieldSensor.h>

#include <Inventor/fields/SoSFColor.h>
#include <Inventor/fields/SoSFFloat.h>
#include <Inventor/fields/SoSFNode.h>

/**
 * @VSGEXT @PREVIEWTAG @OIVMETAG Shape node to display a magnified view of a region of the screen.
 *
 * @ingroup MedicalNodes
 *
 * @DESCRIPTION
 *   This node displays a circular magnified (zoomed in) view of the screen
 *   region under the center of the circle.  The magnifier is an
 *   SoTranslate2Dragger so the user can click and drag to move it around
 *   the screen.
 *
 *   @B Limitations@b:
 *   - Can only be used in a "2D" viewing environment using an SoOrthographicCamera.
 *     @BR For example viewing a single slice of a medical volume.
 *
 *   - Can only be used with the default camera orientation (looking toward -Z).
 *
 * @FILE_FORMAT_DEFAULT
 *    Magnifier {
 *    @TABLE_FILE_FORMAT
 *      @TR sceneToMagnify   @TD NULL
 *      @TR magnifierColor   @TD 0 0.6 0
 *      @TR magnifierFactor  @TD 50
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *  InventorMedical, DicomInfo, Gnomon, Ruler
 * 
 * @PREVIEWFEATURES
 */ 

class INVENTORMEDICAL_API Magnifier : public SoTranslate2Dragger {

  SO_NODE_HEADER(Magnifier);

public:
  /** Specifies the root node of the scene that will be magnified.
    * Default is NULL.
    */
  SoSFNode	sceneToMagnify;

  /** Specifies the color of the border around the magnifier.
    * Default is 0,0.6,0 (green).
    */
  SoSFColor	magnifierColor;

  /** Specifies the magnification factor.
    * Default is 50. Technically speaking this field specifies the
    * value to set in the height field of the magnifier's camera.
    */
  SoSFFloat	magnifierFactor;

  /** Constructor. */
   Magnifier( );

  /** Initialize class (called automatically by InventorMedical::init()). */
  static void initClass();

  /** Finish using class (called automatically by InventorMedical::finish()). */
  static void exitClass();

protected:
  virtual ~Magnifier();

private:

  void commonConstructor();

  SoFieldSensor*       m_sceneFieldSensor;
  static void sceneFieldChangedCB( void* data, SoSensor* sensor );

  SoRef<SoSeparator>   m_magnifierSGSep;
  SoRef<SoMaterial>    m_magnifierBorderMaterial;
  SoRef<SoSeparator>   m_sceneToMagnifyParent;
  SoRef<SoOrthographicCamera> m_magnifierCam;
};
#endif
